/*
 * Decompiled with CFR 0.152.
 */
package org.pmw.tinylog;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;
import java.util.regex.Pattern;
import org.pmw.tinylog.EnvironmentHelper;
import org.pmw.tinylog.InternalLogger;
import org.pmw.tinylog.LogEntry;
import org.pmw.tinylog.PreciseLogEntry;
import org.pmw.tinylog.Token;
import org.pmw.tinylog.writers.LogEntryValue;

final class Tokenizer {
    private static final String DEFAULT_DATE_FORMAT_PATTERN = "yyyy-MM-dd HH:mm:ss";
    private static final String NEW_LINE = EnvironmentHelper.getNewLine();
    private static final Pattern NEW_LINE_REPLACER = Pattern.compile("\r\n|\\\\r\\\\n|\n|\\\\n|\r|\\\\r");
    private static final String TAB = "\t";
    private static final Pattern TAB_REPLACER = Pattern.compile("\t|\\\\t");
    private final Locale locale;
    private final int maxStackTraceElements;
    private int index;

    Tokenizer(Locale locale, int n) {
        this.locale = locale;
        this.maxStackTraceElements = n;
    }

    List<Token> parse(String string) {
        ArrayList<Token> arrayList = new ArrayList<Token>();
        this.index = 0;
        while (this.index < string.length()) {
            char c = string.charAt(this.index);
            int n = this.index;
            while (c != '{' && c != '}') {
                ++this.index;
                if (this.index >= string.length()) {
                    arrayList.add(Tokenizer.getPlainTextToken(string.substring(n, this.index)));
                    return arrayList;
                }
                c = string.charAt(this.index);
            }
            if (this.index > n) {
                arrayList.add(Tokenizer.getPlainTextToken(string.substring(n, this.index)));
            }
            if (c == '{') {
                Token token = this.parsePartly(string);
                if (token == null) continue;
                arrayList.add(token);
                continue;
            }
            if (c != '}') continue;
            InternalLogger.warn("Opening curly brace is missing for: \"{}\"", string.substring(0, this.index + 1));
            ++this.index;
        }
        return arrayList;
    }

    private Token parsePartly(String string) {
        ArrayList<Token> arrayList = new ArrayList<Token>();
        int[] nArray = new int[]{0, 0};
        int n = this.index++;
        while (this.index < string.length()) {
            char c = string.charAt(this.index);
            int n2 = this.index;
            while (c != '{' && c != '|' && c != '}') {
                ++this.index;
                if (this.index >= string.length()) {
                    InternalLogger.warn("Closing curly brace is missing for: \"{}\"", string.substring(n, this.index));
                    arrayList.add(this.getToken(string.substring(n2, this.index)));
                    return Tokenizer.combine(arrayList, nArray);
                }
                c = string.charAt(this.index);
            }
            if (this.index > n2) {
                if (c == '{') {
                    arrayList.add(Tokenizer.getPlainTextToken(string.substring(n2, this.index)));
                } else {
                    arrayList.add(this.getToken(string.substring(n2, this.index)));
                }
            }
            if (c == '{') {
                Token token = this.parsePartly(string);
                if (token == null) continue;
                arrayList.add(token);
                continue;
            }
            if (c == '|') {
                ++this.index;
                n2 = this.index;
                while (c != '{' && c != '}') {
                    ++this.index;
                    if (this.index >= string.length()) {
                        InternalLogger.warn("Closing curly brace is missing for: \"{}\"", string.substring(n, this.index));
                        nArray = Tokenizer.parseOptions(string.substring(n2));
                        return Tokenizer.combine(arrayList, nArray);
                    }
                    c = string.charAt(this.index);
                }
                if (this.index <= n2) continue;
                nArray = Tokenizer.parseOptions(string.substring(n2, this.index));
                continue;
            }
            if (c != '}') continue;
            ++this.index;
            return Tokenizer.combine(arrayList, nArray);
        }
        InternalLogger.warn("Closing curly brace is missing for: \"{}\"", string.substring(n, this.index));
        return Tokenizer.combine(arrayList, nArray);
    }

    private Token getToken(String string) {
        if (string.equals("date")) {
            return Tokenizer.getDateToken(DEFAULT_DATE_FORMAT_PATTERN, this.locale);
        }
        if (string.startsWith("date:")) {
            String string2 = string.substring(5, string.length()).trim();
            try {
                return Tokenizer.getDateToken(string2, this.locale);
            }
            catch (IllegalArgumentException illegalArgumentException) {
                InternalLogger.error(illegalArgumentException, "\"{}\" is an invalid date format pattern", string2);
                return Tokenizer.getDateToken(DEFAULT_DATE_FORMAT_PATTERN, this.locale);
            }
        }
        if ("pid".equals(string)) {
            return new PlainTextToken(EnvironmentHelper.getRuntimeDialect().getProcessId());
        }
        if (string.startsWith("pid:")) {
            InternalLogger.warn("\"{pid}\" does not support parameters");
            return new PlainTextToken(EnvironmentHelper.getRuntimeDialect().getProcessId());
        }
        if ("thread".equals(string)) {
            return new ThreadNameToken();
        }
        if (string.startsWith("thread:")) {
            InternalLogger.warn("\"{thread}\" does not support parameters");
            return new ThreadNameToken();
        }
        if ("thread_id".equals(string)) {
            return new ThreadIdToken();
        }
        if (string.startsWith("thread_id:")) {
            InternalLogger.warn("\"{thread_id}\" does not support parameters");
            return new ThreadIdToken();
        }
        if (string.equals("context")) {
            InternalLogger.error("\"{context}\" requires a key");
            return Tokenizer.getPlainTextToken("");
        }
        if (string.startsWith("context:")) {
            String string3 = string.substring(8, string.length()).trim();
            if (string3.length() == 0) {
                InternalLogger.error("\"{context}\" requires a key");
                return Tokenizer.getPlainTextToken("");
            }
            return new ContextToken(string3);
        }
        if ("class".equals(string)) {
            return new ClassToken();
        }
        if (string.startsWith("class:")) {
            InternalLogger.warn("\"{class}\" does not support parameters");
            return new ClassToken();
        }
        if ("class_name".equals(string)) {
            return new ClassNameToken();
        }
        if (string.startsWith("class_name:")) {
            InternalLogger.warn("\"{class_name}\" does not support parameters");
            return new ClassNameToken();
        }
        if ("package".equals(string)) {
            return new PackageToken();
        }
        if (string.startsWith("package:")) {
            InternalLogger.warn("\"{package}\" does not support parameters");
            return new PackageToken();
        }
        if ("method".equals(string)) {
            return new MethodToken();
        }
        if (string.startsWith("method:")) {
            InternalLogger.warn("\"{method}\" does not support parameters");
            return new MethodToken();
        }
        if ("file".equals(string)) {
            return new FileToken();
        }
        if (string.startsWith("file:")) {
            InternalLogger.warn("\"{file}\" does not support parameters");
            return new FileToken();
        }
        if ("line".equals(string)) {
            return new LineToken();
        }
        if (string.startsWith("line:")) {
            InternalLogger.warn("\"{line}\" does not support parameters");
            return new LineToken();
        }
        if ("level".equals(string)) {
            return new LevelToken();
        }
        if (string.startsWith("level:")) {
            InternalLogger.warn("\"{level}\" does not support parameters");
            return new LevelToken();
        }
        if ("message".equals(string)) {
            return new MessageToken(this.maxStackTraceElements);
        }
        if (string.startsWith("message:")) {
            InternalLogger.warn("\"{message}\" does not support parameters");
            return new MessageToken(this.maxStackTraceElements);
        }
        return Tokenizer.getPlainTextToken(string);
    }

    private static Token getDateToken(String string, Locale locale) {
        if (string.contains("SSSS") || string.contains("n") || string.contains("N")) {
            if (EnvironmentHelper.isAtLeastJava9()) {
                return new PreciseDateToken(string, locale);
            }
            InternalLogger.warn("Java prior version 9 does not support microseconds or nanoseconds");
            return new LegacyDateToken(string, locale);
        }
        return new LegacyDateToken(string, locale);
    }

    private static Token getPlainTextToken(String string) {
        String string2 = NEW_LINE_REPLACER.matcher(string).replaceAll(NEW_LINE);
        string2 = TAB_REPLACER.matcher(string2).replaceAll(TAB);
        return new PlainTextToken(string2);
    }

    private static int[] parseOptions(String string) {
        int n = 0;
        int n2 = 0;
        int n3 = 0;
        while (n3 < string.length()) {
            char c = string.charAt(n3);
            while (c == ',') {
                if (++n3 >= string.length()) {
                    return new int[]{n, n2};
                }
                c = string.charAt(n3);
            }
            int n4 = n3;
            while (c != ',' && ++n3 < string.length()) {
                c = string.charAt(n3);
            }
            if (n3 <= n4) continue;
            String string2 = string.substring(n4, n3);
            int n5 = string2.indexOf(61);
            if (n5 == -1) {
                if ("min-size".equals(string2 = string2.trim())) {
                    InternalLogger.warn("No value set for \"min-size\"");
                    continue;
                }
                if ("indent".equals(string2)) {
                    InternalLogger.warn("No value set for \"indent\"");
                    continue;
                }
                InternalLogger.warn("Unknown option \"{}\"", string2);
                continue;
            }
            String string3 = string2.substring(0, n5).trim();
            String string4 = string2.substring(n5 + 1).trim();
            if ("min-size".equals(string3)) {
                if (string4.length() == 0) {
                    InternalLogger.warn("No value set for \"min-size\"");
                    continue;
                }
                try {
                    n = Tokenizer.parsePositiveInt(string4);
                }
                catch (NumberFormatException numberFormatException) {
                    InternalLogger.warn("\"{}\" is an invalid number for \"min-size\"", string4);
                }
                continue;
            }
            if ("indent".equals(string3)) {
                if (string4.length() == 0) {
                    InternalLogger.warn("No value set for \"indent\"");
                    continue;
                }
                try {
                    n2 = Tokenizer.parsePositiveInt(string4);
                }
                catch (NumberFormatException numberFormatException) {
                    InternalLogger.warn("\"{}\" is an invalid number for \"indent\"", string4);
                }
                continue;
            }
            InternalLogger.warn("Unknown option \"{}\"", string3);
        }
        return new int[]{n, n2};
    }

    private static int parsePositiveInt(String string) throws NumberFormatException {
        int n = Integer.parseInt(string);
        if (n >= 0) {
            return n;
        }
        throw new NumberFormatException();
    }

    private static Token combine(List<Token> list, int[] nArray) {
        int n = nArray[0];
        int n2 = nArray[1];
        if (list.isEmpty()) {
            return null;
        }
        if (list.size() == 1) {
            if (n2 > 0) {
                return new IndentToken(list.get(0), n2);
            }
            if (n > 0) {
                return new MinSizeToken(list.get(0), n);
            }
            return list.get(0);
        }
        if (n2 > 0) {
            return new IndentToken(new BundlerToken(list), n2);
        }
        if (n > 0) {
            return new MinSizeToken(new BundlerToken(list), n);
        }
        return new BundlerToken(list);
    }

    private static final class PlainTextToken
    implements Token {
        private final String text;

        private PlainTextToken(String string) {
            this.text = string;
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return Collections.emptyList();
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            stringBuilder.append(this.text);
        }
    }

    private static final class MessageToken
    implements Token {
        private static final String NEW_LINE = EnvironmentHelper.getNewLine();
        private final int maxStackTraceElements;

        private MessageToken(int n) {
            this.maxStackTraceElements = n;
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return EnumSet.of(LogEntryValue.MESSAGE, LogEntryValue.EXCEPTION);
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            Throwable throwable;
            String string = logEntry.getMessage();
            if (string != null) {
                stringBuilder.append(string);
            }
            if ((throwable = logEntry.getException()) != null) {
                if (string != null) {
                    stringBuilder.append(": ");
                }
                MessageToken.formatException(stringBuilder, throwable, this.maxStackTraceElements);
            }
        }

        private static void formatException(StringBuilder stringBuilder, Throwable throwable, int n) {
            if (n == 0) {
                stringBuilder.append(throwable.getClass().getName());
                String string = throwable.getMessage();
                if (string != null) {
                    stringBuilder.append(": ");
                    stringBuilder.append(string);
                }
            } else {
                MessageToken.formatExceptionWithStackTrace(stringBuilder, throwable, n);
            }
        }

        private static void formatExceptionWithStackTrace(StringBuilder stringBuilder, Throwable throwable, int n) {
            stringBuilder.append(throwable.getClass().getName());
            String string = throwable.getMessage();
            if (string != null) {
                stringBuilder.append(": ");
                stringBuilder.append(string);
            }
            StackTraceElement[] stackTraceElementArray = throwable.getStackTrace();
            int n2 = Math.min(stackTraceElementArray.length, Math.max(1, n));
            for (int i = 0; i < n2; ++i) {
                stringBuilder.append(NEW_LINE);
                stringBuilder.append('\t');
                stringBuilder.append("at ");
                stringBuilder.append(stackTraceElementArray[i]);
            }
            if (stackTraceElementArray.length > n2) {
                stringBuilder.append(NEW_LINE);
                stringBuilder.append('\t');
                stringBuilder.append("...");
            } else {
                Throwable throwable2 = throwable.getCause();
                if (throwable2 != null) {
                    stringBuilder.append(NEW_LINE);
                    stringBuilder.append("Caused by: ");
                    MessageToken.formatExceptionWithStackTrace(stringBuilder, throwable2, n - n2);
                }
            }
        }
    }

    private static final class LevelToken
    implements Token {
        private LevelToken() {
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return Collections.singletonList(LogEntryValue.LEVEL);
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            stringBuilder.append((Object)logEntry.getLevel());
        }
    }

    private static final class LineToken
    implements Token {
        private LineToken() {
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return Collections.singletonList(LogEntryValue.LINE);
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            stringBuilder.append(logEntry.getLineNumber());
        }
    }

    private static final class FileToken
    implements Token {
        private FileToken() {
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return Collections.singletonList(LogEntryValue.FILE);
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            stringBuilder.append(logEntry.getFilename());
        }
    }

    private static final class MethodToken
    implements Token {
        private MethodToken() {
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return Collections.singletonList(LogEntryValue.METHOD);
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            stringBuilder.append(logEntry.getMethodName());
        }
    }

    private static final class PackageToken
    implements Token {
        private PackageToken() {
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return Collections.singletonList(LogEntryValue.CLASS);
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            String string = logEntry.getClassName();
            int n = string.lastIndexOf(46);
            if (n != -1) {
                stringBuilder.append(string.substring(0, n));
            }
        }
    }

    private static final class ClassNameToken
    implements Token {
        private ClassNameToken() {
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return Collections.singletonList(LogEntryValue.CLASS);
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            String string = logEntry.getClassName();
            int n = string.lastIndexOf(46);
            if (n < 0) {
                stringBuilder.append(string);
            } else {
                stringBuilder.append(string.substring(n + 1));
            }
        }
    }

    private static final class ClassToken
    implements Token {
        private ClassToken() {
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return Collections.singletonList(LogEntryValue.CLASS);
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            stringBuilder.append(logEntry.getClassName());
        }
    }

    private static final class ContextToken
    implements Token {
        private final String key;

        private ContextToken(String string) {
            this.key = string;
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return Collections.singletonList(LogEntryValue.CONTEXT);
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            stringBuilder.append(logEntry.getContext().get(this.key));
        }
    }

    private static final class ThreadIdToken
    implements Token {
        private ThreadIdToken() {
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return Collections.singletonList(LogEntryValue.THREAD);
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            stringBuilder.append(logEntry.getThread().getId());
        }
    }

    private static final class ThreadNameToken
    implements Token {
        private ThreadNameToken() {
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return Collections.singletonList(LogEntryValue.THREAD);
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            stringBuilder.append(logEntry.getThread().getName());
        }
    }

    private static final class PreciseDateToken
    implements Token {
        private final DateTimeFormatter formatter;

        private PreciseDateToken(String string, Locale locale) {
            this.formatter = DateTimeFormatter.ofPattern(string, locale).withZone(ZoneId.systemDefault());
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return Collections.singletonList(LogEntryValue.PRECISE_DATE);
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            stringBuilder.append(this.formatter.format(((PreciseLogEntry)logEntry).getInstant()));
        }
    }

    private static final class LegacyDateToken
    implements Token {
        private final DateFormat formatter;
        private final long divisor;
        private Date lastDate;
        private String lastFormat;

        private LegacyDateToken(String string, Locale locale) {
            this.formatter = new SimpleDateFormat(string, locale);
            this.divisor = string.contains("S") ? 1L : (string.contains("s") ? 1000L : 60000L);
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return Collections.singletonList(LogEntryValue.DATE);
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            stringBuilder.append(this.format(logEntry.getDate()));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private String format(Date date) {
            DateFormat dateFormat = this.formatter;
            synchronized (dateFormat) {
                if (this.lastDate != null && date.getTime() / this.divisor == this.lastDate.getTime() / this.divisor) {
                    return this.lastFormat;
                }
                this.lastDate = date;
                this.lastFormat = this.formatter.format(date);
                return this.lastFormat;
            }
        }
    }

    private static final class IndentToken
    implements Token {
        private final Token token;
        private final char[] spaces;

        private IndentToken(Token token, int n) {
            this.token = token;
            this.spaces = new char[n];
            Arrays.fill(this.spaces, ' ');
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return this.token.getRequiredLogEntryValues();
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            StringBuilder stringBuilder2 = new StringBuilder(1024);
            this.token.render(logEntry, stringBuilder2);
            int n = 0;
            int n2 = 0;
            if (stringBuilder.length() == 0 || stringBuilder.charAt(stringBuilder.length() - 1) == '\n' || stringBuilder.charAt(stringBuilder.length() - 1) == '\r') {
                n += this.addSpaces(stringBuilder, stringBuilder2, n);
                n2 = n;
            }
            while (n < stringBuilder2.length()) {
                int n3 = IndentToken.readLineBreak(stringBuilder2, n);
                if (n3 > 0) {
                    stringBuilder.append(stringBuilder2, n2, n += n3);
                    n += this.addSpaces(stringBuilder, stringBuilder2, n);
                    n2 = n;
                    continue;
                }
                ++n;
            }
            if (n2 < stringBuilder2.length()) {
                stringBuilder.append(stringBuilder2, n2, stringBuilder2.length());
            }
        }

        private int addSpaces(StringBuilder stringBuilder, StringBuilder stringBuilder2, int n) {
            stringBuilder.append(this.spaces);
            int n2 = 0;
            for (int i = n; i < stringBuilder2.length() && stringBuilder2.charAt(i) == '\t'; ++i) {
                stringBuilder.append(this.spaces);
                ++n2;
            }
            return n2;
        }

        private static int readLineBreak(StringBuilder stringBuilder, int n) {
            char c = stringBuilder.charAt(n);
            if (c == '\n') {
                return 1;
            }
            if (c == '\r') {
                if (n + 1 < stringBuilder.length() && stringBuilder.charAt(n + 1) == '\n') {
                    return 2;
                }
                return 1;
            }
            return 0;
        }
    }

    private static final class MinSizeToken
    implements Token {
        private final Token token;
        private final int minSize;

        private MinSizeToken(Token token, int n) {
            this.token = token;
            this.minSize = n;
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            return this.token.getRequiredLogEntryValues();
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            int n = stringBuilder.length();
            this.token.render(logEntry, stringBuilder);
            int n2 = stringBuilder.length() - n;
            if (n2 < this.minSize) {
                char[] cArray = new char[this.minSize - n2];
                Arrays.fill(cArray, ' ');
                stringBuilder.append(cArray);
            }
        }
    }

    private static final class BundlerToken
    implements Token {
        private final List<Token> tokens;

        private BundlerToken(List<Token> list) {
            this.tokens = list;
        }

        @Override
        public Collection<LogEntryValue> getRequiredLogEntryValues() {
            EnumSet<LogEntryValue> enumSet = EnumSet.noneOf(LogEntryValue.class);
            for (Token token : this.tokens) {
                enumSet.addAll(token.getRequiredLogEntryValues());
            }
            return enumSet;
        }

        @Override
        public void render(LogEntry logEntry, StringBuilder stringBuilder) {
            for (Token token : this.tokens) {
                token.render(logEntry, stringBuilder);
            }
        }
    }
}

