/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.api.batch.fs.internal;

import java.io.File;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import javax.annotation.Nullable;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.FileUtils;
import org.assertj.core.api.Assertions;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.sonar.api.batch.fs.InputFile;
import org.sonar.api.batch.fs.internal.DefaultInputFile;
import org.sonar.api.batch.fs.internal.FileMetadata;
import org.sonar.api.batch.fs.internal.Metadata;
import org.sonar.api.batch.fs.internal.TestInputFileBuilder;
import org.sonar.api.utils.log.LogTester;
import org.sonar.api.utils.log.LoggerLevel;

public class FileMetadataTest {
    @Rule
    public ExpectedException thrown = ExpectedException.none();
    @Rule
    public TemporaryFolder temp = new TemporaryFolder();
    @Rule
    public LogTester logTester = new LogTester();

    @Test
    public void empty_file() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.touch((File)tempFile);
        Metadata metadata = new FileMetadata().readMetadata(tempFile, StandardCharsets.UTF_8);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(1);
        Assertions.assertThat((int)metadata.nonBlankLines()).isEqualTo(0);
        Assertions.assertThat((String)metadata.hash()).isNotEmpty();
        Assertions.assertThat((int[])metadata.originalLineOffsets()).containsOnly(new int[]{0});
        Assertions.assertThat((int)metadata.lastValidOffset()).isEqualTo(0);
    }

    @Test
    public void windows_without_latest_eol() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)"foo\r\nbar\r\nbaz", (Charset)StandardCharsets.UTF_8, (boolean)true);
        Metadata metadata = new FileMetadata().readMetadata(tempFile, StandardCharsets.UTF_8);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(3);
        Assertions.assertThat((int)metadata.nonBlankLines()).isEqualTo(3);
        Assertions.assertThat((String)metadata.hash()).isEqualTo((Object)DigestUtils.md5Hex((String)"foo\nbar\nbaz"));
        Assertions.assertThat((int[])metadata.originalLineOffsets()).containsOnly(new int[]{0, 5, 10});
        Assertions.assertThat((int)metadata.lastValidOffset()).isEqualTo(13);
    }

    @Test
    public void read_with_wrong_encoding() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)"marker\u00b4s\n", (Charset)Charset.forName("cp1252"));
        Metadata metadata = new FileMetadata().readMetadata(tempFile, StandardCharsets.UTF_8);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(2);
        Assertions.assertThat((String)metadata.hash()).isEqualTo((Object)DigestUtils.md5Hex((String)"marker\ufffds\n"));
        Assertions.assertThat((int[])metadata.originalLineOffsets()).containsOnly(new int[]{0, 9});
    }

    @Test
    public void non_ascii_utf_8() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)"f\u00f6o\r\nb\u00e0r\r\n\u1d11Eba\u00dfz\r\n", (Charset)StandardCharsets.UTF_8, (boolean)true);
        Metadata metadata = new FileMetadata().readMetadata(tempFile, StandardCharsets.UTF_8);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(4);
        Assertions.assertThat((int)metadata.nonBlankLines()).isEqualTo(3);
        Assertions.assertThat((String)metadata.hash()).isEqualTo((Object)DigestUtils.md5Hex((String)"f\u00f6o\nb\u00e0r\n\u1d11Eba\u00dfz\n"));
        Assertions.assertThat((int[])metadata.originalLineOffsets()).containsOnly(new int[]{0, 5, 10, 18});
    }

    @Test
    public void non_ascii_utf_16() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)"f\u00f6o\r\nb\u00e0r\r\n\u1d11Eba\u00dfz\r\n", (Charset)StandardCharsets.UTF_16, (boolean)true);
        Metadata metadata = new FileMetadata().readMetadata(tempFile, StandardCharsets.UTF_16);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(4);
        Assertions.assertThat((int)metadata.nonBlankLines()).isEqualTo(3);
        Assertions.assertThat((String)metadata.hash()).isEqualTo((Object)DigestUtils.md5Hex((byte[])"f\u00f6o\nb\u00e0r\n\u1d11Eba\u00dfz\n".getBytes(StandardCharsets.UTF_8)));
        Assertions.assertThat((int[])metadata.originalLineOffsets()).containsOnly(new int[]{0, 5, 10, 18});
    }

    @Test
    public void unix_without_latest_eol() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)"foo\nbar\nbaz", (Charset)StandardCharsets.UTF_8, (boolean)true);
        Metadata metadata = new FileMetadata().readMetadata(tempFile, StandardCharsets.UTF_8);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(3);
        Assertions.assertThat((int)metadata.nonBlankLines()).isEqualTo(3);
        Assertions.assertThat((String)metadata.hash()).isEqualTo((Object)DigestUtils.md5Hex((String)"foo\nbar\nbaz"));
        Assertions.assertThat((int[])metadata.originalLineOffsets()).containsOnly(new int[]{0, 4, 8});
        Assertions.assertThat((int)metadata.lastValidOffset()).isEqualTo(11);
    }

    @Test
    public void unix_with_latest_eol() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)"foo\nbar\nbaz\n", (Charset)StandardCharsets.UTF_8, (boolean)true);
        Metadata metadata = new FileMetadata().readMetadata(tempFile, StandardCharsets.UTF_8);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(4);
        Assertions.assertThat((int)metadata.nonBlankLines()).isEqualTo(3);
        Assertions.assertThat((String)metadata.hash()).isEqualTo((Object)DigestUtils.md5Hex((String)"foo\nbar\nbaz\n"));
        Assertions.assertThat((int[])metadata.originalLineOffsets()).containsOnly(new int[]{0, 4, 8, 12});
        Assertions.assertThat((int)metadata.lastValidOffset()).isEqualTo(12);
    }

    @Test
    public void mac_without_latest_eol() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)"foo\rbar\rbaz", (Charset)StandardCharsets.UTF_8, (boolean)true);
        Metadata metadata = new FileMetadata().readMetadata(tempFile, StandardCharsets.UTF_8);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(3);
        Assertions.assertThat((int)metadata.nonBlankLines()).isEqualTo(3);
        Assertions.assertThat((String)metadata.hash()).isEqualTo((Object)DigestUtils.md5Hex((String)"foo\nbar\nbaz"));
        Assertions.assertThat((int[])metadata.originalLineOffsets()).containsOnly(new int[]{0, 4, 8});
        Assertions.assertThat((int)metadata.lastValidOffset()).isEqualTo(11);
    }

    @Test
    public void mac_with_latest_eol() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)"foo\rbar\rbaz\r", (Charset)StandardCharsets.UTF_8, (boolean)true);
        Metadata metadata = new FileMetadata().readMetadata(tempFile, StandardCharsets.UTF_8);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(4);
        Assertions.assertThat((int)metadata.nonBlankLines()).isEqualTo(3);
        Assertions.assertThat((String)metadata.hash()).isEqualTo((Object)DigestUtils.md5Hex((String)"foo\nbar\nbaz\n"));
        Assertions.assertThat((int[])metadata.originalLineOffsets()).containsOnly(new int[]{0, 4, 8, 12});
        Assertions.assertThat((int)metadata.lastValidOffset()).isEqualTo(12);
    }

    @Test
    public void mix_of_newlines_with_latest_eol() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)"foo\nbar\r\nbaz\n", (Charset)StandardCharsets.UTF_8, (boolean)true);
        Metadata metadata = new FileMetadata().readMetadata(tempFile, StandardCharsets.UTF_8);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(4);
        Assertions.assertThat((int)metadata.nonBlankLines()).isEqualTo(3);
        Assertions.assertThat((String)metadata.hash()).isEqualTo((Object)DigestUtils.md5Hex((String)"foo\nbar\nbaz\n"));
        Assertions.assertThat((int[])metadata.originalLineOffsets()).containsOnly(new int[]{0, 4, 9, 13});
    }

    @Test
    public void several_new_lines() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)"foo\n\n\nbar", (Charset)StandardCharsets.UTF_8, (boolean)true);
        Metadata metadata = new FileMetadata().readMetadata(tempFile, StandardCharsets.UTF_8);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(4);
        Assertions.assertThat((int)metadata.nonBlankLines()).isEqualTo(2);
        Assertions.assertThat((String)metadata.hash()).isEqualTo((Object)DigestUtils.md5Hex((String)"foo\n\n\nbar"));
        Assertions.assertThat((int[])metadata.originalLineOffsets()).containsOnly(new int[]{0, 4, 5, 6});
    }

    @Test
    public void mix_of_newlines_without_latest_eol() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)"foo\nbar\r\nbaz", (Charset)StandardCharsets.UTF_8, (boolean)true);
        Metadata metadata = new FileMetadata().readMetadata(tempFile, StandardCharsets.UTF_8);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(3);
        Assertions.assertThat((int)metadata.nonBlankLines()).isEqualTo(3);
        Assertions.assertThat((String)metadata.hash()).isEqualTo((Object)DigestUtils.md5Hex((String)"foo\nbar\nbaz"));
        Assertions.assertThat((int[])metadata.originalLineOffsets()).containsOnly(new int[]{0, 4, 9});
    }

    @Test
    public void start_with_newline() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)"\nfoo\nbar\r\nbaz", (Charset)StandardCharsets.UTF_8, (boolean)true);
        Metadata metadata = new FileMetadata().readMetadata(tempFile, StandardCharsets.UTF_8);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(4);
        Assertions.assertThat((int)metadata.nonBlankLines()).isEqualTo(3);
        Assertions.assertThat((String)metadata.hash()).isEqualTo((Object)DigestUtils.md5Hex((String)"\nfoo\nbar\nbaz"));
        Assertions.assertThat((int[])metadata.originalLineOffsets()).containsOnly(new int[]{0, 1, 5, 10});
    }

    @Test
    public void start_with_bom() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)"\ufefffoo\nbar\r\nbaz", (Charset)StandardCharsets.UTF_8, (boolean)true);
        Metadata metadata = new FileMetadata().readMetadata(tempFile, StandardCharsets.UTF_8);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(3);
        Assertions.assertThat((int)metadata.nonBlankLines()).isEqualTo(3);
        Assertions.assertThat((String)metadata.hash()).isEqualTo((Object)DigestUtils.md5Hex((String)"foo\nbar\nbaz"));
        Assertions.assertThat((int[])metadata.originalLineOffsets()).containsOnly(new int[]{0, 4, 9});
    }

    @Test
    public void ignore_whitespace_when_computing_line_hashes() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)" foo\nb ar\r\nbaz \t", (Charset)StandardCharsets.UTF_8, (boolean)true);
        DefaultInputFile f = new TestInputFileBuilder("foo", tempFile.getName()).setModuleBaseDir(tempFile.getParentFile().toPath()).setCharset(StandardCharsets.UTF_8).build();
        FileMetadata.computeLineHashesForIssueTracking((InputFile)f, (FileMetadata.LineHashConsumer)new FileMetadata.LineHashConsumer(){

            public void consume(int lineIdx, @Nullable byte[] hash) {
                switch (lineIdx) {
                    case 1: {
                        Assertions.assertThat((String)Hex.encodeHexString((byte[])hash)).isEqualTo((Object)DigestUtils.md5Hex((String)"foo"));
                        break;
                    }
                    case 2: {
                        Assertions.assertThat((String)Hex.encodeHexString((byte[])hash)).isEqualTo((Object)DigestUtils.md5Hex((String)"bar"));
                        break;
                    }
                    case 3: {
                        Assertions.assertThat((String)Hex.encodeHexString((byte[])hash)).isEqualTo((Object)DigestUtils.md5Hex((String)"baz"));
                        break;
                    }
                    default: {
                        Assertions.fail((String)"Invalid line");
                    }
                }
            }
        });
    }

    @Test
    public void dont_fail_on_empty_file() throws Exception {
        File tempFile = this.temp.newFile();
        FileUtils.write((File)tempFile, (CharSequence)"", (Charset)StandardCharsets.UTF_8, (boolean)true);
        DefaultInputFile f = new TestInputFileBuilder("foo", tempFile.getName()).setModuleBaseDir(tempFile.getParentFile().toPath()).setCharset(StandardCharsets.UTF_8).build();
        FileMetadata.computeLineHashesForIssueTracking((InputFile)f, (FileMetadata.LineHashConsumer)new FileMetadata.LineHashConsumer(){

            public void consume(int lineIdx, @Nullable byte[] hash) {
                switch (lineIdx) {
                    case 1: {
                        Assertions.assertThat((byte[])hash).isNull();
                        break;
                    }
                    default: {
                        Assertions.fail((String)"Invalid line");
                    }
                }
            }
        });
    }

    @Test
    public void should_throw_if_file_does_not_exist() throws Exception {
        File tempFolder = this.temp.newFolder();
        File file = new File(tempFolder, "doesNotExist.txt");
        this.thrown.expect(IllegalStateException.class);
        this.thrown.expectMessage("Fail to read file '" + file.getAbsolutePath() + "' with encoding 'UTF-8'");
        new FileMetadata().readMetadata(file, StandardCharsets.UTF_8);
    }

    @Test
    public void line_feed_is_included_into_hash() throws Exception {
        File file1 = this.temp.newFile();
        FileUtils.write((File)file1, (CharSequence)"foo\nbar\n", (Charset)StandardCharsets.UTF_8, (boolean)true);
        File file1a = this.temp.newFile();
        FileUtils.write((File)file1a, (CharSequence)"foo\r\nbar\n", (Charset)StandardCharsets.UTF_8, (boolean)true);
        File file2 = this.temp.newFile();
        FileUtils.write((File)file2, (CharSequence)"foo\nbar", (Charset)StandardCharsets.UTF_8, (boolean)true);
        String hash1 = new FileMetadata().readMetadata(file1, StandardCharsets.UTF_8).hash();
        String hash1a = new FileMetadata().readMetadata(file1a, StandardCharsets.UTF_8).hash();
        String hash2 = new FileMetadata().readMetadata(file2, StandardCharsets.UTF_8).hash();
        Assertions.assertThat((String)hash1).isEqualTo((Object)hash1a);
        Assertions.assertThat((String)hash1).isNotEqualTo((Object)hash2);
    }

    @Test
    public void binary_file_with_unmappable_character() throws Exception {
        File woff = new File(this.getClass().getResource("glyphicons-halflings-regular.woff").toURI());
        Metadata metadata = new FileMetadata().readMetadata(woff, StandardCharsets.UTF_8);
        Assertions.assertThat((int)metadata.lines()).isEqualTo(135);
        Assertions.assertThat((int)metadata.nonBlankLines()).isEqualTo(133);
        Assertions.assertThat((String)metadata.hash()).isNotEmpty();
        Assertions.assertThat((String)((String)this.logTester.logs(LoggerLevel.WARN).get(0))).contains(new CharSequence[]{"Invalid character encountered in file"});
        Assertions.assertThat((String)((String)this.logTester.logs(LoggerLevel.WARN).get(0))).contains(new CharSequence[]{"glyphicons-halflings-regular.woff at line 1 for encoding UTF-8. Please fix file content or configure the encoding to be used using property 'sonar.sourceEncoding'."});
    }
}

