/*
 * Decompiled with CFR 0.152.
 */
package org.apache.metamodel.excel;

import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.NumberFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import org.apache.metamodel.MetaModelException;
import org.apache.metamodel.data.DataSet;
import org.apache.metamodel.data.DataSetHeader;
import org.apache.metamodel.data.DefaultRow;
import org.apache.metamodel.data.EmptyDataSet;
import org.apache.metamodel.data.Style;
import org.apache.metamodel.data.StyleBuilder;
import org.apache.metamodel.excel.ExcelConfiguration;
import org.apache.metamodel.excel.ExcelDataContext;
import org.apache.metamodel.excel.XlsDataSet;
import org.apache.metamodel.excel.ZeroBasedRowIterator;
import org.apache.metamodel.query.SelectItem;
import org.apache.metamodel.schema.Table;
import org.apache.metamodel.util.Action;
import org.apache.metamodel.util.DateUtils;
import org.apache.metamodel.util.FileHelper;
import org.apache.metamodel.util.FileResource;
import org.apache.metamodel.util.FormatHelper;
import org.apache.metamodel.util.InMemoryResource;
import org.apache.metamodel.util.Resource;
import org.apache.poi.hssf.usermodel.HSSFDateUtil;
import org.apache.poi.hssf.usermodel.HSSFFont;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.ss.formula.FormulaParseException;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Color;
import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.FontUnderline;
import org.apache.poi.ss.usermodel.FormulaError;
import org.apache.poi.ss.usermodel.FormulaEvaluator;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFColor;
import org.apache.poi.xssf.usermodel.XSSFFont;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.XMLReader;

final class ExcelUtils {
    private static final Logger logger = LoggerFactory.getLogger(ExcelUtils.class);
    private static final NumberFormat _numberFormat = FormatHelper.getUiNumberFormat();

    private ExcelUtils() {
    }

    public static XMLReader createXmlReader() {
        try {
            SAXParserFactory saxFactory = SAXParserFactory.newInstance();
            SAXParser saxParser = saxFactory.newSAXParser();
            XMLReader sheetParser = saxParser.getXMLReader();
            return sheetParser;
        }
        catch (Exception e) {
            throw new MetaModelException(e);
        }
    }

    public static Workbook readWorkbook(Resource resource) {
        if (!resource.isExists()) {
            if (ExcelUtils.isXlsxFile(resource)) {
                return new SXSSFWorkbook(1000);
            }
            return new HSSFWorkbook();
        }
        if (resource instanceof FileResource) {
            File file = ((FileResource)resource).getFile();
            try {
                return WorkbookFactory.create((File)file);
            }
            catch (Exception e) {
                logger.error("Could not open workbook", (Throwable)e);
                throw new IllegalStateException("Could not open workbook", e);
            }
        }
        return (Workbook)resource.read(inputStream -> {
            try {
                return WorkbookFactory.create((InputStream)inputStream);
            }
            catch (Exception e) {
                logger.error("Could not open workbook", (Throwable)e);
                throw new IllegalStateException("Could not open workbook", e);
            }
        });
    }

    public static boolean isXlsxFile(Resource resource) {
        if (resource == null) {
            return false;
        }
        return resource.getName().toLowerCase().endsWith(".xlsx");
    }

    public static Workbook readWorkbook(ExcelDataContext dataContext) {
        Resource resource = dataContext.getResource();
        return ExcelUtils.readWorkbook(resource);
    }

    public static void writeAndCloseWorkbook(ExcelDataContext dataContext, final Workbook wb) {
        Resource realResource = dataContext.getResource();
        InMemoryResource tempResource = new InMemoryResource(realResource.getQualifiedPath());
        tempResource.write((Action)new Action<OutputStream>(){

            public void run(OutputStream outputStream) throws Exception {
                wb.write(outputStream);
            }
        });
        if (wb instanceof HSSFWorkbook && realResource instanceof FileResource && realResource.isExists()) {
            System.gc();
            System.runFinalization();
            try {
                Thread.sleep(800L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        FileHelper.safeClose((Object[])new Object[]{wb});
        FileHelper.copy((Resource)tempResource, (Resource)realResource);
    }

    public static String getCellValue(Workbook wb, Cell cell) {
        String result;
        if (cell == null) {
            return null;
        }
        String cellCoordinate = "(" + cell.getRowIndex() + "," + cell.getColumnIndex() + ")";
        switch (cell.getCellTypeEnum()) {
            case BLANK: 
            case _NONE: {
                result = null;
                break;
            }
            case BOOLEAN: {
                result = Boolean.toString(cell.getBooleanCellValue());
                break;
            }
            case ERROR: {
                String errorResult;
                try {
                    byte errorCode = cell.getErrorCellValue();
                    FormulaError formulaError = FormulaError.forInt((byte)errorCode);
                    errorResult = formulaError.getString();
                }
                catch (RuntimeException e) {
                    logger.debug("Getting error code for {} failed!: {}", (Object)cellCoordinate, (Object)e.getMessage());
                    if (cell instanceof XSSFCell) {
                        String value;
                        errorResult = value = ((XSSFCell)cell).getErrorCellString();
                    }
                    logger.error("Couldn't handle unexpected error scenario in cell: " + cellCoordinate, (Throwable)e);
                    throw e;
                }
                result = errorResult;
                break;
            }
            case FORMULA: {
                result = ExcelUtils.getFormulaCellValue(wb, cell);
                break;
            }
            case NUMERIC: {
                if (HSSFDateUtil.isCellDateFormatted((Cell)cell)) {
                    Date date = cell.getDateCellValue();
                    if (date == null) {
                        result = null;
                        break;
                    }
                    result = DateUtils.createDateFormat().format(date);
                    break;
                }
                result = _numberFormat.format(cell.getNumericCellValue());
                break;
            }
            case STRING: {
                result = cell.getRichStringCellValue().getString();
                break;
            }
            default: {
                throw new IllegalStateException("Unknown cell type: " + cell.getCellTypeEnum());
            }
        }
        logger.debug("cell {} resolved to value: {}", (Object)cellCoordinate, (Object)result);
        return result;
    }

    private static String getFormulaCellValue(Workbook wb, Cell cell) {
        try {
            double numericCellValue = cell.getNumericCellValue();
            return _numberFormat.format(numericCellValue);
        }
        catch (Exception e) {
            if (logger.isInfoEnabled()) {
                logger.info("Failed to fetch cached/precalculated formula value of cell: " + cell, (Throwable)e);
            }
            try {
                if (logger.isInfoEnabled()) {
                    logger.info("cell({},{}) is a formula. Attempting to evaluate: {}", new Object[]{cell.getRowIndex(), cell.getColumnIndex(), cell.getCellFormula()});
                }
                FormulaEvaluator evaluator = wb.getCreationHelper().createFormulaEvaluator();
                Cell evaluatedCell = evaluator.evaluateInCell(cell);
                return ExcelUtils.getCellValue(wb, evaluatedCell);
            }
            catch (RuntimeException e2) {
                logger.warn("Exception occurred while evaluating formula at position ({},{}): {}", new Object[]{cell.getRowIndex(), cell.getColumnIndex(), e2.getMessage()});
                if (e2 instanceof FormulaParseException) {
                    logger.error("Parse exception occurred while evaluating cell formula: " + cell, (Throwable)e2);
                } else if (e2 instanceof IllegalArgumentException) {
                    logger.error("Illegal formula argument occurred while evaluating cell formula: " + cell, (Throwable)e2);
                } else {
                    logger.error("Unexpected exception occurred while evaluating cell formula: " + cell, (Throwable)e2);
                }
                return cell.getCellFormula();
            }
        }
    }

    public static Style getCellStyle(Workbook workbook, Cell cell) {
        XSSFColor color;
        if (cell == null) {
            return Style.NO_STYLE;
        }
        CellStyle cellStyle = cell.getCellStyle();
        short fontIndex = cellStyle.getFontIndex();
        Font font = workbook.getFontAt(fontIndex);
        StyleBuilder styleBuilder = new StyleBuilder();
        if (font.getBold()) {
            styleBuilder.bold();
        }
        if (font.getItalic()) {
            styleBuilder.italic();
        }
        if (font.getUnderline() != FontUnderline.NONE.getByteValue()) {
            styleBuilder.underline();
        }
        Font stdFont = workbook.getFontAt((short)0);
        short fontSize = font.getFontHeightInPoints();
        if (stdFont.getFontHeightInPoints() != fontSize) {
            styleBuilder.fontSize((int)fontSize, Style.SizeUnit.PT);
        }
        short colorIndex = font.getColor();
        if (font instanceof HSSFFont) {
            HSSFWorkbook wb;
            if (colorIndex != Short.MAX_VALUE && (color = (wb = (HSSFWorkbook)workbook).getCustomPalette().getColor(colorIndex)) != null) {
                short[] triplet = color.getTriplet();
                styleBuilder.foreground(triplet);
            }
        } else if (font instanceof XSSFFont) {
            String argbHex;
            XSSFFont xssfFont = (XSSFFont)font;
            color = xssfFont.getXSSFColor();
            if (color != null && (argbHex = color.getARGBHex()) != null) {
                styleBuilder.foreground(argbHex.substring(2));
            }
        } else {
            throw new IllegalStateException("Unexpected font type: " + (font == null ? "null" : font.getClass()) + ")");
        }
        if (cellStyle.getFillPatternEnum() == FillPatternType.SOLID_FOREGROUND) {
            Color color2 = cellStyle.getFillForegroundColorColor();
            if (color2 instanceof HSSFColor) {
                short[] triplet = ((HSSFColor)color2).getTriplet();
                if (triplet != null) {
                    styleBuilder.background(triplet);
                }
            } else if (color2 instanceof XSSFColor) {
                String argb = ((XSSFColor)color2).getARGBHex();
                if (argb != null) {
                    styleBuilder.background(argb.substring(2));
                }
            } else {
                throw new IllegalStateException("Unexpected color type: " + (color2 == null ? "null" : color2.getClass()) + ")");
            }
        }
        switch (cellStyle.getAlignmentEnum()) {
            case LEFT: {
                styleBuilder.leftAligned();
                break;
            }
            case RIGHT: {
                styleBuilder.rightAligned();
                break;
            }
            case CENTER: {
                styleBuilder.centerAligned();
                break;
            }
            case JUSTIFY: {
                styleBuilder.justifyAligned();
                break;
            }
        }
        return styleBuilder.create();
    }

    public static Iterator<Row> getRowIterator(Sheet sheet, ExcelConfiguration configuration, boolean jumpToDataRows) {
        int columnNameLineNumber;
        Iterator iterator = configuration.isSkipEmptyLines() ? sheet.rowIterator() : new ZeroBasedRowIterator(sheet);
        if (jumpToDataRows && (columnNameLineNumber = configuration.getColumnNameLineNumber()) != 0) {
            if (iterator.hasNext()) {
                iterator.next();
            }
            for (int i = 1; i < columnNameLineNumber && iterator.hasNext(); ++i) {
                iterator.next();
            }
        }
        return iterator;
    }

    public static DefaultRow createRow(Workbook workbook, Row row, DataSetHeader header) {
        int size = header.size();
        Object[] values = new String[size];
        Style[] styles = new Style[size];
        if (row != null) {
            for (int i = 0; i < size; ++i) {
                int columnNumber = header.getSelectItem(i).getColumn().getColumnNumber();
                Cell cell = row.getCell(columnNumber);
                String value = ExcelUtils.getCellValue(workbook, cell);
                Style style = ExcelUtils.getCellStyle(workbook, cell);
                values[i] = value;
                styles[i] = style;
            }
        }
        return new DefaultRow(header, values, styles);
    }

    public static DataSet getDataSet(Workbook workbook, Sheet sheet, Table table, ExcelConfiguration configuration) {
        List<SelectItem> selectItems = table.getColumns().stream().map(SelectItem::new).collect(Collectors.toList());
        Iterator<Row> rowIterator = ExcelUtils.getRowIterator(sheet, configuration, true);
        if (!rowIterator.hasNext()) {
            FileHelper.safeClose((Object[])new Object[]{workbook});
            return new EmptyDataSet(selectItems);
        }
        XlsDataSet dataSet = new XlsDataSet(selectItems, workbook, rowIterator);
        return dataSet;
    }
}

