/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.record;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Constructor;
import java.nio.ByteBuffer;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.record.ByteBufferInputStream;
import org.apache.kafka.common.record.ByteBufferOutputStream;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.record.LogEntry;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.TimestampType;

public class MemoryRecordsBuilder {
    private static final float COMPRESSION_RATE_DAMPING_FACTOR = 0.9f;
    private static final float COMPRESSION_RATE_ESTIMATION_FACTOR = 1.05f;
    private static final int COMPRESSION_DEFAULT_BUFFER_SIZE = 1024;
    private static final float[] TYPE_TO_RATE;
    private static MemoizingConstructorSupplier snappyOutputStreamSupplier;
    private static MemoizingConstructorSupplier lz4OutputStreamSupplier;
    private static MemoizingConstructorSupplier snappyInputStreamSupplier;
    private static MemoizingConstructorSupplier lz4InputStreamSupplier;
    private final TimestampType timestampType;
    private final CompressionType compressionType;
    private final DataOutputStream appendStream;
    private final ByteBufferOutputStream bufferStream;
    private final byte magic;
    private final int initPos;
    private final long baseOffset;
    private final long logAppendTime;
    private final int writeLimit;
    private final int initialCapacity;
    private long writtenUncompressed = 0L;
    private long numRecords = 0L;
    private float compressionRate = 1.0f;
    private long maxTimestamp = -1L;
    private long offsetOfMaxTimestamp = -1L;
    private long lastOffset = -1L;
    private MemoryRecords builtRecords;

    public MemoryRecordsBuilder(ByteBuffer buffer, byte magic, CompressionType compressionType, TimestampType timestampType, long baseOffset, long logAppendTime, int writeLimit) {
        this.magic = magic;
        this.timestampType = timestampType;
        this.compressionType = compressionType;
        this.baseOffset = baseOffset;
        this.logAppendTime = logAppendTime;
        this.initPos = buffer.position();
        this.writeLimit = writeLimit;
        this.initialCapacity = buffer.capacity();
        if (compressionType != CompressionType.NONE) {
            buffer.position(this.initPos + 12 + Record.recordOverhead(magic));
        }
        this.bufferStream = new ByteBufferOutputStream(buffer);
        this.appendStream = MemoryRecordsBuilder.wrapForOutput(this.bufferStream, compressionType, magic, 1024);
    }

    public ByteBuffer buffer() {
        return this.bufferStream.buffer();
    }

    public int initialCapacity() {
        return this.initialCapacity;
    }

    public double compressionRate() {
        return this.compressionRate;
    }

    public MemoryRecords build() {
        this.close();
        return this.builtRecords;
    }

    public RecordsInfo info() {
        if (this.timestampType == TimestampType.LOG_APPEND_TIME) {
            return new RecordsInfo(this.logAppendTime, this.lastOffset);
        }
        if (this.maxTimestamp == -1L) {
            return new RecordsInfo(-1L, this.lastOffset);
        }
        return new RecordsInfo(this.maxTimestamp, this.compressionType == CompressionType.NONE ? this.offsetOfMaxTimestamp : this.lastOffset);
    }

    public void close() {
        if (this.builtRecords != null) {
            return;
        }
        try {
            this.appendStream.close();
        }
        catch (IOException e) {
            throw new KafkaException(e);
        }
        if (this.compressionType != CompressionType.NONE) {
            this.writerCompressedWrapperHeader();
        }
        ByteBuffer buffer = this.buffer().duplicate();
        buffer.flip();
        buffer.position(this.initPos);
        this.builtRecords = MemoryRecords.readableRecords(buffer.slice());
    }

    private void writerCompressedWrapperHeader() {
        ByteBuffer buffer = this.bufferStream.buffer();
        int pos = buffer.position();
        buffer.position(this.initPos);
        int wrapperSize = pos - this.initPos - 12;
        int writtenCompressed = wrapperSize - Record.recordOverhead(this.magic);
        LogEntry.writeHeader(buffer, this.lastOffset, wrapperSize);
        long timestamp = this.timestampType == TimestampType.LOG_APPEND_TIME ? this.logAppendTime : this.maxTimestamp;
        Record.writeCompressedRecordHeader(buffer, this.magic, wrapperSize, timestamp, this.compressionType, this.timestampType);
        buffer.position(pos);
        this.compressionRate = (float)writtenCompressed / (float)this.writtenUncompressed;
        MemoryRecordsBuilder.TYPE_TO_RATE[this.compressionType.id] = TYPE_TO_RATE[this.compressionType.id] * 0.9f + this.compressionRate * 0.100000024f;
    }

    public long appendWithOffset(long offset, long timestamp, byte[] key, byte[] value) {
        try {
            if (this.lastOffset >= 0L && offset <= this.lastOffset) {
                throw new IllegalArgumentException(String.format("Illegal offset %s following previous offset %s (Offsets must increase monotonically).", offset, this.lastOffset));
            }
            int size = Record.recordSize(this.magic, key, value);
            LogEntry.writeHeader(this.appendStream, this.toInnerOffset(offset), size);
            if (this.timestampType == TimestampType.LOG_APPEND_TIME) {
                timestamp = this.logAppendTime;
            }
            long crc = Record.write(this.appendStream, this.magic, timestamp, key, value, CompressionType.NONE, this.timestampType);
            this.recordWritten(offset, timestamp, size + 12);
            return crc;
        }
        catch (IOException e) {
            throw new KafkaException("I/O exception when writing to the append stream, closing", e);
        }
    }

    public long append(long timestamp, byte[] key, byte[] value) {
        return this.appendWithOffset(this.lastOffset < 0L ? this.baseOffset : this.lastOffset + 1L, timestamp, key, value);
    }

    public void convertAndAppend(Record record) {
        this.convertAndAppendWithOffset(this.lastOffset < 0L ? this.baseOffset : this.lastOffset + 1L, record);
    }

    public void convertAndAppendWithOffset(long offset, Record record) {
        if (this.magic == record.magic()) {
            this.appendWithOffset(offset, record);
            return;
        }
        if (this.lastOffset >= 0L && offset <= this.lastOffset) {
            throw new IllegalArgumentException(String.format("Illegal offset %s following previous offset %s (Offsets must increase monotonically).", offset, this.lastOffset));
        }
        try {
            int size = record.convertedSize(this.magic);
            LogEntry.writeHeader(this.appendStream, this.toInnerOffset(offset), size);
            long timestamp = this.timestampType == TimestampType.LOG_APPEND_TIME ? this.logAppendTime : record.timestamp();
            record.convertTo(this.appendStream, this.magic, timestamp, this.timestampType);
            this.recordWritten(offset, timestamp, size + 12);
        }
        catch (IOException e) {
            throw new KafkaException("I/O exception when writing to the append stream, closing", e);
        }
    }

    public void appendUnchecked(long offset, Record record) {
        try {
            int size = record.sizeInBytes();
            LogEntry.writeHeader(this.appendStream, this.toInnerOffset(offset), size);
            ByteBuffer buffer = record.buffer().duplicate();
            this.appendStream.write(buffer.array(), buffer.arrayOffset(), buffer.limit());
            this.recordWritten(offset, record.timestamp(), size + 12);
        }
        catch (IOException e) {
            throw new KafkaException("I/O exception when writing to the append stream, closing", e);
        }
    }

    public void appendWithOffset(long offset, Record record) {
        if (record.magic() != this.magic) {
            throw new IllegalArgumentException("Inner log entries must have matching magic values as the wrapper");
        }
        if (this.lastOffset >= 0L && offset <= this.lastOffset) {
            throw new IllegalArgumentException(String.format("Illegal offset %s following previous offset %s (Offsets must increase monotonically).", offset, this.lastOffset));
        }
        this.appendUnchecked(offset, record);
    }

    public void append(Record record) {
        this.appendWithOffset(this.lastOffset < 0L ? this.baseOffset : this.lastOffset + 1L, record);
    }

    private long toInnerOffset(long offset) {
        if (this.magic > 0 && this.compressionType != CompressionType.NONE) {
            return offset - this.baseOffset;
        }
        return offset;
    }

    private void recordWritten(long offset, long timestamp, int size) {
        ++this.numRecords;
        this.writtenUncompressed += (long)size;
        this.lastOffset = offset;
        if (timestamp > this.maxTimestamp) {
            this.maxTimestamp = timestamp;
            this.offsetOfMaxTimestamp = offset;
        }
    }

    private int estimatedBytesWritten() {
        if (this.compressionType == CompressionType.NONE) {
            return this.buffer().position();
        }
        return (int)((float)this.writtenUncompressed * TYPE_TO_RATE[this.compressionType.id] * 1.05f);
    }

    public boolean hasRoomFor(byte[] key, byte[] value) {
        return !this.isFull() && (this.numRecords == 0L ? this.initialCapacity >= 12 + Record.recordSize(this.magic, key, value) : this.writeLimit >= this.estimatedBytesWritten() + 12 + Record.recordSize(this.magic, key, value));
    }

    public boolean isClosed() {
        return this.builtRecords != null;
    }

    public boolean isFull() {
        return this.isClosed() || this.numRecords > 0L && this.writeLimit <= this.estimatedBytesWritten();
    }

    public int sizeInBytes() {
        return this.builtRecords != null ? this.builtRecords.sizeInBytes() : this.estimatedBytesWritten();
    }

    private static DataOutputStream wrapForOutput(ByteBufferOutputStream buffer, CompressionType type, byte messageVersion, int bufferSize) {
        try {
            switch (type) {
                case NONE: {
                    return buffer;
                }
                case GZIP: {
                    return new DataOutputStream(new GZIPOutputStream((OutputStream)buffer, bufferSize));
                }
                case SNAPPY: {
                    try {
                        OutputStream stream = (OutputStream)snappyOutputStreamSupplier.get().newInstance(buffer, bufferSize);
                        return new DataOutputStream(stream);
                    }
                    catch (Exception e) {
                        throw new KafkaException(e);
                    }
                }
                case LZ4: {
                    try {
                        OutputStream stream = (OutputStream)lz4OutputStreamSupplier.get().newInstance(buffer, messageVersion == 0);
                        return new DataOutputStream(stream);
                    }
                    catch (Exception e) {
                        throw new KafkaException(e);
                    }
                }
            }
            throw new IllegalArgumentException("Unknown compression type: " + (Object)((Object)type));
        }
        catch (IOException e) {
            throw new KafkaException(e);
        }
    }

    public static DataInputStream wrapForInput(ByteBufferInputStream buffer, CompressionType type, byte messageVersion) {
        try {
            switch (type) {
                case NONE: {
                    return buffer;
                }
                case GZIP: {
                    return new DataInputStream(new GZIPInputStream(buffer));
                }
                case SNAPPY: {
                    try {
                        InputStream stream = (InputStream)snappyInputStreamSupplier.get().newInstance(buffer);
                        return new DataInputStream(stream);
                    }
                    catch (Exception e) {
                        throw new KafkaException(e);
                    }
                }
                case LZ4: {
                    try {
                        InputStream stream = (InputStream)lz4InputStreamSupplier.get().newInstance(buffer, messageVersion == 0);
                        return new DataInputStream(stream);
                    }
                    catch (Exception e) {
                        throw new KafkaException(e);
                    }
                }
            }
            throw new IllegalArgumentException("Unknown compression type: " + (Object)((Object)type));
        }
        catch (IOException e) {
            throw new KafkaException(e);
        }
    }

    static {
        int maxTypeId = -1;
        for (CompressionType type : CompressionType.values()) {
            maxTypeId = Math.max(maxTypeId, type.id);
        }
        TYPE_TO_RATE = new float[maxTypeId + 1];
        for (CompressionType type : CompressionType.values()) {
            MemoryRecordsBuilder.TYPE_TO_RATE[type.id] = type.rate;
        }
        snappyOutputStreamSupplier = new MemoizingConstructorSupplier(new ConstructorSupplier(){

            @Override
            public Constructor get() throws ClassNotFoundException, NoSuchMethodException {
                return Class.forName("org.xerial.snappy.SnappyOutputStream").getConstructor(OutputStream.class, Integer.TYPE);
            }
        });
        lz4OutputStreamSupplier = new MemoizingConstructorSupplier(new ConstructorSupplier(){

            @Override
            public Constructor get() throws ClassNotFoundException, NoSuchMethodException {
                return Class.forName("org.apache.kafka.common.record.KafkaLZ4BlockOutputStream").getConstructor(OutputStream.class, Boolean.TYPE);
            }
        });
        snappyInputStreamSupplier = new MemoizingConstructorSupplier(new ConstructorSupplier(){

            @Override
            public Constructor get() throws ClassNotFoundException, NoSuchMethodException {
                return Class.forName("org.xerial.snappy.SnappyInputStream").getConstructor(InputStream.class);
            }
        });
        lz4InputStreamSupplier = new MemoizingConstructorSupplier(new ConstructorSupplier(){

            @Override
            public Constructor get() throws ClassNotFoundException, NoSuchMethodException {
                return Class.forName("org.apache.kafka.common.record.KafkaLZ4BlockInputStream").getConstructor(InputStream.class, Boolean.TYPE);
            }
        });
    }

    public static class RecordsInfo {
        public final long maxTimestamp;
        public final long shallowOffsetOfMaxTimestamp;

        public RecordsInfo(long maxTimestamp, long shallowOffsetOfMaxTimestamp) {
            this.maxTimestamp = maxTimestamp;
            this.shallowOffsetOfMaxTimestamp = shallowOffsetOfMaxTimestamp;
        }
    }

    private static class MemoizingConstructorSupplier {
        final ConstructorSupplier delegate;
        volatile transient boolean initialized;
        transient Constructor value;

        public MemoizingConstructorSupplier(ConstructorSupplier delegate) {
            this.delegate = delegate;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Constructor get() throws NoSuchMethodException, ClassNotFoundException {
            if (!this.initialized) {
                MemoizingConstructorSupplier memoizingConstructorSupplier = this;
                synchronized (memoizingConstructorSupplier) {
                    if (!this.initialized) {
                        this.value = this.delegate.get();
                        this.initialized = true;
                    }
                }
            }
            return this.value;
        }
    }

    private static interface ConstructorSupplier {
        public Constructor get() throws ClassNotFoundException, NoSuchMethodException;
    }
}

