/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.catalog;

import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import org.apache.flink.annotation.Internal;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.ExecutionConfig;
import org.apache.flink.configuration.ReadableConfig;
import org.apache.flink.table.api.CatalogNotExistException;
import org.apache.flink.table.api.TableException;
import org.apache.flink.table.api.ValidationException;
import org.apache.flink.table.catalog.Catalog;
import org.apache.flink.table.catalog.CatalogBaseTable;
import org.apache.flink.table.catalog.CatalogPartition;
import org.apache.flink.table.catalog.CatalogPartitionSpec;
import org.apache.flink.table.catalog.CatalogTable;
import org.apache.flink.table.catalog.CatalogView;
import org.apache.flink.table.catalog.DataTypeFactory;
import org.apache.flink.table.catalog.DataTypeFactoryImpl;
import org.apache.flink.table.catalog.DefaultSchemaResolver;
import org.apache.flink.table.catalog.ObjectIdentifier;
import org.apache.flink.table.catalog.ObjectPath;
import org.apache.flink.table.catalog.ResolvedCatalogBaseTable;
import org.apache.flink.table.catalog.ResolvedCatalogTable;
import org.apache.flink.table.catalog.ResolvedCatalogView;
import org.apache.flink.table.catalog.ResolvedSchema;
import org.apache.flink.table.catalog.SchemaResolver;
import org.apache.flink.table.catalog.TemporaryOperationListener;
import org.apache.flink.table.catalog.UnresolvedIdentifier;
import org.apache.flink.table.catalog.exceptions.CatalogException;
import org.apache.flink.table.catalog.exceptions.DatabaseNotExistException;
import org.apache.flink.table.catalog.exceptions.PartitionNotExistException;
import org.apache.flink.table.catalog.exceptions.TableAlreadyExistException;
import org.apache.flink.table.catalog.exceptions.TableNotExistException;
import org.apache.flink.table.expressions.resolver.ExpressionResolver;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public final class CatalogManager {
    private static final Logger LOG = LoggerFactory.getLogger(CatalogManager.class);
    private final Map<String, Catalog> catalogs;
    private final Map<ObjectIdentifier, CatalogBaseTable> temporaryTables;
    private String currentCatalogName;
    private String currentDatabaseName;
    private DefaultSchemaResolver schemaResolver;
    private final String builtInCatalogName;
    private final DataTypeFactory typeFactory;

    private CatalogManager(String defaultCatalogName, Catalog defaultCatalog, DataTypeFactory typeFactory) {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)defaultCatalogName) ? 1 : 0) != 0, (Object)"Default catalog name cannot be null or empty");
        Preconditions.checkNotNull((Object)defaultCatalog, (String)"Default catalog cannot be null");
        this.catalogs = new LinkedHashMap<String, Catalog>();
        this.catalogs.put(defaultCatalogName, defaultCatalog);
        this.currentCatalogName = defaultCatalogName;
        this.currentDatabaseName = defaultCatalog.getDefaultDatabase();
        this.temporaryTables = new HashMap<ObjectIdentifier, CatalogBaseTable>();
        this.builtInCatalogName = defaultCatalogName;
        this.typeFactory = typeFactory;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public void initSchemaResolver(boolean isStreamingMode, ExpressionResolver.ExpressionResolverBuilder expressionResolverBuilder) {
        this.schemaResolver = new DefaultSchemaResolver(isStreamingMode, this.typeFactory, expressionResolverBuilder);
    }

    public SchemaResolver getSchemaResolver() {
        return this.schemaResolver;
    }

    public DataTypeFactory getDataTypeFactory() {
        return this.typeFactory;
    }

    public void registerCatalog(String catalogName, Catalog catalog) {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)catalogName) ? 1 : 0) != 0, (Object)"Catalog name cannot be null or empty.");
        Preconditions.checkNotNull((Object)catalog, (String)"Catalog cannot be null");
        if (this.catalogs.containsKey(catalogName)) {
            throw new CatalogException(String.format("Catalog %s already exists.", catalogName));
        }
        catalog.open();
        this.catalogs.put(catalogName, catalog);
    }

    public void unregisterCatalog(String catalogName, boolean ignoreIfNotExists) {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)catalogName) ? 1 : 0) != 0, (Object)"Catalog name cannot be null or empty.");
        if (this.catalogs.containsKey(catalogName)) {
            Catalog catalog = this.catalogs.remove(catalogName);
            catalog.close();
        } else if (!ignoreIfNotExists) {
            throw new CatalogException(String.format("Catalog %s does not exist.", catalogName));
        }
    }

    public Optional<Catalog> getCatalog(String catalogName) {
        return Optional.ofNullable(this.catalogs.get(catalogName));
    }

    public String getCurrentCatalog() {
        return this.currentCatalogName;
    }

    public void setCurrentCatalog(String catalogName) throws CatalogNotExistException {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)catalogName) ? 1 : 0) != 0, (Object)"Catalog name cannot be null or empty.");
        Catalog potentialCurrentCatalog = this.catalogs.get(catalogName);
        if (potentialCurrentCatalog == null) {
            throw new CatalogException(String.format("A catalog with name [%s] does not exist.", catalogName));
        }
        if (!this.currentCatalogName.equals(catalogName)) {
            this.currentCatalogName = catalogName;
            this.currentDatabaseName = potentialCurrentCatalog.getDefaultDatabase();
            LOG.info("Set the current default catalog as [{}] and the current default database as [{}].", (Object)this.currentCatalogName, (Object)this.currentDatabaseName);
        }
    }

    public String getCurrentDatabase() {
        return this.currentDatabaseName;
    }

    public void setCurrentDatabase(String databaseName) {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)databaseName) ? 1 : 0) != 0, (Object)"The database name cannot be null or empty.");
        if (!this.catalogs.get(this.currentCatalogName).databaseExists(databaseName)) {
            throw new CatalogException(String.format("A database with name [%s] does not exist in the catalog: [%s].", databaseName, this.currentCatalogName));
        }
        if (!this.currentDatabaseName.equals(databaseName)) {
            this.currentDatabaseName = databaseName;
            LOG.info("Set the current default database as [{}] in the current default catalog [{}].", (Object)this.currentDatabaseName, (Object)this.currentCatalogName);
        }
    }

    public String getBuiltInCatalogName() {
        return this.builtInCatalogName;
    }

    public String getBuiltInDatabaseName() {
        return this.catalogs.get(this.getBuiltInCatalogName()).getDefaultDatabase();
    }

    public Optional<TableLookupResult> getTable(ObjectIdentifier objectIdentifier) {
        CatalogBaseTable temporaryTable = this.temporaryTables.get(objectIdentifier);
        if (temporaryTable != null) {
            ResolvedCatalogBaseTable<?> resolvedTable = this.resolveCatalogBaseTable(temporaryTable);
            return Optional.of(TableLookupResult.temporary(resolvedTable));
        }
        return this.getPermanentTable(objectIdentifier);
    }

    public Optional<CatalogPartition> getPartition(ObjectIdentifier tableIdentifier, CatalogPartitionSpec partitionSpec) {
        Catalog catalog = this.catalogs.get(tableIdentifier.getCatalogName());
        if (catalog != null) {
            try {
                return Optional.of(catalog.getPartition(tableIdentifier.toObjectPath(), partitionSpec));
            }
            catch (PartitionNotExistException partitionNotExistException) {
                // empty catch block
            }
        }
        return Optional.empty();
    }

    private Optional<TableLookupResult> getPermanentTable(ObjectIdentifier objectIdentifier) {
        Catalog currentCatalog = this.catalogs.get(objectIdentifier.getCatalogName());
        ObjectPath objectPath = objectIdentifier.toObjectPath();
        if (currentCatalog != null) {
            try {
                CatalogBaseTable table = currentCatalog.getTable(objectPath);
                ResolvedCatalogBaseTable<?> resolvedTable = this.resolveCatalogBaseTable(table);
                return Optional.of(TableLookupResult.permanent(resolvedTable));
            }
            catch (TableNotExistException tableNotExistException) {
                // empty catch block
            }
        }
        return Optional.empty();
    }

    private Optional<CatalogBaseTable> getUnresolvedTable(ObjectIdentifier objectIdentifier) {
        Catalog currentCatalog = this.catalogs.get(objectIdentifier.getCatalogName());
        ObjectPath objectPath = objectIdentifier.toObjectPath();
        if (currentCatalog != null) {
            try {
                CatalogBaseTable table = currentCatalog.getTable(objectPath);
                return Optional.of(table);
            }
            catch (TableNotExistException tableNotExistException) {
                // empty catch block
            }
        }
        return Optional.empty();
    }

    public Set<String> listCatalogs() {
        return Collections.unmodifiableSet(this.catalogs.keySet());
    }

    public Set<String> listTables() {
        return this.listTables(this.getCurrentCatalog(), this.getCurrentDatabase());
    }

    public Set<String> listTables(String catalogName, String databaseName) {
        Catalog currentCatalog = this.catalogs.get(this.getCurrentCatalog());
        try {
            return Stream.concat(currentCatalog.listTables(this.getCurrentDatabase()).stream(), this.listTemporaryTablesInternal(catalogName, databaseName).map(e -> ((ObjectIdentifier)e.getKey()).getObjectName())).collect(Collectors.toSet());
        }
        catch (DatabaseNotExistException e2) {
            throw new ValidationException("Current database does not exist", (Throwable)e2);
        }
    }

    public Set<String> listTemporaryTables() {
        return this.listTemporaryTablesInternal(this.getCurrentCatalog(), this.getCurrentDatabase()).map(e -> ((ObjectIdentifier)e.getKey()).getObjectName()).collect(Collectors.toSet());
    }

    public Set<String> listTemporaryViews() {
        return this.listTemporaryViewsInternal(this.getCurrentCatalog(), this.getCurrentDatabase()).map(e -> ((ObjectIdentifier)e.getKey()).getObjectName()).collect(Collectors.toSet());
    }

    private Stream<Map.Entry<ObjectIdentifier, CatalogBaseTable>> listTemporaryTablesInternal(String catalogName, String databaseName) {
        return this.temporaryTables.entrySet().stream().filter(e -> {
            ObjectIdentifier identifier = (ObjectIdentifier)e.getKey();
            return identifier.getCatalogName().equals(catalogName) && identifier.getDatabaseName().equals(databaseName);
        });
    }

    public Set<String> listViews() {
        return this.listViews(this.getCurrentCatalog(), this.getCurrentDatabase());
    }

    public Set<String> listViews(String catalogName, String databaseName) {
        Catalog currentCatalog = this.catalogs.get(this.getCurrentCatalog());
        try {
            return Stream.concat(currentCatalog.listViews(this.getCurrentDatabase()).stream(), this.listTemporaryViewsInternal(catalogName, databaseName).map(e -> ((ObjectIdentifier)e.getKey()).getObjectName())).collect(Collectors.toSet());
        }
        catch (DatabaseNotExistException e2) {
            throw new ValidationException("Current database does not exist", (Throwable)e2);
        }
    }

    private Stream<Map.Entry<ObjectIdentifier, CatalogBaseTable>> listTemporaryViewsInternal(String catalogName, String databaseName) {
        return this.listTemporaryTablesInternal(catalogName, databaseName).filter(e -> e.getValue() instanceof CatalogView);
    }

    public Set<String> listSchemas() {
        return Stream.concat(this.catalogs.keySet().stream(), this.temporaryTables.keySet().stream().map(ObjectIdentifier::getCatalogName)).collect(Collectors.toSet());
    }

    public Set<String> listSchemas(String catalogName) {
        return Stream.concat(Optional.ofNullable(this.catalogs.get(catalogName)).map(Catalog::listDatabases).orElse(Collections.emptyList()).stream(), this.temporaryTables.keySet().stream().filter(i -> i.getCatalogName().equals(catalogName)).map(ObjectIdentifier::getDatabaseName)).collect(Collectors.toSet());
    }

    public boolean schemaExists(String catalogName) {
        return this.getCatalog(catalogName).isPresent() || this.temporaryTables.keySet().stream().anyMatch(i -> i.getCatalogName().equals(catalogName));
    }

    public boolean schemaExists(String catalogName, String databaseName) {
        return this.temporaryDatabaseExists(catalogName, databaseName) || this.permanentDatabaseExists(catalogName, databaseName);
    }

    private boolean temporaryDatabaseExists(String catalogName, String databaseName) {
        return this.temporaryTables.keySet().stream().anyMatch(i -> i.getCatalogName().equals(catalogName) && i.getDatabaseName().equals(databaseName));
    }

    private boolean permanentDatabaseExists(String catalogName, String databaseName) {
        return this.getCatalog(catalogName).map(c -> c.databaseExists(databaseName)).orElse(false);
    }

    public ObjectIdentifier qualifyIdentifier(UnresolvedIdentifier identifier) {
        return ObjectIdentifier.of((String)identifier.getCatalogName().orElseGet(this::getCurrentCatalog), (String)identifier.getDatabaseName().orElseGet(this::getCurrentDatabase), (String)identifier.getObjectName());
    }

    public void createTable(CatalogBaseTable table, ObjectIdentifier objectIdentifier, boolean ignoreIfExists) {
        this.execute((catalog, path) -> catalog.createTable(path, this.resolveCatalogBaseTable(table), ignoreIfExists), objectIdentifier, false, "CreateTable");
    }

    public void createTemporaryTable(CatalogBaseTable table, ObjectIdentifier objectIdentifier, boolean ignoreIfExists) {
        Optional<TemporaryOperationListener> listener = this.getTemporaryOperationListener(objectIdentifier);
        this.temporaryTables.compute(objectIdentifier, (k, v) -> {
            if (v != null) {
                if (!ignoreIfExists) {
                    throw new ValidationException(String.format("Temporary table '%s' already exists", objectIdentifier));
                }
                return v;
            }
            ResolvedCatalogBaseTable<?> resolvedTable = this.resolveCatalogBaseTable(table);
            if (listener.isPresent()) {
                return ((TemporaryOperationListener)listener.get()).onCreateTemporaryTable(objectIdentifier.toObjectPath(), resolvedTable);
            }
            return resolvedTable;
        });
    }

    public void dropTemporaryTable(ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        this.dropTemporaryTableInternal(objectIdentifier, table -> table instanceof CatalogTable, ignoreIfNotExists);
    }

    public void dropTemporaryView(ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        this.dropTemporaryTableInternal(objectIdentifier, table -> table instanceof CatalogView, ignoreIfNotExists);
    }

    private void dropTemporaryTableInternal(ObjectIdentifier objectIdentifier, Predicate<CatalogBaseTable> filter, boolean ignoreIfNotExists) {
        CatalogBaseTable catalogBaseTable = this.temporaryTables.get(objectIdentifier);
        if (filter.test(catalogBaseTable)) {
            this.getTemporaryOperationListener(objectIdentifier).ifPresent(l -> l.onDropTemporaryTable(objectIdentifier.toObjectPath()));
            this.temporaryTables.remove(objectIdentifier);
        } else if (!ignoreIfNotExists) {
            throw new ValidationException(String.format("Temporary table or view with identifier '%s' does not exist.", objectIdentifier.asSummaryString()));
        }
    }

    protected Optional<TemporaryOperationListener> getTemporaryOperationListener(ObjectIdentifier identifier) {
        return this.getCatalog(identifier.getCatalogName()).map(c -> c instanceof TemporaryOperationListener ? (TemporaryOperationListener)c : null);
    }

    public void alterTable(CatalogBaseTable table, ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        this.execute((catalog, path) -> {
            ResolvedCatalogBaseTable<?> resolvedTable = this.resolveCatalogBaseTable(table);
            catalog.alterTable(path, resolvedTable, ignoreIfNotExists);
        }, objectIdentifier, ignoreIfNotExists, "AlterTable");
    }

    public void dropTable(ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        this.dropTableInternal(objectIdentifier, ignoreIfNotExists, true);
    }

    public void dropView(ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        this.dropTableInternal(objectIdentifier, ignoreIfNotExists, false);
    }

    private void dropTableInternal(ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists, boolean isDropTable) {
        Predicate<CatalogBaseTable> filter;
        Predicate<CatalogBaseTable> predicate = filter = isDropTable ? table -> table instanceof CatalogTable : table -> table instanceof CatalogView;
        if (filter.test(this.temporaryTables.get(objectIdentifier))) {
            String tableOrView = isDropTable ? "table" : "view";
            throw new ValidationException(String.format("Temporary %s with identifier '%s' exists. Drop it first before removing the permanent %s.", tableOrView, objectIdentifier, tableOrView));
        }
        Optional<CatalogBaseTable> resultOpt = this.getUnresolvedTable(objectIdentifier);
        if (resultOpt.isPresent() && filter.test(resultOpt.get())) {
            this.execute((catalog, path) -> catalog.dropTable(path, ignoreIfNotExists), objectIdentifier, ignoreIfNotExists, "DropTable");
        } else if (!ignoreIfNotExists) {
            String tableOrView = isDropTable ? "Table" : "View";
            throw new ValidationException(String.format("%s with identifier '%s' does not exist.", tableOrView, objectIdentifier.asSummaryString()));
        }
    }

    private void execute(ModifyCatalog command, ObjectIdentifier objectIdentifier, boolean ignoreNoCatalog, String commandName) {
        Optional<Catalog> catalog = this.getCatalog(objectIdentifier.getCatalogName());
        if (catalog.isPresent()) {
            try {
                command.execute(catalog.get(), objectIdentifier.toObjectPath());
            }
            catch (DatabaseNotExistException | TableAlreadyExistException | TableNotExistException e) {
                throw new ValidationException(this.getErrorMessage(objectIdentifier, commandName), e);
            }
            catch (Exception e) {
                throw new TableException(this.getErrorMessage(objectIdentifier, commandName), (Throwable)e);
            }
        } else if (!ignoreNoCatalog) {
            throw new ValidationException(String.format("Catalog %s does not exist.", objectIdentifier.getCatalogName()));
        }
    }

    private String getErrorMessage(ObjectIdentifier objectIdentifier, String commandName) {
        return String.format("Could not execute %s in path %s", commandName, objectIdentifier);
    }

    public ResolvedCatalogBaseTable<?> resolveCatalogBaseTable(CatalogBaseTable baseTable) {
        Preconditions.checkState((this.schemaResolver != null ? 1 : 0) != 0, (Object)"Schema resolver is not initialized.");
        if (baseTable instanceof CatalogTable) {
            return this.resolveCatalogTable((CatalogTable)baseTable);
        }
        if (baseTable instanceof CatalogView) {
            return this.resolveCatalogView((CatalogView)baseTable);
        }
        throw new IllegalArgumentException("Unknown kind of catalog base table: " + baseTable.getClass());
    }

    public ResolvedCatalogTable resolveCatalogTable(CatalogTable table) {
        Preconditions.checkState((this.schemaResolver != null ? 1 : 0) != 0, (Object)"Schema resolver is not initialized.");
        if (table instanceof ResolvedCatalogTable) {
            return (ResolvedCatalogTable)table;
        }
        ResolvedSchema resolvedSchema = table.getUnresolvedSchema().resolve((SchemaResolver)this.schemaResolver);
        return new ResolvedCatalogTable(table, resolvedSchema);
    }

    public ResolvedCatalogView resolveCatalogView(CatalogView view) {
        Preconditions.checkState((this.schemaResolver != null ? 1 : 0) != 0, (Object)"Schema resolver is not initialized.");
        if (view instanceof ResolvedCatalogView) {
            return (ResolvedCatalogView)view;
        }
        ResolvedSchema resolvedSchema = view.getUnresolvedSchema().resolve((SchemaResolver)this.schemaResolver);
        return new ResolvedCatalogView(view, resolvedSchema);
    }

    private static interface ModifyCatalog {
        public void execute(Catalog var1, ObjectPath var2) throws Exception;
    }

    public static class TableLookupResult {
        private final boolean isTemporary;
        private final ResolvedCatalogBaseTable<?> resolvedTable;

        @VisibleForTesting
        public static TableLookupResult temporary(ResolvedCatalogBaseTable<?> resolvedTable) {
            return new TableLookupResult(true, resolvedTable);
        }

        @VisibleForTesting
        public static TableLookupResult permanent(ResolvedCatalogBaseTable<?> resolvedTable) {
            return new TableLookupResult(false, resolvedTable);
        }

        private TableLookupResult(boolean isTemporary, ResolvedCatalogBaseTable<?> resolvedTable) {
            this.isTemporary = isTemporary;
            this.resolvedTable = resolvedTable;
        }

        public boolean isTemporary() {
            return this.isTemporary;
        }

        public ResolvedCatalogBaseTable<?> getResolvedTable() {
            return this.resolvedTable;
        }

        public ResolvedSchema getResolvedSchema() {
            return this.resolvedTable.getResolvedSchema();
        }

        public CatalogBaseTable getTable() {
            return this.resolvedTable.getOrigin();
        }
    }

    public static final class Builder {
        @Nullable
        private ClassLoader classLoader;
        @Nullable
        private ReadableConfig config;
        @Nullable
        private String defaultCatalogName;
        @Nullable
        private Catalog defaultCatalog;
        @Nullable
        private ExecutionConfig executionConfig;

        public Builder classLoader(ClassLoader classLoader) {
            this.classLoader = classLoader;
            return this;
        }

        public Builder config(ReadableConfig config) {
            this.config = config;
            return this;
        }

        public Builder defaultCatalog(String defaultCatalogName, Catalog defaultCatalog) {
            this.defaultCatalogName = defaultCatalogName;
            this.defaultCatalog = defaultCatalog;
            return this;
        }

        public Builder executionConfig(ExecutionConfig executionConfig) {
            this.executionConfig = executionConfig;
            return this;
        }

        public CatalogManager build() {
            Preconditions.checkNotNull((Object)this.classLoader, (String)"Class loader cannot be null");
            Preconditions.checkNotNull((Object)this.config, (String)"Config cannot be null");
            return new CatalogManager(this.defaultCatalogName, this.defaultCatalog, new DataTypeFactoryImpl(this.classLoader, this.config, this.executionConfig));
        }
    }
}

