/*
 * Decompiled with CFR 0.152.
 */
package io.jhdf.dataset.chunked.indexing;

import io.jhdf.Utils;
import io.jhdf.checksum.ChecksumUtils;
import io.jhdf.dataset.chunked.Chunk;
import io.jhdf.dataset.chunked.DatasetInfo;
import io.jhdf.dataset.chunked.indexing.ChunkImpl;
import io.jhdf.dataset.chunked.indexing.ChunkIndex;
import io.jhdf.exceptions.HdfException;
import io.jhdf.storage.HdfBackingStorage;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collection;
import java.util.List;
import org.apache.commons.lang3.concurrent.ConcurrentException;
import org.apache.commons.lang3.concurrent.LazyInitializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FixedArrayIndex
implements ChunkIndex {
    private static final byte[] FIXED_ARRAY_HEADER_SIGNATURE = "FAHD".getBytes(StandardCharsets.US_ASCII);
    private static final byte[] FIXED_ARRAY_DATA_BLOCK_SIGNATURE = "FADB".getBytes(StandardCharsets.US_ASCII);
    private static final Logger logger = LoggerFactory.getLogger(FixedArrayIndex.class);
    private final long address;
    private final int unfilteredChunkSize;
    private final int[] datasetDimensions;
    private final int[] chunkDimensions;
    private final int clientId;
    private final int entrySize;
    private final int pageBits;
    private final int maxNumberOfEntries;
    private final long dataBlockAddress;
    private final int pages;
    private final int pageSize;
    private final FixedArrayDataBlockInitializer dataBlockInitializer;

    public FixedArrayIndex(HdfBackingStorage hdfBackingStorage, long address, DatasetInfo datasetInfo) {
        this.address = address;
        this.unfilteredChunkSize = datasetInfo.getChunkSizeInBytes();
        this.datasetDimensions = datasetInfo.getDatasetDimensions();
        this.chunkDimensions = datasetInfo.getChunkDimensions();
        int headerSize = 12 + hdfBackingStorage.getSizeOfOffsets() + hdfBackingStorage.getSizeOfLengths();
        ByteBuffer bb = hdfBackingStorage.readBufferFromAddress(address, headerSize);
        bb.mark();
        byte[] formatSignatureBytes = new byte[4];
        bb.get(formatSignatureBytes, 0, formatSignatureBytes.length);
        if (!Arrays.equals(FIXED_ARRAY_HEADER_SIGNATURE, formatSignatureBytes)) {
            throw new HdfException("Fixed array header signature 'FAHD' not matched, at address " + address);
        }
        byte version = bb.get();
        if (version != 0) {
            throw new HdfException("Unsupported fixed array index version detected. Version: " + version);
        }
        this.clientId = bb.get();
        this.entrySize = bb.get();
        this.pageBits = bb.get();
        this.maxNumberOfEntries = Utils.readBytesAsUnsignedInt(bb, hdfBackingStorage.getSizeOfLengths());
        this.pageSize = 1 << this.pageBits;
        this.pages = (this.maxNumberOfEntries + this.pageSize - 1) / this.pageSize;
        this.dataBlockAddress = Utils.readBytesAsUnsignedLong(bb, hdfBackingStorage.getSizeOfOffsets());
        ChecksumUtils.validateChecksumFromMark(bb);
        this.dataBlockInitializer = new FixedArrayDataBlockInitializer(hdfBackingStorage);
        logger.info("Read fixed array index header. pages=[{}], maxEntries=[{}]", (Object)this.pages, (Object)this.maxNumberOfEntries);
    }

    @Override
    public Collection<Chunk> getAllChunks() {
        try {
            return ((FixedArrayDataBlock)this.dataBlockInitializer.get()).chunks;
        }
        catch (ConcurrentException e) {
            throw new HdfException("Error initializing data block", e);
        }
    }

    private class FixedArrayDataBlock {
        private final List<Chunk> chunks;

        private FixedArrayDataBlock(HdfBackingStorage hdfBackingStorage, long address) {
            this.chunks = new ArrayList<Chunk>(FixedArrayIndex.this.maxNumberOfEntries);
            int pageBitmapBytes = (FixedArrayIndex.this.pages + 7) / 8;
            int headerSize = 6 + hdfBackingStorage.getSizeOfOffsets() + FixedArrayIndex.this.entrySize * FixedArrayIndex.this.maxNumberOfEntries + 4;
            if (FixedArrayIndex.this.pages > 1) {
                headerSize += pageBitmapBytes + 4 * FixedArrayIndex.this.pages;
            }
            ByteBuffer bb = hdfBackingStorage.readBufferFromAddress(address, headerSize);
            bb.mark();
            byte[] formatSignatureBytes = new byte[4];
            bb.get(formatSignatureBytes, 0, formatSignatureBytes.length);
            if (!Arrays.equals(FIXED_ARRAY_DATA_BLOCK_SIGNATURE, formatSignatureBytes)) {
                throw new HdfException("Fixed array data block signature 'FADB' not matched, at address " + address);
            }
            byte version = bb.get();
            if (version != 0) {
                throw new HdfException("Unsupported fixed array data block version detected. Version: " + version);
            }
            byte dataBlockclientId = bb.get();
            if (dataBlockclientId != FixedArrayIndex.this.clientId) {
                throw new HdfException("Fixed array client ID mismatch. Possible file corruption detected");
            }
            long headerAddress = Utils.readBytesAsUnsignedLong(bb, hdfBackingStorage.getSizeOfOffsets());
            if (headerAddress != FixedArrayIndex.this.address) {
                throw new HdfException("Fixed array data block header address missmatch");
            }
            if (FixedArrayIndex.this.pages > 1) {
                this.readPaged(hdfBackingStorage, pageBitmapBytes, bb, dataBlockclientId);
            } else {
                logger.info("Reading unpaged");
                if (dataBlockclientId == 0) {
                    for (int i = 0; i < FixedArrayIndex.this.maxNumberOfEntries; ++i) {
                        this.readUnfiltered(hdfBackingStorage.getSizeOfOffsets(), bb, i);
                    }
                } else if (dataBlockclientId == 1) {
                    for (int i = 0; i < FixedArrayIndex.this.maxNumberOfEntries; ++i) {
                        this.readFiltered(hdfBackingStorage, bb, i);
                    }
                } else {
                    throw new HdfException("Unrecognized client ID  = " + dataBlockclientId);
                }
                ChecksumUtils.validateChecksumFromMark(bb);
            }
        }

        private void readPaged(HdfBackingStorage hdfBackingStorage, int pageBitmapBytes, ByteBuffer bb, int dataBlockclientId) {
            logger.info("Reading paged");
            byte[] pageBitmap = new byte[pageBitmapBytes];
            bb.get(pageBitmap);
            ChecksumUtils.validateChecksumFromMark(bb);
            int chunkIndex = 0;
            for (int page = 0; page < FixedArrayIndex.this.pages; ++page) {
                int i;
                int currentPageSize = this.getCurrentPageSize(page);
                if (dataBlockclientId == 0) {
                    for (i = 0; i < currentPageSize; ++i) {
                        this.readUnfiltered(hdfBackingStorage.getSizeOfOffsets(), bb, chunkIndex++);
                    }
                } else if (dataBlockclientId == 1) {
                    for (i = 0; i < currentPageSize; ++i) {
                        this.readFiltered(hdfBackingStorage, bb, chunkIndex++);
                    }
                } else {
                    throw new HdfException("Unrecognized client ID  = " + dataBlockclientId);
                }
                ChecksumUtils.validateChecksumFromMark(bb);
            }
        }

        private int getCurrentPageSize(int page) {
            int lastPageSize;
            int currentPageSize = page == FixedArrayIndex.this.pages - 1 ? ((lastPageSize = FixedArrayIndex.this.maxNumberOfEntries % FixedArrayIndex.this.pageSize) == 0 ? FixedArrayIndex.this.pageSize : lastPageSize) : FixedArrayIndex.this.pageSize;
            return currentPageSize;
        }

        private void readFiltered(HdfBackingStorage hdfBackingStorage, ByteBuffer bb, int i) {
            long chunkAddress = Utils.readBytesAsUnsignedLong(bb, hdfBackingStorage.getSizeOfOffsets());
            int chunkSizeInBytes = Utils.readBytesAsUnsignedInt(bb, FixedArrayIndex.this.entrySize - hdfBackingStorage.getSizeOfOffsets() - 4);
            BitSet filterMask = BitSet.valueOf(new byte[]{bb.get(), bb.get(), bb.get(), bb.get()});
            int[] chunkOffset = Utils.chunkIndexToChunkOffset(i, FixedArrayIndex.this.chunkDimensions, FixedArrayIndex.this.datasetDimensions);
            this.chunks.add(new ChunkImpl(chunkAddress, chunkSizeInBytes, chunkOffset, filterMask));
        }

        private void readUnfiltered(int sizeOfOffsets, ByteBuffer bb, int chunkIndex) {
            long chunkAddress = Utils.readBytesAsUnsignedLong(bb, sizeOfOffsets);
            int[] chunkOffset = Utils.chunkIndexToChunkOffset(chunkIndex, FixedArrayIndex.this.chunkDimensions, FixedArrayIndex.this.datasetDimensions);
            this.chunks.add(new ChunkImpl(chunkAddress, FixedArrayIndex.this.unfilteredChunkSize, chunkOffset));
        }
    }

    private class FixedArrayDataBlockInitializer
    extends LazyInitializer<FixedArrayDataBlock> {
        private final HdfBackingStorage hdfBackingStorage;

        public FixedArrayDataBlockInitializer(HdfBackingStorage hdfBackingStorage) {
            this.hdfBackingStorage = hdfBackingStorage;
        }

        protected FixedArrayDataBlock initialize() {
            logger.info("Initializing data block");
            return new FixedArrayDataBlock(this.hdfBackingStorage, FixedArrayIndex.this.dataBlockAddress);
        }
    }
}

