/*
 * Copyright (c) 2012 Nicolas Morel
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */
package com.googlecode.gflot.client.options;


import com.google.gwt.core.client.JavaScriptObject;
import com.googlecode.gflot.client.Series;
import com.googlecode.gflot.client.jsni.JsonObject;

/**
 * Options for pie series
 *
 * @author Nicolas Morel
 */
public class PieSeriesOptions
    extends AbstractSeriesOptions<PieSeriesOptions>
{
    public static class Offset
        extends JsonObject
    {
        private static final String TOP_KEY = "top";
        private static final String LEFT_KEY = "left";

        /**
         * Creates a {@link Offset}
         */
        public static final Offset create()
        {
            return JavaScriptObject.createObject().cast();
        }

        /**
         * Creates a {@link Offset}
         */
        public static final Offset of( int top, int left )
        {
            Offset offset = Offset.create();
            offset.setTop( top );
            offset.setLeft( left );
            return offset;
        }

        protected Offset()
        {
        }

        /**
         * Set the top. integer value to move the pie up or down
         */
        public final Offset setTop( int top )
        {
            put( TOP_KEY, top );
            return this;
        }

        /**
         * @return the top
         */
        public final Integer getTop()
        {
            return getInteger( TOP_KEY );
        }

        /**
         * Clear the top option
         */
        public final Offset clearTop()
        {
            clear( TOP_KEY );
            return this;
        }

        /**
         * Set the left. integer value to move the pie left or right
         */
        public final Offset setLeft( int left )
        {
            put( LEFT_KEY, left );
            return this;
        }

        /**
         * @return the left
         */
        public final Integer getLeft()
        {
            return getInteger( LEFT_KEY );
        }

        /**
         * Clear the left option
         */
        public final Offset clearLeft()
        {
            clear( LEFT_KEY );
            return this;
        }
    }

    public static class Shadow
        extends JsonObject
    {
        private static final String TOP_KEY = "top";
        private static final String LEFT_KEY = "left";
        private static final String ALPHA_KEY = "alpha";

        /**
         * Creates a {@link Shadow}
         */
        public static final Shadow create()
        {
            return JavaScriptObject.createObject().cast();
        }

        /**
         * Creates a {@link Shadow}
         */
        public static final Shadow of( int top, int left )
        {
            Shadow shadow = Shadow.create();
            shadow.setTop( top );
            shadow.setLeft( left );
            return shadow;
        }

        /**
         * Creates a {@link Shadow}
         */
        public static final Shadow of( int top, int left, double alpha )
        {
            Shadow shadow = Shadow.of( top, left );
            shadow.setAlpha( alpha );
            return shadow;
        }

        protected Shadow()
        {
        }

        /**
         * Set the top. integer value to move the pie up or down
         */
        public final Shadow setTop( int top )
        {
            put( TOP_KEY, top );
            return this;
        }

        /**
         * @return the top
         */
        public final Integer getTop()
        {
            return getInteger( TOP_KEY );
        }

        /**
         * Clear the top option
         */
        public final Shadow clearTop()
        {
            clear( TOP_KEY );
            return this;
        }

        /**
         * Set the left. integer value to move the pie left or right
         */
        public final Shadow setLeft( int left )
        {
            put( LEFT_KEY, left );
            return this;
        }

        /**
         * @return the left
         */
        public final Integer getLeft()
        {
            return getInteger( LEFT_KEY );
        }

        /**
         * Clear the left option
         */
        public final Shadow clearLeft()
        {
            clear( LEFT_KEY );
            return this;
        }

        /**
         * Set the alpha.
         */
        public final Shadow setAlpha( double alpha )
        {
            put( ALPHA_KEY, alpha );
            return this;
        }

        /**
         * @return the alpha
         */
        public final Double getAlpha()
        {
            return getDouble( ALPHA_KEY );
        }

        /**
         * Clear the alpha option
         */
        public final Shadow clearAlpha()
        {
            clear( ALPHA_KEY );
            return this;
        }
    }

    public static class Stroke
        extends JsonObject
    {
        private static final String COLOR_KEY = "color";
        private static final String WIDTH_KEY = "width";

        /**
         * Creates a {@link Stroke}
         */
        public static final Stroke create()
        {
            return JavaScriptObject.createObject().cast();
        }

        protected Stroke()
        {
        }

        /**
         * Set the color. any hexidecimal color value (other formats may or may not work, so best to stick with
         * something like '#FFF')
         */
        public final Stroke setColor( String color )
        {
            put( COLOR_KEY, color );
            return this;
        }

        /**
         * @return the color
         */
        public final String getColor()
        {
            return getString( COLOR_KEY );
        }

        /**
         * Clear the color
         */
        public final Stroke clearColor()
        {
            clear( COLOR_KEY );
            return this;
        }

        /**
         * Set the integer pixel width of the stroke
         */
        public final Stroke setWidth( int width )
        {
            put( WIDTH_KEY, width );
            return this;
        }

        /**
         * @return the width
         */
        public final Integer getWidth()
        {
            return getInteger( WIDTH_KEY );
        }

        /**
         * Clear the width
         */
        public final Stroke clearWidth()
        {
            clear( WIDTH_KEY );
            return this;
        }
    }

    public static class Label
        extends JsonObject
    {
        public interface Formatter
        {
            String format( String label, Series series );
        }

        public static class Background
            extends JsonObject
        {
            private static final String COLOR_KEY = "color";
            private static final String OPACITY_KEY = "opacity";

            /**
             * Creates a {@link Background}
             */
            public static final Background create()
            {
                return JavaScriptObject.createObject().cast();
            }

            protected Background()
            {
            }

            /**
             * Set the color. any hexidecimal color value (other formats may or may not work, so best to stick with
             * something like '#000')
             */
            public final Background setColor( String color )
            {
                put( COLOR_KEY, color );
                return this;
            }

            /**
             * @return the color
             */
            public final String getColor()
            {
                return getString( COLOR_KEY );
            }

            /**
             * Clear the color
             */
            public final Background clearColor()
            {
                clear( COLOR_KEY );
                return this;
            }

            /**
             * Set the background opacity. Opacity range from 0.0 to 1.0.
             */
            public final Background setOpacity( double opacity )
            {
                assert opacity >= 0 && opacity <= 1 : "opacity range from 0.0 to 1.0";

                put( OPACITY_KEY, opacity );
                return this;
            }

            /**
             * @return the opacity
             */
            public final Double getOpacity()
            {
                return getDouble( OPACITY_KEY );
            }

            /**
             * Clear the opacity
             */
            public final Background clearOpacity()
            {
                clear( OPACITY_KEY );
                return this;
            }
        }

        private static final String SHOW_KEY = "show";
        private static final String RADIUS_KEY = "radius";
        private static final String BACKGROUND_KEY = "background";
        private static final String THRESHOLD_KEY = "threshold";

        /**
         * Creates a {@link Label}
         */
        public static final Label create()
        {
            return JavaScriptObject.createObject().cast();
        }

        protected Label()
        {
        }

        /**
         * Set the visibility of the label.
         */
        public final Label setShow( boolean show )
        {
            put( SHOW_KEY, show );
            return this;
        }

        /**
         * @return the visibility of the label
         */
        public final Boolean getShow()
        {
            return getBoolean( SHOW_KEY );
        }

        /**
         * Clear the visibility of the label
         */
        public final Label clearShow()
        {
            clear( SHOW_KEY );
            return this;
        }

        /**
         * Set the labelFormatter if you want to format the labels in some way, e.g. make them to links.
         */
        public final Label setFormatter( Formatter formatter )
        {
            assert null != formatter : "formatter can't be null";

            setFormatterNative( formatter );
            return this;
        }

        private native void setFormatterNative( Formatter formatter )
        /*-{
            this.formatter = function (label, series) {
                return formatter.@com.googlecode.gflot.client.options.PieSeriesOptions.Label.Formatter::format(Ljava/lang/String;Lcom/googlecode/gflot/client/Series;)(label, series);
            };
        }-*/;

        /**
         * Clear the label formatter
         */
        public final Label clearLabelFormatter()
        {
            clear( "formatter" );
            return this;
        }

        /**
         * Set the radius. 0-1 for percentage of fullsize, or a specified pixel length
         */
        public final Label setRadius( double radius )
        {
            put( RADIUS_KEY, radius );
            return this;
        }

        /**
         * @return the radius
         */
        public final Double getRadius()
        {
            return getDouble( RADIUS_KEY );
        }

        /**
         * Clear the radius
         */
        public final Label clearRadius()
        {
            clear( RADIUS_KEY );
            return this;
        }

        /**
         * Set the background
         */
        public final Label setBackground( Background background )
        {
            put( BACKGROUND_KEY, background );
            return this;
        }

        /**
         * @return the background
         */
        public final Background getBackground()
        {
            return getJsObject( BACKGROUND_KEY );
        }

        /**
         * Clear the background
         */
        public final Label clearBackground()
        {
            clear( BACKGROUND_KEY );
            return this;
        }

        /**
         * Set the threshold. 0-1 for the percentage value at which to hide labels (if they're too small)
         */
        public final Label setThreshold( double threshold )
        {
            assert threshold >= 0 && threshold <= 1 : "threshold range from 0.0 to 1.0";

            put( THRESHOLD_KEY, threshold );
            return this;
        }

        /**
         * @return the threshold
         */
        public final Double getThreshold()
        {
            return getDouble( THRESHOLD_KEY );
        }

        /**
         * Clear the threshold
         */
        public final Label clearThreshold()
        {
            clear( THRESHOLD_KEY );
            return this;
        }
    }

    public static class Combine
        extends JsonObject
    {
        private static final String THRESHOLD_KEY = "threshold";
        private static final String COLOR_KEY = "color";
        private static final String LABEL_KEY = "label";

        /**
         * Creates a {@link Combine}
         */
        public static final Combine create()
        {
            return JavaScriptObject.createObject().cast();
        }

        protected Combine()
        {
        }

        /**
         * Set the threshold. 0-1 for the percentage value at which to combine slices (if they're too small)
         */
        public final Combine setThreshold( double threshold )
        {
            assert threshold >= 0 && threshold <= 1 : "threshold range from 0.0 to 1.0";

            put( THRESHOLD_KEY, threshold );
            return this;
        }

        /**
         * @return the threshold
         */
        public final Double getThreshold()
        {
            return getDouble( THRESHOLD_KEY );
        }

        /**
         * Clear the threshold
         */
        public final Combine clearThreshold()
        {
            clear( THRESHOLD_KEY );
            return this;
        }

        /**
         * Set the color. any hexidecimal color value (other formats may or may not work, so best to stick with
         * something like '#CCC'), if null, the plugin will automatically use the color of the first slice to be
         * combined
         */
        public final Combine setColor( String color )
        {
            put( COLOR_KEY, color );
            return this;
        }

        /**
         * @return the color
         */
        public final String getColor()
        {
            return getString( COLOR_KEY );
        }

        /**
         * Clear the color
         */
        public final Combine clearColor()
        {
            clear( COLOR_KEY );
            return this;
        }

        /**
         * Set the label. Any text value of what the combined slice should be labeled
         */
        public final Combine setLabel( String label )
        {
            put( LABEL_KEY, label );
            return this;
        }

        /**
         * @return the label
         */
        public final String getLabel()
        {
            return getString( LABEL_KEY );
        }

        /**
         * Clear the label
         */
        public final Combine clearLabel()
        {
            clear( LABEL_KEY );
            return this;
        }
    }

    public static class Highlight
        extends JsonObject
    {
        private static final String OPACITY_KEY = "opacity";

        /**
         * Creates a {@link Highlight}
         */
        public static final Highlight create()
        {
            return JavaScriptObject.createObject().cast();
        }

        protected Highlight()
        {
        }

        /**
         * Set the opacity. Opacity range from 0.0 to 1.0.
         */
        public final Highlight setOpacity( double opacity )
        {
            assert opacity >= 0 && opacity <= 1 : "opacity range from 0.0 to 1.0";

            put( OPACITY_KEY, opacity );
            return this;
        }

        /**
         * @return the opacity
         */
        public final Double getOpacity()
        {
            return getDouble( OPACITY_KEY );
        }

        /**
         * Clear the opacity
         */
        public final Highlight clearOpacity()
        {
            clear( OPACITY_KEY );
            return this;
        }
    }

    private static final String RADIUS_KEY = "radius";
    private static final String INNER_RADIUS_KEY = "innerRadius";
    private static final String START_ANGLE_KEY = "startAngle";
    private static final String TILT_KEY = "tilt";
    private static final String OFFSET_KEY = "startAngle";
    private static final String STROKE_KEY = "stroke";
    private static final String LABEL_KEY = "label";
    private static final String COMBINE_KEY = "combine";
    private static final String HIGHLIGHT_KEY = "highlight";
    private static final String SHADOW_KEY = "shadow";

    /**
     * Creates a {@link PieSeriesOptions}
     */
    public static final PieSeriesOptions create()
    {
        return JavaScriptObject.createObject().cast();
    }

    protected PieSeriesOptions()
    {
    }

    /**
     * Set the radius. 0-1 for percentage of fullsize, or a specified pixel length
     */
    public final PieSeriesOptions setRadius( double radius )
    {
        put( RADIUS_KEY, radius );
        return this;
    }

    /**
     * @return the radius
     */
    public final Double getRadius()
    {
        return getDouble( RADIUS_KEY );
    }

    /**
     * Clear the radius
     */
    public final PieSeriesOptions clearRadius()
    {
        clear( RADIUS_KEY );
        return this;
    }

    /**
     * Set the inner radius to create a donut effect. 0-1 for percentage of fullsize or a specified pixel length
     */
    public final PieSeriesOptions setInnerRadius( double innerRadius )
    {
        put( INNER_RADIUS_KEY, innerRadius );
        return this;
    }

    /**
     * @return the inner radius
     */
    public final Double getInnerRadius()
    {
        return getDouble( INNER_RADIUS_KEY );
    }

    /**
     * Clear the inner radius
     */
    public final PieSeriesOptions clearInnerRadius()
    {
        clear( INNER_RADIUS_KEY );
        return this;
    }

    /**
     * Set the start angle. 0-2 factor of PI used for starting angle (in radians) i.e 3/2 starts at the top, 0 and 2
     * have the same result
     */
    public final PieSeriesOptions setStartAngle( double startAngle )
    {
        put( START_ANGLE_KEY, startAngle );
        return this;
    }

    /**
     * @return the start angle
     */
    public final Double getStartAngle()
    {
        return getDouble( START_ANGLE_KEY );
    }

    /**
     * Clear the start angle
     */
    public final PieSeriesOptions clearStartAngle()
    {
        clear( START_ANGLE_KEY );
        return this;
    }

    /**
     * Set the tilt. 0-1 for percentage to tilt the pie, where 1 is no tilt, and 0 is completely flat (nothing will
     * show)
     */
    public final PieSeriesOptions setTilt( double tilt )
    {
        put( TILT_KEY, tilt );
        return this;
    }

    /**
     * @return the tilt
     */
    public final Double getTilt()
    {
        return getDouble( TILT_KEY );
    }

    /**
     * Clear the tilt
     */
    public final PieSeriesOptions clearTilt()
    {
        clear( TILT_KEY );
        return this;
    }

    /**
     * Set the offset.
     */
    public final PieSeriesOptions setOffset( Offset offset )
    {
        put( OFFSET_KEY, offset );
        return this;
    }

    /**
     * @return the offset
     */
    public final Offset getOffset()
    {
        return getJsObject( OFFSET_KEY );
    }

    /**
     * Clear the offset
     */
    public final PieSeriesOptions clearOffset()
    {
        clear( OFFSET_KEY );
        return this;
    }

    /**
     * Set the stroke.
     */
    public final PieSeriesOptions setStroke( Stroke stroke )
    {
        put( STROKE_KEY, stroke );
        return this;
    }

    /**
     * @return the stroke
     */
    public final Stroke getStroke()
    {
        return getJsObject( STROKE_KEY );
    }

    /**
     * Set the label.
     */
    public final PieSeriesOptions setLabel( Label label )
    {
        put( LABEL_KEY, label );
        return this;
    }

    /**
     * @return the label
     */
    public final Label getLabel()
    {
        return getJsObject( LABEL_KEY );
    }

    /**
     * Set the combine.
     */
    public final PieSeriesOptions setCombine( Combine combine )
    {
        put( COMBINE_KEY, combine );
        return this;
    }

    /**
     * @return the combine
     */
    public final Combine getCombine()
    {
        return getJsObject( COMBINE_KEY );
    }

    /**
     * Set the Highlight.
     */
    public final PieSeriesOptions setHighlight( Highlight highlight )
    {
        put( HIGHLIGHT_KEY, highlight );
        return this;
    }

    /**
     * @return the highlight
     */
    public final Highlight getHighlight()
    {
        return getJsObject( HIGHLIGHT_KEY );
    }

    /**
     * Set the shadow.
     */
    public final PieSeriesOptions setShadow( Shadow shadow )
    {
        put( SHADOW_KEY, shadow );
        return this;
    }

    /**
     * @return the shadow
     */
    public final Shadow getShadow()
    {
        return getJsObject( SHADOW_KEY );
    }

    /**
     * Clear the shadow
     */
    public final PieSeriesOptions clearShadow()
    {
        clear( SHADOW_KEY );
        return this;
    }
}
