package com.github.yulichang.query;

import com.baomidou.mybatisplus.core.conditions.AbstractWrapper;
import com.baomidou.mybatisplus.core.conditions.SharedString;
import com.baomidou.mybatisplus.core.conditions.query.Query;
import com.baomidou.mybatisplus.core.conditions.segments.MergeSegments;
import com.baomidou.mybatisplus.core.exceptions.MybatisPlusException;
import com.baomidou.mybatisplus.core.metadata.TableFieldInfo;
import com.baomidou.mybatisplus.core.metadata.TableInfo;
import com.baomidou.mybatisplus.core.toolkit.*;
import com.github.yulichang.adapter.AdapterHelper;
import com.github.yulichang.adapter.base.tookit.VersionUtils;
import com.github.yulichang.config.ConfigProperties;
import com.github.yulichang.query.interfaces.CompareIfExists;
import com.github.yulichang.query.interfaces.StringJoin;
import com.github.yulichang.toolkit.MPJSqlInjectionUtils;
import com.github.yulichang.toolkit.StrUtils;
import com.github.yulichang.toolkit.TableHelper;
import com.github.yulichang.toolkit.ThrowOptional;
import com.github.yulichang.wrapper.enums.IfExistsSqlKeyWordEnum;
import com.github.yulichang.wrapper.interfaces.Chain;
import lombok.Getter;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * copy {@link com.baomidou.mybatisplus.core.conditions.query.QueryWrapper}
 * 推荐使用 JoinWrappers.&lt;UserDO&gt;query();构造
 *
 * @author yulichang
 * @see com.github.yulichang.toolkit.JoinWrappers
 */
@SuppressWarnings("unused")
public class MPJQueryWrapper<T> extends AbstractWrapper<T, String, MPJQueryWrapper<T>> implements
        Query<MPJQueryWrapper<T>, T, String>, StringJoin<MPJQueryWrapper<T>, T>, Chain<T>,
        CompareIfExists<MPJQueryWrapper<T>, String> {

    /**
     * 查询字段
     */
    private SharedString sqlSelect = new SharedString();

    /**
     * 连表字段
     */
    private SharedString from = SharedString.emptyString();

    /**
     * 主表别名
     */
    @Getter
    private String alias = ConfigProperties.tableAlias;

    /**
     * 查询的列
     */
    private List<String> selectColumns = new ArrayList<>();

    /**
     * 排除的字段
     */
    private List<String> ignoreColumns = new ArrayList<>();

    /**
     * 是否 select distinct
     */
    private boolean selectDistinct = false;
    /**
     * 主表逻辑删除
     */
    private boolean logicSql = true;

    /**
     * 动态表名
     */
    private Function<String, String> tableNameFunc;

    /**
     * 检查 SQL 注入过滤
     */
    private boolean checkSqlInjection = false;

    @Getter
    private BiPredicate<Object, IfExistsSqlKeyWordEnum> ifExists = ConfigProperties.ifExists;


    public MPJQueryWrapper() {
        super.initNeed();
    }

    public MPJQueryWrapper(Class<T> clazz) {
        try {
            super.setEntityClass(clazz);
        } catch (NoSuchMethodError error) {
            if (VersionUtils.compare(VersionUtils.getVersion(), "3.3.0") > 0) {
                throw error;
            }
        }
        super.initNeed();
    }

    public MPJQueryWrapper(T entity) {
        super.setEntity(entity);
        super.initNeed();
    }

    /**
     * 非对外公开的构造方法,只用于生产嵌套 sql
     *
     * @param entityClass 本不应该需要的
     */
    public MPJQueryWrapper(T entity, Class<T> entityClass, AtomicInteger paramNameSeq,
                           Map<String, Object> paramNameValuePairs, MergeSegments mergeSegments,
                           SharedString sqlSelect, SharedString from, SharedString lastSql,
                           SharedString sqlComment, SharedString sqlFirst,
                           List<String> selectColumns, List<String> ignoreColumns, boolean selectDistinct,
                           BiPredicate<Object, IfExistsSqlKeyWordEnum> IfExists) {
        super.setEntity(entity);
        setEntityClass(entityClass);
        this.paramNameSeq = paramNameSeq;
        this.paramNameValuePairs = paramNameValuePairs;
        this.expression = mergeSegments;
        this.sqlSelect = sqlSelect;
        this.lastSql = lastSql;
        this.from = from;
        this.sqlComment = sqlComment;
        ThrowOptional.tryDo(() -> this.sqlFirst = sqlFirst).catchDo();
        this.selectColumns = selectColumns;
        this.ignoreColumns = ignoreColumns;
        this.selectDistinct = selectDistinct;
        this.ifExists = IfExists;
    }

    /**
     * 开启检查 SQL 注入
     */
    public MPJQueryWrapper<T> checkSqlInjection() {
        this.checkSqlInjection = true;
        return this;
    }

    public MPJQueryWrapper<T> setIfExists(BiPredicate<Object, IfExistsSqlKeyWordEnum> IfExists) {
        this.ifExists = IfExists;
        return this;
    }

    public MPJQueryWrapper<T> setIfExists(Predicate<Object> IfExists) {
        this.ifExists = (o, k) -> IfExists.test(o);
        return this;
    }

    @Override
    protected String columnToString(String column) {
        if (checkSqlInjection && MPJSqlInjectionUtils.check(column)) {
            throw new MybatisPlusException("Discovering SQL injection column: " + column);
        }
        return column;
    }

    /**
     * sql去重
     * select distinct
     */
    public MPJQueryWrapper<T> distinct() {
        this.selectDistinct = true;
        return typedThis;
    }

    @Override
    public MPJQueryWrapper<T> select(String... columns) {
        if (ArrayUtils.isNotEmpty(columns)) {
            selectColumns.addAll(Arrays.asList(columns));
        }
        return typedThis;
    }

    @Override
    public MPJQueryWrapper<T> select(boolean condition, List<String> columns) {
        if (condition && CollectionUtils.isNotEmpty(columns)) {
            selectColumns.addAll(columns);
        }
        return typedThis;
    }

    /**
     * 忽略查询字段
     * <p>
     * 用法: selectIgnore("t.id","t.sex","a.area")
     *
     * @since 1.1.3
     */
    public MPJQueryWrapper<T> selectIgnore(String... columns) {
        if (ArrayUtils.isNotEmpty(columns)) {
            ignoreColumns.addAll(Arrays.asList(columns));
        }
        return typedThis;
    }

    /**
     * 此方法只能用于主表
     * 不含主键
     *
     * @param entityClass 主表class
     * @param predicate   条件lambda
     */
    @Override
    public MPJQueryWrapper<T> select(Class<T> entityClass, Predicate<TableFieldInfo> predicate) {
        TableInfo info = TableHelper.getAssert(entityClass);
        selectColumns.addAll(info.getFieldList().stream().filter(predicate).map(c ->
                alias + StringPool.DOT + c.getSqlSelect()).collect(Collectors.toList()));
        return typedThis;
    }


    /**
     * 查询主表全部字段
     *
     * @param clazz 主表class
     */
    public final MPJQueryWrapper<T> selectAll(Class<T> clazz) {
        selectAll(clazz, alias);
        return typedThis;
    }

    /**
     * 查询指定实体全部字段
     *
     * @param as 实体对应的别名
     */
    @SuppressWarnings({"DuplicatedCode", "UnusedReturnValue"})
    public final MPJQueryWrapper<T> selectAll(Class<?> clazz, String as) {
        TableInfo info = TableHelper.getAssert(clazz);
        if (AdapterHelper.getAdapter().mpjHasPK(info)) {
            selectColumns.add(as + StringPool.DOT + info.getKeySqlSelect());
        }
        selectColumns.addAll(info.getFieldList().stream().map(i ->
                as + StringPool.DOT + i.getSqlSelect()).collect(Collectors.toList()));
        return typedThis;
    }

    @Override
    public String getSqlSelect() {
        if (StrUtils.isBlank(sqlSelect.getStringValue())) {
            if (CollectionUtils.isNotEmpty(ignoreColumns)) {
                selectColumns.removeIf(ignoreColumns::contains);
            }
            sqlSelect.setStringValue(String.join(StringPool.COMMA, selectColumns));
        }
        return sqlSelect.getStringValue();
    }

    public boolean getSelectDistinct() {
        return selectDistinct;
    }

    public String getFrom() {
        return from.getStringValue();
    }

    /**
     * 设置主表别名
     * 如果要用，请最先调用，
     * <pre>
     * 正例  new QueryWrapper().setAlias("a").selectAll(UserDO.class)....
     * 反例  new QueryWrapper().selectAll(UserDO.class).setAlias("a")....
     * <pre/>
     *
     * @param alias 主表别名
     */
    public MPJQueryWrapper<T> setAlias(String alias) {
        Assert.isTrue(StrUtils.isNotBlank(alias), "别名不能为空");
        this.alias = alias;
        return this;
    }

    /**
     * 逻辑删除
     */
    public String getSubLogicSql() {
        return StringPool.EMPTY;
    }

    /**
     * 关闭主表逻辑删除
     */
    public MPJQueryWrapper<T> disableLogicDel() {
        this.logicSql = false;
        return typedThis;
    }

    /**
     * 启用主表逻辑删除
     */
    public MPJQueryWrapper<T> enableLogicDel() {
        this.logicSql = true;
        return typedThis;
    }

    /**
     * 逻辑删除
     */
    public boolean getLogicSql() {
        return logicSql;
    }

    /**
     * 动态表名
     * <p>
     * 如果主表需要动态表名
     * <p>
     */
    public MPJQueryWrapper<T> setTableName(Function<String, String> func) {
        this.tableNameFunc = func;
        return typedThis;
    }

    public String getTableName(String tableName) {
        if (this.tableNameFunc == null) {
            return tableName;
        }
        return this.tableNameFunc.apply(tableName);
    }

    @SuppressWarnings("DuplicatedCode")
    public String getTableNameEnc(String tableName) {
        String decode;
        try {
            decode = URLDecoder.decode(tableName, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            decode = tableName;
        }
        if (this.tableNameFunc == null) {
            return decode;
        }
        return this.tableNameFunc.apply(decode);
    }

    /**
     * 返回一个支持 lambda 函数写法的 wrapper
     */
    public MPJLambdaQueryWrapper<T> lambda() {
        return new MPJLambdaQueryWrapper<>(getEntity(), getEntityClass(), from, sqlSelect, paramNameSeq, paramNameValuePairs,
                expression, lastSql, sqlComment, getSqlFirstField(), selectColumns, ignoreColumns, selectDistinct, ifExists)
                .setAlias(this.alias);
    }

    @Override
    public Class<T> getEntityClass() {
        try {
            return super.getEntityClass();
        } catch (Throwable throwable) {
            return null;
        }
    }

    @Override
    public boolean isResultMapCollection() {
        return false;
    }

    @Override
    public MPJQueryWrapper<T> setEntityClass(Class<T> entityClass) {
        try {
            return super.setEntityClass(entityClass);
        } catch (Throwable throwable) {
            return this;
        }
    }

    public SharedString getSqlFirstField() {
        try {
            return sqlSelect;
        } catch (Throwable throwable) {
            return SharedString.emptyString();
        }
    }

    /**
     * 用于生成嵌套 sql
     * <p>故 sqlSelect selectColumn ignoreColumns from不向下传递</p>
     */
    @Override
    protected MPJQueryWrapper<T> instance() {
        return new MPJQueryWrapper<>(getEntity(), getEntityClass(), paramNameSeq, paramNameValuePairs, new MergeSegments(),
                null, null, SharedString.emptyString(), SharedString.emptyString(), SharedString.emptyString(),
                null, null, selectDistinct, ifExists);
    }


    @Override
    public void clear() {
        super.clear();
        sqlSelect.toNull();
        from.toNull();
        selectColumns.clear();
        ignoreColumns.clear();
        ifExists = ConfigProperties.ifExists;
    }

    @Override
    public MPJQueryWrapper<T> join(String keyWord, boolean condition, String joinSql, Object... params) {
        if (condition) {
            from.setStringValue(from.getStringValue() + StringPool.SPACE + keyWord +
                    StringPool.SPACE + mpjFormatSqlMaybeWithParam(joinSql, params));
        }
        return typedThis;
    }

    @SuppressWarnings("DuplicatedCode")
    protected final String mpjFormatSqlMaybeWithParam(String sqlStr, Object... params) {
        if (StrUtils.isBlank(sqlStr)) {
            return null;
        }
        if (ArrayUtils.isNotEmpty(params)) {
            for (int i = 0; i < params.length; ++i) {
                String target = Constants.LEFT_BRACE + i + Constants.RIGHT_BRACE;
                if (sqlStr.contains(target)) {
                    sqlStr = sqlStr.replace(target, formatParam(null, params[i]));
                } else {
                    Matcher matcher = Pattern.compile("[{]" + i + ",[a-zA-Z0-9.,=]+}").matcher(sqlStr);
                    if (!matcher.find()) {
                        throw ExceptionUtils.mpe("Please check the syntax correctness! sql not contains: \"%s\"", target);
                    }
                    String group = matcher.group();
                    sqlStr = sqlStr.replace(group, formatParam(group.substring(target.length(), group.length() - 1), params[i]));
                }
            }
        }
        return sqlStr;
    }
}
