/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.cloud.context.edas;

import com.alibaba.cloud.context.AliCloudSdk;

import com.aliyuncs.edas.model.v20170801.GetSecureTokenRequest;
import com.aliyuncs.edas.model.v20170801.GetSecureTokenResponse;
import com.aliyuncs.edas.model.v20170801.InsertApplicationRequest;
import com.aliyuncs.edas.model.v20170801.InsertApplicationResponse;
import com.aliyuncs.exceptions.ClientException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.ConcurrentHashMap;

/**
 * EDAS-SDK实现，可以方便的调用EDAS的POP-API。
 *
 * @author xiaolongzuo
 */
public class AliCloudEdasSdk {

    private static final Logger log = LoggerFactory.getLogger(AliCloudEdasSdk.class);

    private static final int SUCCESS_CODE = 200;

    private static final int DEFAULT_BUILD_PACK = -2;

    private AliCloudSdk aliCloudSdk;

    private ConcurrentHashMap<String, GetSecureTokenResponse.SecureToken> secureTokenMap = new ConcurrentHashMap<>();

    private ConcurrentHashMap<String, InsertApplicationResponse.ApplicationInfo> applicationInfoMap
        = new ConcurrentHashMap<>();

    public AliCloudEdasSdk(AliCloudSdk aliCloudSdk) {
        this.aliCloudSdk = aliCloudSdk;
    }

    public AliCloudSdk getAliCloudSdk() {
        return aliCloudSdk;
    }

    public GetSecureTokenResponse.SecureToken getSecureToken(String namespace) {
        GetSecureTokenResponse.SecureToken secureToken = secureTokenMap.get(namespace);
        if (secureToken == null) {
            GetSecureTokenRequest getSecureTokenRequest = new GetSecureTokenRequest();
            getSecureTokenRequest.setNamespaceId(namespace);

            GetSecureTokenResponse getSecureTokenResponse;
            try {
                getSecureTokenResponse = aliCloudSdk.getAcsResponse(getSecureTokenRequest);
            } catch (ClientException e) {
                log.error("Get info from edas failed.", e);
                throw new RuntimeException(e);
            }
            if (!isSuccess(getSecureTokenResponse.getCode())) {
                throw new RuntimeException(null == getSecureTokenResponse.getMessage()
                    ? "Get info from edas failed, response is null." : getSecureTokenResponse.getMessage());
            }
            secureTokenMap.putIfAbsent(namespace, getSecureTokenResponse.getSecureToken());
            secureToken = secureTokenMap.get(namespace);
        }
        return secureToken;
    }

    public InsertApplicationResponse.ApplicationInfo getApplicationInfo(
        String applicationName, String namespace) {
        String key = applicationName + "---" + namespace;
        InsertApplicationResponse.ApplicationInfo applicationInfo = applicationInfoMap.get(key);
        if (applicationInfo == null) {
            InsertApplicationRequest request = new InsertApplicationRequest();
            request.setApplicationName(applicationName);
            request.setLogicalRegionId(namespace);
            request.setBuildPackId(DEFAULT_BUILD_PACK);

            InsertApplicationResponse insertApplicationResponse;
            try {
                insertApplicationResponse = aliCloudSdk.getAcsResponse(request);
            } catch (ClientException e) {
                log.error("Register app from edas failed.", e);
                throw new RuntimeException(e);
            }
            if (!isSuccess(insertApplicationResponse.getCode())) {
                throw new RuntimeException(
                    null == insertApplicationResponse.getMessage() ? "" : insertApplicationResponse.getMessage());
            }
            applicationInfoMap.putIfAbsent(key, insertApplicationResponse.getApplicationInfo());
            applicationInfo = applicationInfoMap.get(key);
        }
        return applicationInfo;
    }

    public boolean isSuccess(int code) {
        return code == SUCCESS_CODE;
    }

}
