package org.violet.commom.crypto.advice;

import com.alibaba.fastjson2.JSONObject;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.Order;
import org.springframework.http.MediaType;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.servlet.mvc.method.annotation.ResponseBodyAdvice;
import org.violet.commom.crypto.annotation.encrypt.ApiEncrypt;
import org.violet.commom.crypto.bean.CryptoInfoBean;
import org.violet.commom.crypto.config.ApiCryptoProperties;
import org.violet.commom.crypto.exception.EncryptBodyFailException;
import org.violet.commom.crypto.util.ApiCryptoUtil;
import org.violet.common.core.util.ClassUtil;
import org.violet.common.core.util.JsonUtil;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;


/**
 * 响应数据的加密处理<br>
 * 本类只对控制器参数中含有<strong>{@link org.springframework.web.bind.annotation.ResponseBody}</strong>
 * 或者控制类上含有<strong>{@link org.springframework.web.bind.annotation.RestController}</strong>
 * 以及package为<strong><code>org.violet.commom.crypto.annotation.encrypt</code></strong>下的注解有效
 *
 * @see ResponseBodyAdvice
 */
@Slf4j
@Order(1)
@AutoConfiguration
@ControllerAdvice
@RequiredArgsConstructor
@ConditionalOnProperty(value = ApiCryptoProperties.PREFIX + ".enabled", havingValue = "true", matchIfMissing = true)
public class ApiEncryptResponseBodyAdvice implements ResponseBodyAdvice<Object> {
    private final ApiCryptoProperties properties;

    @Override
    public boolean supports(MethodParameter returnType, @NonNull Class converterType) {
        return ClassUtil.isAnnotated(returnType.getMethod(), ApiEncrypt.class);
    }

    @Nullable
    @Override
    public Object beforeBodyWrite(Object body, @NonNull MethodParameter returnType, @NonNull MediaType selectedContentType,
                                  @NonNull Class selectedConverterType, @NonNull ServerHttpRequest request, @NonNull ServerHttpResponse response) {
        if (body == null) {
            return null;
        }
        response.getHeaders().setContentType(MediaType.TEXT_PLAIN);
        CryptoInfoBean cryptoInfoBean = ApiCryptoUtil.getEncryptInfo(returnType);
        if (cryptoInfoBean != null) {
            byte[] bodyJsonBytes = JsonUtil.toJsonAsBytes(body);

            try (OutputStream stream = response.getBody()) {
                // 将加密后的字符串写入输出流之前，先将其转换为字节数组，然后再写入输出流。这样可以确保不会在写入过程中引入额外的引号或其他格式问题。
                stream.write(ApiCryptoUtil.encryptData(properties, bodyJsonBytes, cryptoInfoBean).getBytes(StandardCharsets.UTF_8));
                stream.flush();
            } catch (IOException e) {
                log.error("beforeBodyWrite stream write error", e);
            }
            return null;
        }
        throw new EncryptBodyFailException();
    }

}
