package com.kdgcsoft.power.filepreview;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.beetl.core.Configuration;
import org.beetl.core.GroupTemplate;
import org.beetl.core.Template;
import org.beetl.core.resource.ClasspathResourceLoader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.filepreview.provider.IPreviewProvider;
import com.kdgcsoft.power.filepreview.provider.PreviewInfo;
import com.kdgcsoft.power.filepreview.provider.PreviewBtn;
import com.kdgcsoft.power.filepreview.utils.CompressFileReader;
import com.xiaoleilu.hutool.util.ArrayUtil;


public class FilePreviewManager {
	private static Logger logger = LoggerFactory.getLogger(FilePreviewManager.class);
	/**
	 * 开源压缩文件格式
	 */
	public static final String[] zipfile= {"zip","jar","tar","7z"};
	/**
	 * rar压缩格式
	 */
	public static final String[] rarfile= {"rar"};
	/**
	 * pdf,html
	 */
	public static final String[] pdffile= {"pdf","html"};
	/**
	 * 音频类型
	 */
	public static final String[] audiofile= {"mp3","m4a"};
	/**
	 * 视频类型
	 */
	public static final String[] videofile= {"m4v","webm","ogv","mp4"};
	
	/**
	 * 文本类型文件
	 */
	public static final String[] textfile= {"txt","sql","ini","properties","log","js","css","json","java","md","ini","xml"};
	/**
	 * 图片类型文件
	 */
	public static final String[] imgfile= {"img","png","jpg","bmp","jpeg","gif","ico"};
	private static final String templatePath="com/kdgcsoft/power/filepreview/resource/template";
	private static FilePreviewManager instance;
	private IPreviewProvider previewProvider;
	private GroupTemplate gt;
	private String contextPath;
	/**
	 * 初始化单例模式的FilePreviewManager
	 * @param provider
	 * @param contextPath Servlet对应的上下文路径
	 * @return 返回FilePreviewManager 的单例
	 * @throws IOException
	 */
	public synchronized static FilePreviewManager init(IPreviewProvider provider,String contextPath) throws IOException {
		if(instance == null) {
			instance = new FilePreviewManager(provider,contextPath);
		}
		return instance;
	}
	/**
	 * 取得初始化过后的FilePreviewManager
	 * @return 取得初始化过后的FilePreviewManager
	 * @throws IOException
	 */
	public static FilePreviewManager get() throws IOException {
		if(instance==null) {
			logger.error("FilePreviewManager没有初始化,请检查web.xml中是否配置了servlet 及load-on-startup=1");
			return null;
		}
		return instance;
	}
	
	public FilePreviewManager(IPreviewProvider previewProvider,String contextPath) throws IOException {
		this.previewProvider = previewProvider;
		this.contextPath=contextPath;
		ClasspathResourceLoader resourceLoader = new ClasspathResourceLoader();
		Configuration cfg = Configuration.defaultConfiguration();
		this.gt = new GroupTemplate(resourceLoader, cfg);
	}


	public IPreviewProvider getPreviewProvider() {
		return previewProvider;
	}
	public void setPreviewProvider(IPreviewProvider provider) {
		this.previewProvider = provider;
	}
	/**
	 * 获得预览页面的html内容,会根据不同的预览类型返回相应模版生成的html页面
	 * @param request
	 * @return 返回生成的html页面字符串
	 */
	public  String getPreviewHtml(HttpServletRequest request) {
		String html="";
		String queryString=request.getQueryString();//获得请求参数
		if(previewProvider.canPreview(request)) {//有权限预览
			PreviewInfo fileinfo=previewProvider.getPreviewInfo(request);
			logger.debug("fileinfo:"+fileinfo);
			boolean candownload=previewProvider.canDownload(request);//预览页面是否显示下载连接
			logger.debug("下载权限:"+candownload);
			List<PreviewBtn> btnlist=previewProvider.morePreviewBtn(request);//用来构建更多的操作按钮
			logger.debug("构造更多按钮:"+btnlist);
			Template t=null;
			if(!fileinfo.isMultiView()&&ArrayUtil.containsIgnoreCase(pdffile, fileinfo.getViewtype())) {//如果是PDF 则调用PDF的预览页面
				logger.debug("使用模版preview_pdf.btl");
				t = gt.getTemplate(templatePath+"/preview_pdf.btl");
			}else if(!fileinfo.isMultiView()&&ArrayUtil.containsIgnoreCase(textfile, fileinfo.getViewtype())) {//如果是简单的文本文件
				logger.debug("使用模版preview_txt.btl");
				t = gt.getTemplate(templatePath+"/preview_txt.btl");
			}else if(!fileinfo.isMultiView()&&ArrayUtil.containsIgnoreCase(imgfile, fileinfo.getViewtype())) {//如果是单图片
				logger.debug("使用模版preview_img.btl");
				t = gt.getTemplate(templatePath+"/preview_img.btl");
			}else if(!fileinfo.isMultiView()&&ArrayUtil.containsIgnoreCase(audiofile, fileinfo.getViewtype())) {//如果是音频
				logger.debug("使用模版preview_audio.btl");
				t = gt.getTemplate(templatePath+"/preview_audio.btl");
			}else if(!fileinfo.isMultiView()&&ArrayUtil.containsIgnoreCase(videofile, fileinfo.getViewtype())) {//如果是视频
				logger.debug("使用模版preview_video.btl");
				t = gt.getTemplate(templatePath+"/preview_video.btl");
			}else if(!fileinfo.isMultiView()&&ArrayUtil.containsIgnoreCase(zipfile, fileinfo.getViewtype())) {//如果zip压缩文件
				logger.debug("使用模版preview_compress.btl");
				t = gt.getTemplate(templatePath+"/preview_compress.btl");
				InputStream previewstream=previewProvider.getPreviewStream(request);
				List<Map<String,String>> filelist=CompressFileReader.readZipFile(previewstream,fileinfo.getViewtype());
				t.binding("filelist", filelist);
			}else if(!fileinfo.isMultiView()&&ArrayUtil.containsIgnoreCase(rarfile, fileinfo.getViewtype())) {//如果rar压缩文件
				logger.debug("使用模版preview_compress.btl");
				t = gt.getTemplate(templatePath+"/preview_compress.btl");
				InputStream previewstream=previewProvider.getPreviewStream(request);
				List<Map<String,String>> filelist=CompressFileReader.readRarFile(previewstream);
				t.binding("filelist", filelist);
			}else if(fileinfo.isMultiView()&&ArrayUtil.containsIgnoreCase(imgfile, fileinfo.getViewtype())) {//如果是多图片
				logger.debug("使用模版preview_mulitimg.btl");
				t = gt.getTemplate(templatePath+"/preview_mulitimg.btl");
			}else {//不支持的文件类型
				logger.debug("不支持的文件类型:"+fileinfo.getViewtype());
				t = gt.getTemplate(templatePath+"/notsupport.btl");
			}
			t.binding("ctx", contextPath);
			t.binding("queryString", queryString);
			t.binding("fileinfo", fileinfo);
			t.binding("candownload", candownload);
			t.binding("btnlist", btnlist);
			html = t.render();
		}else {
			logger.debug("没有预览权限");
			Template t = gt.getTemplate(templatePath+"/nopromission.btl");
			t.binding("ctx", contextPath);
			t.binding("queryString", queryString);
			html = t.render();
		}
		return html;
	}
}
