package com.kdgcsoft.power.filepreview.utils;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.Collator;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.compress.archivers.sevenz.SevenZArchiveEntry;
import org.apache.commons.compress.archivers.sevenz.SevenZFile;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.junrar.Archive;
import com.github.junrar.rarfile.FileHeader;
import com.xiaoleilu.hutool.io.FileUtil;
import com.xiaoleilu.hutool.io.IoUtil;

public class CompressFileReader {

	private static Logger logger = LoggerFactory.getLogger(CompressFileReader.class);
	
	private final static Collator TEXT_COMPARATOR = Collator.getInstance();
	private final static int MAX_DISPLAY_ITEMS = 100;
	
	/**
	 * 用于显示错误信息的条目
	 */
	private static List<Map<String, String>> READ_ERROR_RESULT = new ArrayList<Map<String, String>>();
	/**
	 * 用于显示达到最大显示数量的条目
	 */
	private static Map<String, String> MAX_ITEMS_MESSAGE = createFileInfo("---预览界面最多只显示" + MAX_DISPLAY_ITEMS + "条记录---", "");
	static {
		READ_ERROR_RESULT.add(createFileInfo("读取文件失败！", ""));
	}

	/**
	 * 读取zip文件内容，返回文件名和大小的列表。如果失败，则读取的内容为一条失败消息条目。
	 * @param is zip文件输入流
	 * @param viewtype 文件扩展名，不包括"."，例如jar、zip等。
	 * @return 文件名及其大小的明细列表
	 */
	public static List<Map<String, String>> readZipFile(InputStream is, String viewtype) {
		List<Map<String, String>> filelist = new ArrayList<Map<String, String>>();
		int count = 0;
		if ("zip".equalsIgnoreCase(viewtype) || "jar".equalsIgnoreCase(viewtype)) {
			File tmpfile = StreamToTmpFile(is);
			if (tmpfile == null) {
				return READ_ERROR_RESULT;
			}

			ZipFile zipFile = null;
			try {
				zipFile = new ZipFile(tmpfile, ResourceUtil.getFileEncodeUTFGBK(tmpfile.getAbsolutePath()));
				Enumeration<ZipArchiveEntry> entries = zipFile.getEntries();
				while (entries.hasMoreElements()) {
					ZipArchiveEntry entry = entries.nextElement();
					filelist.add(createFileInfo(entry.getName(), entry.isDirectory(), entry.getCompressedSize()));
					if (++count >= 100) {
						filelist.add(MAX_ITEMS_MESSAGE);
						break;
					}
				}
			} catch (IOException e1) {
				logger.error("读取压缩文件异常", e1);
				return READ_ERROR_RESULT;
			} finally {
				IoUtil.close(zipFile);
				deleteTmpFile(tmpfile);
			}
		} else if ("tar".equalsIgnoreCase(viewtype)) {
			TarArchiveInputStream tarIn = new TarArchiveInputStream(is, 2048);
			TarArchiveEntry entry = null;
			try {
				while ((entry = tarIn.getNextTarEntry()) != null) {
					filelist.add(createFileInfo(entry.getName(), entry.isDirectory(), entry.getSize()));
					if (++count >= 100) {
						filelist.add(MAX_ITEMS_MESSAGE);
						break;
					}

				}
			} catch (IOException e) {
				logger.error("读取压缩文件异常", e);
				return READ_ERROR_RESULT;
			} finally {
				IoUtil.close(tarIn);
			}
		} else if ("7z".equalsIgnoreCase(viewtype)) {
			File tmpfile = StreamToTmpFile(is);
			if (tmpfile == null) {
				return READ_ERROR_RESULT;
			}
			SevenZFile sevenZFile = null;
			try {
				sevenZFile = new SevenZFile(tmpfile);
				SevenZArchiveEntry entry = null;
				while ((entry = sevenZFile.getNextEntry()) != null) {
					filelist.add(createFileInfo(entry.getName(), entry.isDirectory(), entry.getSize()));
					if (++count >= 100) {
						filelist.add(MAX_ITEMS_MESSAGE);
						break;
					}
				}
			} catch (IOException e) {
				logger.error("读取压缩文件异常", e);
				return READ_ERROR_RESULT;
			} finally {
				IoUtil.close(sevenZFile);
				deleteTmpFile(tmpfile);
			}
		}

		return filelist;
	}

	/**
	 * 读取rar文件内容，返回文件名和大小的列表。如果失败，则读取的内容为一条失败消息条目。
	 * @param is rar文件输入流
	 * @return 文件名及其大小的明细列表
	 */
	public static List<Map<String, String>> readRarFile(InputStream is) {
		List<Map<String, String>> filelist = new ArrayList<Map<String, String>>();
		File tmpfile = StreamToTmpFile(is);
		if (tmpfile == null) {
			return READ_ERROR_RESULT; 
		}
		Archive archive = null;
		try {
			archive = new Archive(tmpfile);
			List<FileHeader> headers = archive.getFileHeaders();
			if (headers.size() == 0) {
				return READ_ERROR_RESULT;
			}
			sortHeaders(headers);
			int count = 0;
			for (FileHeader header : headers) {
				String fullName;
				if (header.isUnicode()) {
					fullName = header.getFileNameW();
				} else {
					fullName = header.getFileNameString();
				}
				filelist.add(createFileInfo(fullName, header.isDirectory(), header.getDataSize()));
				if (++count >= 100) {
					filelist.add(MAX_ITEMS_MESSAGE);
					break;
				}
			}
		} catch (Exception e) {
			logger.error("读取Rar文件异常", e);
			return READ_ERROR_RESULT; 
		} finally {
			IoUtil.close(archive);
			deleteTmpFile(tmpfile);
		}

		return filelist;
	}

	/**
	 * 把参数按照文件名字符顺序排序
	 * @param headers 被排序的文件条目列表
	 */
	public static void sortHeaders(List<FileHeader> headers) {
		
		Collections.sort(headers, new Comparator<FileHeader>() {
			@Override
			public int compare(FileHeader o1, FileHeader o2) {
				if (o1.isDirectory() == o2.isDirectory()) {
					return TEXT_COMPARATOR.compare(o1.getFileNameW(), o2.getFileNameW());
				} else {
					if (o1.isDirectory()) {
						return -1;
					} else {
						return 1;
					}
				}
			}
		});

	}
	
	private static Map<String, String> createFileInfo(String fileName, String fileSize) {
		Map<String, String> fileInfo = new HashMap<String, String>();
		fileInfo.put("filename", fileName);
		fileInfo.put("filesize", fileSize);
		return fileInfo;
	}
	
	private static Map<String, String> createFileInfo(String fileName, boolean isDirectory, long fileSize) {
		if (isDirectory) {
			return createFileInfo(fileName, "");
		} else {
			return createFileInfo(fileName, getDataSize(fileSize));
		}
		
	}

	public static String getDataSize(long size) {
		DecimalFormat formater = new DecimalFormat("####.00");
		if (size <= 0) {
			return "0";
		} else if (size < 1024) {
			return size + "bytes";
		} else if (size < 1024 * 1024) {
			float kbsize = size / 1024f;
			return formater.format(kbsize) + "KB";
		} else if (size < 1024 * 1024 * 1024) {
			float mbsize = size / 1024f / 1024f;
			return formater.format(mbsize) + "MB";
		} else if (size < 1024 * 1024 * 1024 * 1024) {
			float gbsize = size / 1024f / 1024f / 1024f;
			return formater.format(gbsize) + "GB";
		} else {
			return "size: error";
		}
	}

	/**
	 * 接收流，写入临时文件，并返回文件
	 * @param is 输入流
	 * @return 临时文件。如果创建失败，则返回null
	 */
	public static File StreamToTmpFile(InputStream is) {
		File tmpfile = null;
		OutputStream os = null;
		try {
			tmpfile = FileUtil.createTempFile("filepreview", null, null, true);
			os = new FileOutputStream(tmpfile);
			IoUtil.copy(is, os, 8192);
		} catch (Exception e) {
			logger.error("数据流写入文件异常", e);
		} finally {
			IoUtil.close(os);
			IoUtil.close(is);
		}
		return tmpfile;
	}

	public static void deleteTmpFile(File tmpfile) {
		if (tmpfile != null && tmpfile.exists()) {
			tmpfile.delete();
		}
	}
}
