package org.violet.common.mybatis.support;

import cn.hutool.core.io.IoUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.http.ContentType;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.MethodParameter;
import org.springframework.web.bind.support.WebDataBinderFactory;
import org.springframework.web.context.request.NativeWebRequest;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.method.support.ModelAndViewContainer;
import org.violet.common.mybatis.pojo.PageRequest;

@Slf4j
public class PageRequestHandlerMethodArgumentResolver implements HandlerMethodArgumentResolver {

    public static final String CURRENT_PAGE = "current";
    public static final String PAGE_SIZE = "size";
    public static final String SORTS = "_sorts";
    public static final String ORDERS = "_orders";

    public PageRequestHandlerMethodArgumentResolver() {
    }

    public boolean supportsParameter(MethodParameter methodParameter) {
        return methodParameter.getParameterType().isAssignableFrom(PageRequest.class);
    }

    public Object resolveArgument(MethodParameter methodParameter, ModelAndViewContainer modelAndViewContainer, NativeWebRequest nativeWebRequest, WebDataBinderFactory webDataBinderFactory) {
        HttpServletRequest httpServletRequest = (HttpServletRequest)nativeWebRequest.getNativeRequest();

        try {
            String contentTypeStr = httpServletRequest.getContentType();
            long currentPage;
            long pageSize;
            if (StrUtil.contains(contentTypeStr, ContentType.JSON.getValue())) {
                String bodyJsonStr = IoUtil.readUtf8(httpServletRequest.getInputStream());
                JSONObject jsonObject = JSON.parseObject(bodyJsonStr);
                currentPage = jsonObject.getLongValue("current");
                pageSize = jsonObject.getLongValue("size", 20L);
                PageRequest page = PageRequest.of(currentPage, pageSize);
                page.buildSorts(jsonObject.getString("_sorts"), jsonObject.getString("_orders"));
                return page;
            }

            if (StrUtil.isBlank(contentTypeStr) || StrUtil.contains(contentTypeStr, ContentType.FORM_URLENCODED.getValue())) {
                currentPage = NumberUtil.parseLong(nativeWebRequest.getParameter("current"));
                pageSize = NumberUtil.parseLong(nativeWebRequest.getParameter("size"));
                PageRequest page = PageRequest.of(currentPage, pageSize);
                page.buildSorts(nativeWebRequest.getParameter("_sorts"), nativeWebRequest.getParameter("_orders"));
                return page;
            }

            log.warn("构造PageRequest失败,不支持的请求方式:" + contentTypeStr);
        } catch (Exception var14) {
            log.error(var14.getMessage(), var14);
        }

        return new PageRequest();
    }
}