package com.ustcinfo.ishare.eip.admin.rest.sys.controller;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.ustcinfo.ishare.eip.admin.cache.common.IAdminCache;
import com.ustcinfo.ishare.eip.admin.cache.sys.SysUserTokenCache;
import com.ustcinfo.ishare.eip.admin.common.exception.EIPException;
import com.ustcinfo.ishare.eip.admin.common.validator.Assert;
import com.ustcinfo.ishare.eip.admin.common.validator.ValidatorUtils;
import com.ustcinfo.ishare.eip.admin.common.validator.group.AddGroup;
import com.ustcinfo.ishare.eip.admin.common.validator.group.UpdateGroup;
import com.ustcinfo.ishare.eip.admin.controller.AbstractController;
import com.ustcinfo.ishare.eip.admin.service.sys.annotation.SysLog;
import com.ustcinfo.ishare.eip.admin.service.sys.entity.SysRoleEntity;
import com.ustcinfo.ishare.eip.admin.service.sys.entity.SysUserEntity;
import com.ustcinfo.ishare.eip.admin.service.sys.entity.vo.SysUserTemplate;
import com.ustcinfo.ishare.eip.admin.service.sys.form.JsonResult;
import com.ustcinfo.ishare.eip.admin.service.sys.form.PasswordForm;
import com.ustcinfo.ishare.eip.admin.service.sys.form.SysRegForm;
import com.ustcinfo.ishare.eip.admin.service.sys.form.TokenForm;
import com.ustcinfo.ishare.eip.admin.service.sys.page.PageUtils;
import com.ustcinfo.ishare.eip.admin.service.sys.service.*;
import com.ustcinfo.ishare.eip.admin.service.sys.utils.ExcelUtils;
import io.swagger.annotations.*;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import springfox.documentation.annotations.ApiIgnore;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;
import java.util.Map;


/**
 * 系统用户
 */
@RestController
@RequestMapping("/sys/user")
@Api(tags = "管理员管理")
public class SysUserController extends AbstractController {
    @Autowired
    private SysUserService sysUserService;
    @Autowired
    private SysUserRoleService sysUserRoleService;
    @Autowired
    private SysRoleService roleService;
    @Autowired
    private SysCaptchaService sysCaptchaService;
    @Autowired
    private SysUserTokenService sysUserTokenService;
    @Autowired
    private IAdminCache adminCache;

    @GetMapping("/online/list")
    public JsonResult<List<SysUserTokenCache>> onlineList() {
        List<SysUserTokenCache> list = adminCache.getAll(SysUserTokenCache.class);
        return JsonResult.ok().put(list);
    }

    /**
     * 所有用户列表
     */
    @GetMapping("/list")
    @RequiresPermissions("sys:user:list")
    @ApiOperation(value = "列表", notes = "分页查询管理员列表", produces = MediaType.APPLICATION_JSON_VALUE)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "page", value = "页码", paramType = "query", required = true, dataType = "int"),
            @ApiImplicitParam(name = "limit", value = "每页条数", paramType = "query", required = true, dataType = "int"),
            @ApiImplicitParam(name = "sidx", value = "排序字段", paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "order", value = "排序方式，如：asc、desc", paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "searchKey", value = "查询条件(多个之间使用空格隔开)", paramType = "query", dataType = "string")
    })

    public JsonResult<PageUtils<SysUserEntity>> list(@ApiIgnore @RequestParam Map<String, Object> params) {
        logger.info("查询用户");
        PageUtils<SysUserEntity> page = sysUserService.queryPage(params);
        /**
         * 汉化角色
         */
        List<SysUserEntity> list = page.getList();
        if (list != null) {
            for (SysUserEntity user : list) {
                List<SysRoleEntity> roleList = roleService.queryRoleList(user.getUserId());
                if (roleList != null) {
                    List<String> nameList = Lists.newArrayList();
                    for (SysRoleEntity role : roleList) {
                        if (StringUtils.isNotBlank(role.getRoleName())) {
                            nameList.add(role.getRoleName());
                        }
                    }
                    user.setRoleName(Joiner.on(",").join(nameList));
                }
            }
        }
        return JsonResult.ok().put(page);
    }

    /**
     * 获取登录的用户信息
     */
    @ApiOperation(value = "当前登录用户", notes = "获取当前用户信息", produces = MediaType.APPLICATION_JSON_VALUE)
    @GetMapping("/info")
    public JsonResult<SysUserEntity> info() {
        return JsonResult.ok().put(getUser());
    }

    /**
     * 修改登录用户密码
     */
    @PostMapping("/password")
    @ApiOperation(value = "修改密码", notes = "用户密码修改", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<Void> password(@ApiParam(name = "修改密码对象", value = "传入json格式", required = true) @RequestBody PasswordForm form) {
        Assert.isBlank(form.getNewPassword(), "新密码不为能空");
        //更新密码
        boolean flag = sysUserService.updatePassword(getUserId(), form.getPassword(), form.getNewPassword());
        if (!flag) {
            return JsonResult.error("原密码不正确");
        }
        return JsonResult.ok();
    }

    /**
     * 用户信息
     */
    @GetMapping("/info/{userId}")
    @RequiresPermissions("sys:user:info")
    @ApiOperation(value = "用户详情", notes = "获取指定用户详细信息", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<SysUserEntity> info(@PathVariable("userId") String userId) {
        SysUserEntity user = sysUserService.getById(userId);
        //获取用户所属的角色列表
        List<String> roleIdList = sysUserRoleService.queryRoleIdList(userId);
        user.setRoleIdList(roleIdList);
        return JsonResult.ok().put(user);
    }

    /**
     * 保存用户
     */
    @PostMapping("/save")
    @RequiresPermissions("sys:user:save")
    @ApiOperation(value = "添加", notes = "添加用户", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<Void> save(@ApiParam(name = "添加用户对象", value = "传入json格式", required = true) @RequestBody SysUserEntity user) {
        validate(user, AddGroup.class);
        sysUserService.add(user);
        return JsonResult.ok();
    }

    /**
     * 修改用户
     */
    @SysLog("修改用户")
    @PostMapping("/update")
    @RequiresPermissions("sys:user:update")
    @ApiOperation(value = "修改", notes = "修改用户", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<Void> update(@ApiParam(name = "修改用户对象", value = "传入json格式", required = true) @RequestBody SysUserEntity user) {
        validate(user, UpdateGroup.class);
        sysUserService.update(user);
        return JsonResult.ok();
    }

    /**
     * 删除用户
     */
    @SysLog("删除用户")
    @PostMapping("/delete")
    @RequiresPermissions("sys:user:delete")
    @ApiOperation(value = "删除", notes = "删除用户", produces = MediaType.APPLICATION_JSON_VALUE)
    @ApiImplicitParam(name = "userIds", value = "用户ID", paramType = "body", required = true, dataType = "array")
    public JsonResult<Void> delete(@RequestBody String[] userIds) {
        if (ArrayUtils.contains(userIds, "1")) {
            return JsonResult.error("系统管理员不能删除");
        }

        if (ArrayUtils.contains(userIds, getUserId())) {
            return JsonResult.error("当前用户不能删除");
        }

        sysUserService.deleteBatch(userIds);
        return JsonResult.ok();
    }

    @SysLog("禁用用户")
    @PostMapping("/lock")
    @RequiresPermissions("sys:user:lock")
    @ApiOperation(value = "禁用", notes = "禁用用户", produces = MediaType.APPLICATION_JSON_VALUE)
    @ApiImplicitParam(name = "userIds", value = "用户ID", paramType = "body", required = true, dataType = "array")
    public JsonResult<Void> lock(@RequestBody String[] userIds) {
        if (ArrayUtils.contains(userIds, "1")) {
            return JsonResult.error("系统管理员不能禁用");
        }
        if (ArrayUtils.contains(userIds, getUserId())) {
            return JsonResult.error("当前用户不能禁用");
        }
        sysUserService.locks(userIds);
        return JsonResult.ok();
    }

    @SysLog("解锁用户")
    @PostMapping("/unlock")
    @RequiresPermissions("sys:user:unlock")
    @ApiOperation(value = "解锁", notes = "解锁用户", produces = MediaType.APPLICATION_JSON_VALUE)
    @ApiImplicitParam(name = "userIds", value = "用户ID", paramType = "body", required = true, dataType = "array")
    public JsonResult<Void> unlock(@RequestBody String[] userIds) {
        sysUserService.unlocks(userIds);
        return JsonResult.ok();
    }

    /**
     * 注册用户
     *
     * @param regForm
     * @return
     */
    @PostMapping("/reg")
    @ApiOperation(value = "注册", notes = "注册用户", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<TokenForm> reg(@ApiParam(name = "注册用户对象", value = "传入json格式", required = true) @RequestBody SysRegForm regForm) {
        // 验证 验证码
        boolean captcha = sysCaptchaService.validate(regForm.getUuid(), regForm.getCaptcha());
        if (!captcha) {
            return JsonResult.error("验证码不正确");
        }
        regForm.setStatus(1);
        // 设置默认的角色
        regForm.setRoleIdList(Lists.newArrayList("25"));
        sysUserService.add(regForm);
        JsonResult<TokenForm> r = sysUserTokenService.createToken(regForm.getUserId(), regForm.getUsername());
        return r;
    }


    @SysLog("导出用户模板")
    @RequestMapping("/export/template")
    @ApiOperation(value = "导出模板", notes = "导出模板", produces = MediaType.APPLICATION_JSON_VALUE)
    public void exportTemplate(HttpServletResponse response, HttpServletRequest request, ModelMap map) throws IOException, InterruptedException {
        List<SysUserTemplate> userList = Lists.newArrayList();
        SysUserTemplate user = new SysUserTemplate();
        user.setUsername("test");
        user.setEmail("18155164138@163.com");
        user.setMobile("18155164138");
        user.setPassword("000000");
        user.setStatus(1);
        user.setRoleName("普通用户1,普通用户2");
        userList.add(user);
        ExcelUtils.exportData(response, request, map, "用户信息模板" + System.currentTimeMillis(), "用户信息", userList, SysUserTemplate.class);
    }

    @RequestMapping("/export")
    @RequiresPermissions("sys:user:save")
    @ApiOperation(value = "导出", notes = "导出用户", produces = MediaType.APPLICATION_JSON_VALUE)
    public void export(HttpServletResponse response, HttpServletRequest request, ModelMap map) throws IOException, InterruptedException {
        logger.info("导出");
        List<SysUserEntity> userList = sysUserService.list(null);
        for (SysUserEntity user : userList) {
            List<SysRoleEntity> roList = roleService.queryRoleList(user.getUserId());
            List<String> roleNameList = Lists.newArrayList();
            for (SysRoleEntity role : roList) {
                roleNameList.add(role.getRoleName());
            }
            user.setRoleName(Joiner.on(",").join(roleNameList));
        }
        ExcelUtils.exportData(response, request, map, "用户信息" + System.currentTimeMillis(), "用户信息", userList, SysUserEntity.class);
    }

    @RequestMapping("/import")
    @RequiresPermissions("sys:user:save")
    @ApiOperation(value = "导入", notes = "导入用户", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<String> importUser(@RequestParam("file") MultipartFile file) throws Exception {
        List<SysUserTemplate> templateUserList = ExcelUtils.importData(file, SysUserTemplate.class);
        if (templateUserList == null || templateUserList.size() == 0) {
            throw new EIPException("导入失败,导出数据为空");
        }
        List<SysRoleEntity> roleList = roleService.list();
        Map<String, SysRoleEntity> roleMap = Maps.newHashMap();
        for (SysRoleEntity role : roleList) {
            roleMap.put(role.getRoleName(), role);
        }
        List<SysUserEntity> userList = Lists.newArrayList();
        // 先校验
        for (SysUserTemplate userTemplate : templateUserList) {
            SysUserEntity user = new SysUserEntity();
            BeanUtils.copyProperties(user, userTemplate);
            user.setRoleIdList(Lists.newArrayList());

            if (StringUtils.isEmpty(userTemplate.getUsername())) {
                throw new EIPException("导入失败,用户名不能为空");
            }
            if (StringUtils.isEmpty(userTemplate.getRoleName())) {
                throw new EIPException("角色不能为空");
            }
            Iterable<String> roleNameList = Splitter.on(",").split(userTemplate.getRoleName());
            for (String name : roleNameList) {
                if (roleMap.get(name) == null) {
                    throw new EIPException("不存在" + name + "角色");
                }
                SysRoleEntity role = roleMap.get(name);
                user.getRoleIdList().add(role.getRoleId());
            }
            if (StringUtils.isEmpty(user.getPassword())) {
                user.setPassword("000000");
            }
            userList.add(user);
        }
        sysUserService.importExcel(userList);
        return JsonResult.ok();
    }

    private void validate(SysUserEntity user, Class<?>... groups) {
        ValidatorUtils.validateEntity(user, groups);
        List<String> roleIdList = user.getRoleIdList();
        if (roleIdList == null || roleIdList.size() == 0) {
            throw new EIPException("请至少选择一个角色");
        }
    }
}
