package com.ustcinfo.ishare.eip.admin.rest.sys.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.ustcinfo.ishare.eip.admin.common.constant.MENU;
import com.ustcinfo.ishare.eip.admin.common.constant.MENU_TYPE;
import com.ustcinfo.ishare.eip.admin.common.exception.EIPException;
import com.ustcinfo.ishare.eip.admin.controller.AbstractController;
import com.ustcinfo.ishare.eip.admin.service.sys.annotation.SysLog;
import com.ustcinfo.ishare.eip.admin.service.sys.entity.SysMenuEntity;
import com.ustcinfo.ishare.eip.admin.service.sys.form.HomeMenuForm;
import com.ustcinfo.ishare.eip.admin.service.sys.form.JsonResult;
import com.ustcinfo.ishare.eip.admin.service.sys.form.MenuForm;
import com.ustcinfo.ishare.eip.admin.service.sys.service.ShiroService;
import com.ustcinfo.ishare.eip.admin.service.sys.service.SysMenuService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 系统菜单
 */
@RestController
@RequestMapping("/sys/menu")
@Api(tags = "菜单管理")
public class SysMenuController extends AbstractController {
    @Autowired
    private SysMenuService sysMenuService;
    @Autowired
    private ShiroService shiroService;

    /**
     * 导航菜单
     */
    @GetMapping("/nav")
    @ApiOperation(value = "导航栏列表", notes = "获取该用户能够访问的菜单列表", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<MenuForm> nav() {
        List<SysMenuEntity> menuList = sysMenuService.getUserMenuList(getUserId());
        /**
         * 被隐藏的菜单移除
         */
        sysMenuService.removeHideMenu(menuList);
        /**
         * 获取Home菜单
         */
        HomeMenuForm homeMenu = new HomeMenuForm();
        sysMenuService.getHomeMenu(menuList, homeMenu);
        Set<String> permissions = shiroService.getUserPermissions(getUserId());
        return JsonResult.ok().put(new MenuForm(homeMenu, menuList, permissions));
    }

    @GetMapping("/home")
    @ApiOperation(value = "Home导航菜单", notes = "Home导航菜单", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<HomeMenuForm> homeMenu() {
        List<SysMenuEntity> menuList = sysMenuService.getUserMenuList(getUserId());
        /**
         * 被隐藏的菜单移除
         */
        sysMenuService.removeHideMenu(menuList);
        /**
         * 获取Home菜单
         */
        HomeMenuForm homeMenu = new HomeMenuForm();
        sysMenuService.getHomeMenu(menuList, homeMenu);
        return JsonResult.ok().put(homeMenu);
    }

    /**
     * 所有菜单列表
     */
    @GetMapping("/list")
    @RequiresPermissions("sys:menu:list")
    @ApiOperation(value = "列表", notes = "返回菜单列表", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<List<SysMenuEntity>> list() {
        List<SysMenuEntity> menuList = sysMenuService.list(new QueryWrapper<SysMenuEntity>().orderByAsc("order_num"));
        for (SysMenuEntity sysMenuEntity : menuList) {
            SysMenuEntity parentMenuEntity = sysMenuService.getById(sysMenuEntity.getParentId());
            if (parentMenuEntity != null) {
                sysMenuEntity.setParentName(parentMenuEntity.getName());
            }
        }
        return JsonResult.ok().put(menuList);
    }

    /**
     * 所有菜单列表
     */
    @GetMapping("/tree")
    @ApiOperation(value = "列表tree", notes = "返回菜单列表", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<List<SysMenuEntity>> tree() {
        List<SysMenuEntity> menuList = sysMenuService.list(new QueryWrapper<SysMenuEntity>().orderByAsc("order_num"));
        List<SysMenuEntity> parentNodeList = Lists.newArrayList();
        Map<String, SysMenuEntity> menuMap = Maps.newHashMap();
        for (SysMenuEntity sysMenuEntity : menuList) {
            if (MENU.DEFAULT_PARENT_ID.equals(sysMenuEntity.getParentId())) {
                parentNodeList.add(sysMenuEntity);
            }
            menuMap.put(sysMenuEntity.getMenuId(), sysMenuEntity);
            SysMenuEntity parentMenuEntity = sysMenuService.getById(sysMenuEntity.getParentId());
            if (parentMenuEntity != null) {
                sysMenuEntity.setParentName(parentMenuEntity.getName());
            }
        }

        for (SysMenuEntity sysMenuEntity : menuList) {
            treeData(sysMenuEntity, menuMap);
        }
        return JsonResult.ok().put(parentNodeList);
    }


    public void treeData(SysMenuEntity menu, Map<String, SysMenuEntity> menuMap) {
        String parentId = menu.getParentId();
        if (MENU.DEFAULT_PARENT_ID.equals(parentId)) {
            return;
        }
        SysMenuEntity pMenu = menuMap.get(parentId);
        if (pMenu.getChildren() == null) {
            pMenu.setChildren(new ArrayList<SysMenuEntity>());
        }
        pMenu.getChildren().add(menu);
    }

    /**
     * 选择菜单(添加、修改菜单)
     */
    @GetMapping("/select")
    @RequiresPermissions("sys:menu:select")
    @ApiOperation(value = "选择菜单", notes = "添加、修改菜单的时候，选择上级菜单接口", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<List<SysMenuEntity>> select() {
        //查询列表数据
        List<SysMenuEntity> menuList = sysMenuService.queryNotButtonList();

        //添加顶级菜单
        SysMenuEntity root = new SysMenuEntity();
        root.setMenuId(MENU.DEFAULT_PARENT_ID);
        root.setName("一级菜单");
        root.setParentId("-1");
        menuList.add(root);

        return JsonResult.ok().put(menuList);
    }

    /**
     * 选择菜单(添加、修改菜单)
     */
    @GetMapping("/select/catalogue")
    @RequiresPermissions("sys:menu:select")
    @ApiOperation(value = "选择目录", notes = "添加、修改菜单/目录的时候，选择上级目录", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<List<SysMenuEntity>> selectCatalogue() {
        //查询列表数据
        List<SysMenuEntity> menuList = sysMenuService.queryCatalogueList();
        //添加顶级菜单
        SysMenuEntity root = new SysMenuEntity();
        root.setMenuId(MENU.DEFAULT_PARENT_ID);
        root.setName("一级目录");
        root.setParentId("-1");
        menuList.add(root);
        return JsonResult.ok().put(menuList);
    }

    /**
     * 菜单信息
     */
    @GetMapping("/info/{menuId}")
    @RequiresPermissions("sys:menu:info")
    @ApiOperation(value = "详情", notes = "获取指定菜单详细信息", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<SysMenuEntity> info(@PathVariable("menuId") Long menuId) {
        SysMenuEntity menu = sysMenuService.getById(menuId);
        return JsonResult.ok().put(menu);
    }

    /**
     * 保存
     */
    @SysLog("保存菜单")
    @PostMapping("/save")
    @RequiresPermissions("sys:menu:save")
    @ApiOperation(value = "添加", notes = "添加菜单", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<Void> save(@ApiParam(name = "添加菜单对象", value = "传入json格式", required = true) @RequestBody SysMenuEntity menu) {
        //数据校验
        verifyForm(menu);
        sysMenuService.add(menu);
        return JsonResult.ok();
    }

    /**
     * 修改
     */
    @SysLog("修改菜单")
    @PostMapping("/update")
    @RequiresPermissions("sys:menu:update")
    @ApiOperation(value = "修改", notes = "修改菜单", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<Void> update(@ApiParam(name = "修改参数对象", value = "传入json格式", required = true) @RequestBody SysMenuEntity menu) {
        //数据校验
        verifyForm(menu);
        sysMenuService.update(menu);
        return JsonResult.ok();
    }

    /**
     * 删除
     */
    @SysLog("删除菜单")
    @PostMapping("/delete/{menuId}")
    @RequiresPermissions("sys:menu:delete")
    @ApiOperation(value = "删除", notes = "删除菜单", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<Void> delete(@PathVariable("menuId") String menuId) {

        //判断是否有子菜单或按钮
        List<SysMenuEntity> menuList = sysMenuService.queryListParentId(menuId);
        if (menuList.size() > 0) {
            return JsonResult.error("请先删除子菜单或按钮");
        }

        sysMenuService.delete(menuId);

        return JsonResult.ok();
    }

    /**
     * 验证参数是否正确
     */
    private void verifyForm(SysMenuEntity menu) {
        if (menu.getType() == null) {
            throw new EIPException("类型不能为空");
        }

        if (StringUtils.isBlank(menu.getName())) {
            throw new EIPException("名称不能为空");
        }

        if (StringUtils.isBlank(menu.getParentId())) {
            throw new EIPException("上级菜单不能为空");
        }

        int type = menu.getType().intValue();
        //菜单
        if (type == MENU_TYPE.MENU) {
            if (StringUtils.isBlank(menu.getUrl())) {
                throw new EIPException("菜单URL不能为空");
            }
        }

        //上级菜单类型
        int parentType = MENU_TYPE.CATALOG;
        if (!"0".equals(menu.getParentId())) {
            SysMenuEntity parentMenu = sysMenuService.getById(menu.getParentId());
            parentType = parentMenu.getType();
        }

        //目录、菜单
        if (type == MENU_TYPE.CATALOG || type == MENU_TYPE.MENU) {
            if (parentType != MENU_TYPE.CATALOG) {
                throw new EIPException("上级菜单只能为目录类型");
            }
            return;
        }

        //按钮
        if (type == MENU_TYPE.BUTTON) {
            if (parentType != MENU_TYPE.MENU) {
                throw new EIPException("上级菜单只能为菜单类型");
            }
            return;
        }
    }
}
