package com.ustcinfo.ishare.eip.admin.rest.sys.controller;

import cn.hutool.core.codec.Base64;
import com.ustcinfo.ishare.eip.admin.cache.common.IAdminCache;
import com.ustcinfo.ishare.eip.admin.cache.sys.SysUserTokenCache;
import com.ustcinfo.ishare.eip.admin.common.constant.SysLogType;
import com.ustcinfo.ishare.eip.admin.controller.AbstractController;
import com.ustcinfo.ishare.eip.admin.service.sys.annotation.SysLog;
import com.ustcinfo.ishare.eip.admin.service.sys.entity.SysUserEntity;
import com.ustcinfo.ishare.eip.admin.service.sys.form.JsonResult;
import com.ustcinfo.ishare.eip.admin.service.sys.form.SysLoginForm;
import com.ustcinfo.ishare.eip.admin.service.sys.form.TokenForm;
import com.ustcinfo.ishare.eip.admin.service.sys.service.SysCaptchaService;
import com.ustcinfo.ishare.eip.admin.service.sys.service.SysUserService;
import com.ustcinfo.ishare.eip.admin.service.sys.service.SysUserTokenService;
import com.ustcinfo.ishare.eip.admin.service.sys.utils.UserUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.io.IOUtils;
import org.apache.shiro.crypto.hash.Sha256Hash;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;

import javax.imageio.ImageIO;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.awt.image.BufferedImage;
import java.io.IOException;

/**
 * 登录相关
 */
@RestController
@Api(tags = "登录")
public class SysLoginController extends AbstractController {
    @Autowired
    private SysUserService sysUserService;
    @Autowired
    private SysCaptchaService sysCaptchaService;
    @Autowired
    private SysUserTokenService sysUserTokenService;
    @Autowired
    private IAdminCache adminCache;

    /**
     *
     * 验证码开关
     */
    @Value("${captcha.check.isOn:true}")
    private Boolean captchaCheck;

    /**
     * 验证码
     */
    @GetMapping("captcha.jpg")
    @ApiOperation(value = "获取验证码", notes = "获取验证码", produces = MediaType.IMAGE_JPEG_VALUE)
    public void captcha(HttpServletResponse response, String uuid) throws ServletException, IOException {
        if(!captchaCheck){
            return;
        }
        response.setHeader("Cache-Control", "no-store, n o-cache");
        response.setContentType("image/jpeg");

        //获取图片验证码
        BufferedImage image = sysCaptchaService.getCaptcha(uuid);

        ServletOutputStream out = response.getOutputStream();
        ImageIO.write(image, "jpg", out);
        IOUtils.closeQuietly(out);
    }

    /**
     * 登录
     */
    @SysLog(value = "系统登录", type = SysLogType.LOGIN)
    @PostMapping("/sys/login")
    @ApiOperation(value = "登录", notes = "登录", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<TokenForm> login(@ApiParam(name = "登录对象", value = "传入json格式", required = true) @RequestBody SysLoginForm form) throws IOException {
       if(captchaCheck){
           boolean captcha = sysCaptchaService.validate(form.getUuid(), form.getCaptcha());
           if (!captcha) {
               return JsonResult.error("验证码不正确");
           }
       }

        //用户信息
        SysUserEntity user = sysUserService.queryByCount(form.getUsername());

        //账号不存在、密码错误
        if (user == null || !user.getPassword().equals(form.getPassword())) {
            return JsonResult.error("账号或密码不正确");
        }

        if (user.getStatus() == null) {
            return JsonResult.error("账号状态异常");
        }
        //账号锁定
        if (user.getStatus().equals(SysUserEntity.LOCKED)) {
            return JsonResult.error("账号已被锁定,请联系管理员");
        }
        //生成token，并保存到数据库
        JsonResult<TokenForm> r = sysUserTokenService.createToken(user.getUserId(), form.getUsername());
        return r;
    }

    /**
     * 退出
     */
    @DeleteMapping("/sys/logout/{token}")
    @ApiOperation(value = "退出", notes = "退出", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<Void> logout(@PathVariable("token") String token, HttpServletResponse response, HttpServletRequest request) {
        /**
         * 只有超级管理员才能够退出其他人
         */
        SysUserTokenCache tokenCache = adminCache.get(SysUserTokenCache.class, token);
        if (tokenCache != null) {
            String userId = getUserId();
            if (UserUtils.isAdmin(userId)) {
                sysUserTokenService.logout(token);
                return JsonResult.ok();
            }
            if (!tokenCache.getUserId().equals(userId)) {
                return JsonResult.error("您没有权限下线其他用户");
            }
            sysUserTokenService.logout(token);
        }
        return JsonResult.ok();
    }

}
