package com.ustcinfo.ishare.eip.admin.rest.sys.controller;


import com.ustcinfo.ishare.eip.admin.common.exception.EIPException;
import com.ustcinfo.ishare.eip.admin.common.utils.MapUtils;
import com.ustcinfo.ishare.eip.admin.common.validator.ValidatorUtils;
import com.ustcinfo.ishare.eip.admin.controller.AbstractController;
import com.ustcinfo.ishare.eip.admin.service.sys.annotation.SysLog;
import com.ustcinfo.ishare.eip.admin.service.sys.entity.SysDictEntity;
import com.ustcinfo.ishare.eip.admin.service.sys.form.JsonResult;
import com.ustcinfo.ishare.eip.admin.service.sys.form.SysDictForm;
import com.ustcinfo.ishare.eip.admin.service.sys.page.PageUtils;
import com.ustcinfo.ishare.eip.admin.service.sys.service.SysDictService;
import io.swagger.annotations.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;
import springfox.documentation.annotations.ApiIgnore;

import java.util.Map;

/**
 * 业务字典
 */
@RestController
@RequestMapping("/sys/dict")
@Api(tags = "业务字典")
public class SysDictController extends AbstractController {

    @Autowired
    private SysDictService dictService;

    @GetMapping("/list")
    @RequiresPermissions("sys:dict:list")
    @ApiOperation(value = "列表", notes = "分页查询业务字典列表", produces = MediaType.APPLICATION_JSON_VALUE)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "page", value = "页码", paramType = "query", required = true, dataType = "int"),
            @ApiImplicitParam(name = "limit", value = "每页条数", paramType = "query", required = true, dataType = "int"),
            @ApiImplicitParam(name = "sidx", value = "排序字段", paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "order", value = "排序方式，如：asc、desc", paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "dictTypeId", value = "业务字典ID", paramType = "query", dataType = "int"),
            @ApiImplicitParam(name = "searchKey", value = "查询条件", paramType = "query", dataType = "string")
    })
    public JsonResult<PageUtils<SysDictEntity>> list(@ApiIgnore @RequestParam Map<String, Object> params) {
        PageUtils<SysDictEntity> page = dictService.queryPage(MapUtils.getInstance(params));
        return JsonResult.ok().put(page);
    }

    @GetMapping("/info/{id}")
    @RequiresPermissions("sys:dict:info")
    @ApiOperation(value = "详情", notes = "获取指定ID的详细信息", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<SysDictEntity> info(@PathVariable("id") String id) {
        SysDictEntity dictEntity = dictService.getById(id);
        return JsonResult.ok().put(dictEntity);
    }

    @SysLog("保存业务字典")
    @PostMapping("/save")
    @RequiresPermissions("sys:dict:save")
    @ApiOperation(value = "添加", notes = "添加业务字典信息", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<Void> save(@ApiParam(name = "添加业务字典对象", value = "传入json格式", required = true) @RequestBody SysDictForm dict) {
        validate(dict);
        dictService.add(dict);
        return JsonResult.ok();
    }

    @SysLog("修改业务字典")
    @PostMapping("/update")
    @RequiresPermissions("sys:dict:update")
    @ApiOperation(value = "修改", notes = "修改业务字典信息", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<Void> update(@ApiParam(name = "修改业务字典类型对象", value = "传入json格式", required = true) @RequestBody SysDictForm dict) {
        validate(dict);
        dictService.update(dict);
        return JsonResult.ok();
    }

    @SysLog("删除业务字典")
    @PostMapping("/delete")
    @RequiresPermissions("sys:dict:delete")
    @ApiOperation(value = "删除", notes = "删除业务字典对象", produces = MediaType.APPLICATION_JSON_VALUE)
    public JsonResult<Void> delete(@RequestBody String[] ids) {
        dictService.deleteBatch(ids);
        return JsonResult.ok();
    }

    private void validate(SysDictForm dictEntity) {
        ValidatorUtils.validateEntity(dictEntity);
        if (StringUtils.isBlank(dictEntity.getDictTypeId()) && StringUtils.isBlank(dictEntity.getDictType())) {
            throw new EIPException("字典类型不能为空");
        }
    }
}
