package com.ustcinfo.ishare.eip.admin;

import com.baomidou.mybatisplus.core.parser.ISqlParser;
import com.baomidou.mybatisplus.core.parser.ISqlParserFilter;
import com.baomidou.mybatisplus.core.parser.SqlParserHelper;
import com.baomidou.mybatisplus.extension.plugins.PaginationInterceptor;
import com.baomidou.mybatisplus.extension.plugins.tenant.TenantHandler;
import com.baomidou.mybatisplus.extension.plugins.tenant.TenantSqlParser;
import com.google.common.collect.Maps;
import com.ustcinfo.ishare.eip.admin.cache.common.constant.CacheConstant;
import com.ustcinfo.ishare.eip.admin.cache.common.IAdminCache;
import com.ustcinfo.ishare.eip.admin.service.sys.datasources.DynamicDataSourceConfig;
import com.ustcinfo.ishare.eip.admin.service.sys.entity.SysUserEntity;
import com.ustcinfo.ishare.eip.admin.service.sys.extend.AdminExtendService;
import lombok.extern.slf4j.Slf4j;
import net.sf.jsqlparser.expression.Expression;
import net.sf.jsqlparser.expression.LongValue;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.reflection.MetaObject;
import org.apache.shiro.SecurityUtils;
import org.apache.shiro.subject.Subject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration;
import org.springframework.boot.builder.SpringApplicationBuilder;
import org.springframework.boot.web.servlet.support.SpringBootServletInitializer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.scheduling.annotation.EnableScheduling;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 如果Application类所在的包为：io.github.gefangshuai.app，
 * 则只会扫描io.github.gefangshuai.app包及其所有子包，
 * 如果service或dao所在包不在io.github.gefangshuai.app及其子包下，则不会被扫描
 */
@SpringBootApplication(exclude = {DataSourceAutoConfiguration.class})
@Import({DynamicDataSourceConfig.class})
@EnableScheduling
@Slf4j
public class AdminRestApplication extends SpringBootServletInitializer {

    public static void main(String[] args) {
        SpringApplication.run(AdminRestApplication.class, args);
    }

    @Override
    protected SpringApplicationBuilder configure(SpringApplicationBuilder application) {
        return application.sources(AdminRestApplication.class);
    }

    @Autowired
    private IAdminCache adminCache;

    @Bean
    @ConditionalOnProperty(prefix = "admin.cache", name = "init-main-class", havingValue = "com.ustcinfo.ishare.eip.admin.AdminRestApplication")
    public String init() {
        log.info("启动加载缓存");
        ArrayList<String> envList = new ArrayList<>();
        envList.add(CacheConstant.DEFAULT_TENANT);
        adminCache.load(envList);
        adminCache.reSubscribe(envList);
        return "AdminRestApplicationInit";
    }

    @Bean
    public AdminExtendService adminExtendService() {
        AdminExtendService adminExtendService = new AdminExtendService();

        /**
         * 静态资源处理
         */
        Map<String, String> resourceHandlerRegistryMap = Maps.newHashMap();
        /**为了swagger-bootstrap-ui*/
        resourceHandlerRegistryMap.put("doc.html", "classpath:/META-INF/resources/");
        resourceHandlerRegistryMap.put("/webjars/**", "classpath:/META-INF/resources/webjars/");
        adminExtendService.setResourceHandlerRegistryMap(resourceHandlerRegistryMap);
        adminExtendService.getShiroFilterChainDefinitionMap().put("/static/**", "anon");
        /**
         * 设置自定义分页
         */
        //adminExtendService.setPaginationInterceptor(getPaginationInterceptor());

        return adminExtendService;
    }

    public PaginationInterceptor getPaginationInterceptor() {
        PaginationInterceptor paginationInterceptor = new PaginationInterceptor();
        /*
         * 【测试多租户】 SQL 解析处理拦截器<br>
         * 这里固定写成住户 1 实际情况你可以从cookie读取，因此数据看不到 【 麻花藤 】 这条记录（ 注意观察 SQL ）<br>
         */
        List<ISqlParser> sqlParserList = new ArrayList<>();
        TenantSqlParser tenantSqlParser = new TenantSqlParser();
        tenantSqlParser.setTenantHandler(new TenantHandler() {
            @Override
            public Expression getTenantId() {

                try {
                    Subject subject = SecurityUtils.getSubject();
                    SysUserEntity user = (SysUserEntity) subject.getPrincipal();
                    if (user == null) {
                        // 未登录
                        return new LongValue(-1);
                    }
                    /**
                     * 在这里根据当前登录的用户找到他所在的租户ID
                     */
                    return new LongValue(user.getUserId());
                } catch (Exception e) {
                    log.error(ExceptionUtils.getStackTrace(e));
                }
                return new LongValue(-1);
            }

            @Override
            public String getTenantIdColumn() {
                /**
                 * 标识多租户的字段名称
                 */
                return "create_user_id";
            }

            @Override
            public boolean doTableFilter(String tableName) {
                /**
                 * 这里可以判断是否过滤表
                 * false: 会走多租户
                 * true: 不走多租户id自动注入
                 */
                if ("schedule_job_log".equals(tableName)) {
                    return false;
                }
                return true;
            }
        });
        sqlParserList.add(tenantSqlParser);
        paginationInterceptor.setSqlParserList(sqlParserList);
        paginationInterceptor.setSqlParserFilter(new ISqlParserFilter() {
            @Override
            public boolean doFilter(MetaObject metaObject) {
                MappedStatement ms = SqlParserHelper.getMappedStatement(metaObject);
                /**
                 * 过滤自定义查询此时无租户信息
                 * true: 该方法不自动注入多租户查询
                 * false: 默认走多租户ID自动注入
                 */
                if ("com.baomidou.springboot.mapper.UserMapper.selectListBySQL".equals(ms.getId())) {
                    return true;
                }

                //if ("com.ustcinfo.ishare.eip.admin.quartz.mapper.ScheduleJobLogMapper.selectPage".equals(ms.getId())) {
                //    return true;
                //}
                return false;
            }
        });
        return paginationInterceptor;
    }

}
