package com.ustcinfo.ishare.eip.admin.cache.common;


import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.ustcinfo.ishare.eip.admin.cache.common.annotation.AdminCacheKey;
import com.ustcinfo.ishare.eip.admin.cache.common.annotation.AdminCacheTenantId;
import com.ustcinfo.ishare.eip.admin.cache.common.callback.AdminCacheCallBackContext;
import com.ustcinfo.ishare.eip.admin.common.exception.EIPException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

import static com.ustcinfo.ishare.eip.admin.cache.common.constant.CacheConstant.DEFAULT_TENANT;

/**
 * 内置缓存
 *
 * @author liuchengbiao
 * @date 2019/4/22 下午2:16
 */
@Slf4j
public class Cache {
    /**
     * 内部缓存，支持多租户
     * key： 环境唯一标识
     * vale: Map<String, Map<String, AdminCacheable>>
     * ------key: 资源类型，使用缓存类的 getClass().getSimpleName() 作为资源类型
     * ------value: Map<String, AdminCacheable>
     * ------------key: 资源缓存的唯一标识
     * ------------value: 缓存的资源
     */
    private static final Map<Serializable, Map<Serializable, Map<Serializable, AdminCacheable>>> tenantCache = new HashMap<Serializable, Map<Serializable, Map<Serializable, AdminCacheable>>>();
    /**
     * 锁
     */
    private static ReentrantLock lock = new ReentrantLock();
    /**
     * 缓存注解字段
     * key: 类名
     * value:
     * --------key: 标识注解类名
     * --------value: 该注解对应的字段、可以通过操作该字段来获取属性值
     */
    private static final Map<Class, Map<Class, Field>> reflectFieldMap = Maps.newHashMap();

    /**
     * @param t
     * @return
     * @throws IllegalAccessException
     */
    public static AdminCacheField reflectCacheField(AdminCacheable t) {
        AdminCacheField cacheField = new AdminCacheField();
        Map<Class, Field> fieldMap = reflectFieldMap.get(t.getClass());
        if (fieldMap == null) {
            fieldMap = Maps.newHashMap();
            Class clazz = t.getClass();
            while (clazz != null) {
                if ("java.lang.Object".equals(clazz.getName())) {
                    break;
                }
                Field[] declaredFields = clazz.getDeclaredFields();
                for (Field field : declaredFields) {
                    field.setAccessible(true);
                    // 解析缓存的key
                    AdminCacheKey cacheKey = field.getAnnotation(AdminCacheKey.class);
                    if (cacheKey != null) {
                        fieldMap.put(AdminCacheKey.class, field);
                        try {
                            cacheField.setKey((Serializable) field.get(t));
                        } catch (IllegalAccessException e) {
                            log.error(ExceptionUtils.getStackTrace(e));
                        }
                        continue;
                    }
                    // 解析租户ID
                    AdminCacheTenantId cacheTenantId = field.getAnnotation(AdminCacheTenantId.class);
                    if (cacheTenantId != null) {
                        fieldMap.put(AdminCacheTenantId.class, field);
                        try {
                            cacheField.setTenantId((Serializable) field.get(t));
                        } catch (IllegalAccessException e) {
                            log.error(ExceptionUtils.getStackTrace(e));
                        }
                    }
                }
                clazz = clazz.getSuperclass();
            }
            reflectFieldMap.put(t.getClass(), fieldMap);
        } else {
            // 获取缓存的key
            Field cacheKeyField = fieldMap.get(AdminCacheKey.class);
            if (cacheKeyField != null) {
                try {
                    cacheField.setKey((Serializable) cacheKeyField.get(t));
                } catch (IllegalAccessException e) {
                    log.error(ExceptionUtils.getStackTrace(e));
                }
            }
            // 获取租户ID
            Field cacheTenantIdField = fieldMap.get(AdminCacheTenantId.class);
            if (cacheTenantIdField != null) {
                try {
                    cacheField.setTenantId((Serializable) cacheTenantIdField.get(t));
                } catch (IllegalAccessException e) {
                    log.error(ExceptionUtils.getStackTrace(e));
                }
            }
        }
        Field cacheKeyField = fieldMap.get(AdminCacheKey.class);
        if (cacheKeyField == null) {
            throw new EIPException("必须在字段上设置@AdminCacheKey注解");
        }
        return cacheField;
    }

    /**
     * 获取缓存
     *
     * @return
     */
    public static Map<Serializable, Map<Serializable, Map<Serializable, AdminCacheable>>> getData() {
        return tenantCache;
    }

    public static AdminCacheField put(AdminCacheable entity) {
        AdminCacheField cacheField = reflectCacheField(entity);
        /**
         * 获取对应环境的数据
         */
        Map<Serializable, Map<Serializable, AdminCacheable>> cacheMap = tenantCache.get(cacheField.getTenantId());
        if (cacheMap == null) {
            try {
                lock.lock();
                cacheMap = tenantCache.get(cacheField.getTenantId());
                if (cacheMap == null) {
                    cacheMap = new HashMap<>();
                    tenantCache.put(cacheField.getTenantId(), cacheMap);
                }
            } catch (Exception e) {
                log.error(ExceptionUtils.getStackTrace(e));
                return cacheField;
            } finally {
                lock.unlock();
            }
        }

        /**
         * 获取资源类型
         */
        String simpleClassName = entity.getClass().getSimpleName();
        Map<Serializable, AdminCacheable> map = cacheMap.get(simpleClassName);
        if (map == null) {
            try {
                lock.lock();
                map = cacheMap.get(simpleClassName);
                if (map == null) {
                    map = new HashMap<Serializable, AdminCacheable>(16);
                    cacheMap.put(simpleClassName, map);
                }
            } catch (Exception e) {
                log.error(ExceptionUtils.getStackTrace(e));
                return cacheField;
            } finally {
                lock.unlock();
            }
        }
        /**
         * 是否继续放入内置缓存中、请确保你知道你的处理逻辑
         * 这里可以在回调中处理其他事情、保证从始至终缓存中的对象地址不变
         */
        boolean continuePut = AdminCacheCallBackContext.putCallBack(entity);
        if (continuePut) {
            map.put(cacheField.getKey(), entity);
        }
        return cacheField;
    }

    public static AdminCacheField put(AdminTimeOutCacheable entity, int timeOut, TimeUnit unit) {
        entity.setExpireTimestamp(System.currentTimeMillis() + unit.toMillis(timeOut));
        return put(entity);
    }

    public static void remove(Class<?> clazz, Serializable key) {
        remove(clazz, DEFAULT_TENANT, key);
    }

    public static void remove(Class<?> clazz, Serializable tenant, Serializable key) {
        //先回调、防止后续删除后资源找不到了
        AdminCacheCallBackContext.removeCallBack(clazz, tenant, key);
        Map<Serializable, Map<Serializable, AdminCacheable>> cacheMap = tenantCache.get(tenant);
        if (cacheMap != null) {
            String type = clazz.getSimpleName();
            Map<Serializable, AdminCacheable> map = cacheMap.get(type);
            if (map != null) {
                map.remove(key);
            }
        }
    }

    public static void removeAll(Class<?> clazz) {
        removeAll(clazz, DEFAULT_TENANT);
    }

    public static void removeAll(Class<?> clazz, Serializable tenant) {
        Map<Serializable, Map<Serializable, AdminCacheable>> cacheMap = tenantCache.get(tenant);
        if (cacheMap != null) {
            String type = clazz.getSimpleName();
            Map<Serializable, AdminCacheable> map = cacheMap.get(type);
            if (map != null) {
                /**
                 * 使用迭代器删除想
                 */
                Iterator<Map.Entry<Serializable, AdminCacheable>> it = map.entrySet().iterator();
                while (it.hasNext()) {
                    it.remove();
                }
            }
        }
    }

    /**
     * 获取指定缓存类型的所有key
     *
     * @param clazz
     * @param tenant
     * @return
     */
    public static Set<Serializable> getKeySet(Class clazz, Serializable tenant) {
        Set<Serializable> keySet = Sets.newHashSet();
        Map<Serializable, Map<Serializable, AdminCacheable>> cacheMap = tenantCache.get(tenant);
        if (cacheMap != null) {
            String type = clazz.getSimpleName();
            Map<Serializable, AdminCacheable> map = cacheMap.get(type);
            if (map != null) {
                return map.keySet();
            }
        }
        return keySet;
    }

    public  static Set<Serializable> getOriginalKeySet(Class clazz, Serializable tenant) {
        Set<Serializable> keySet = Sets.newHashSet();
        Map<Serializable, Map<Serializable, AdminCacheable>> cacheMap = tenantCache.get(tenant);
        if (cacheMap != null) {
            String type = clazz.getSimpleName();
            Map<Serializable, AdminCacheable> map = cacheMap.get(type);
            if (map != null) {
                Set<Serializable> keySets = map.keySet();
                Iterator<Serializable> it = keySets.iterator();
                while (it.hasNext()) {
                    Serializable key = it.next();
                    keySet.add(key.toString().split(":")[2]);
                }
            }
        }
        return keySet;
    }

    public static <T> T get(Class<T> clazz, Serializable key) {
        return get(clazz, DEFAULT_TENANT, key);
    }

    public static <T> T get(Class<T> clazz, Serializable tenant, Serializable key) {
        Map<Serializable, Map<Serializable, AdminCacheable>> cacheMap = tenantCache.get(tenant);
        if (cacheMap != null) {
            String type = clazz.getSimpleName();
            Map<Serializable, AdminCacheable> map = cacheMap.get(type);
            if (map != null) {
                AdminCacheable adminCacheable = map.get(key);
                if (adminCacheable == null) {
                    return null;
                }
                if (isExpire(adminCacheable)) {
                    map.remove(key);
                    return null;
                }
                return (T) adminCacheable;
            }
        }
        return null;
    }

    public static <T> List<T> getAll(Class<T> clazz) {
        return getAll(clazz, DEFAULT_TENANT);
    }

    public static <T> List<T> getAll(Class<T> clazz, Serializable tenant) {
        Map<Serializable, Map<Serializable, AdminCacheable>> cacheMap = tenantCache.get(tenant);
        if (cacheMap != null) {
            String type = clazz.getSimpleName();
            Map<Serializable, AdminCacheable> map = cacheMap.get(type);
            if (map != null) {
                List<AdminCacheable> list = new ArrayList<>(map.size());
                Iterator<Map.Entry<Serializable, AdminCacheable>> it = map.entrySet().iterator();
                while (it.hasNext()) {
                    Map.Entry<Serializable, AdminCacheable> next = it.next();
                    AdminCacheable adminCacheable = next.getValue();
                    if (adminCacheable == null) {
                        it.remove();
                        continue;
                    }
                    if (isExpire(adminCacheable)) {
                        it.remove();
                        continue;
                    }
                    list.add(adminCacheable);
                }
                return (List<T>) list;
            }
        }
        return null;
    }

    /**
     * 判断是否到期
     *
     * @param t
     * @return
     */
    public static boolean isExpire(AdminCacheable t) {
        if (t instanceof AdminTimeOutCacheable) {
            Long expireTimestamp = ((AdminTimeOutCacheable) t).getExpireTimestamp();
            if (expireTimestamp != null) {
                long expire = expireTimestamp.longValue();
                if (System.currentTimeMillis() > expire) {
                    return true;
                }
            }
        }
        return false;
    }
}
