package com.kdgcsoft.power.fileconverter.impl;

import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import javax.imageio.ImageIO;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.icepdf.core.pobjects.Document;
import org.icepdf.core.pobjects.Page;
import org.icepdf.core.util.GraphicsRenderingHints;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.fileconverter.FileConverterException;
import com.kdgcsoft.power.fileconverter.FileConverterSettings;
import com.kdgcsoft.power.fileconverter.IFileConverter;


/**
 *  PDF文档转为图片
 * 
 * @author arthur
 */
public class Pdf2ImgConverter implements IFileConverter {

	private static Logger logger = LoggerFactory.getLogger(Pdf2ImgConverter.class);
	/**
	 * pdf转图片
	 * @param pdfFile pdf文件路径
	 * @param imgFilePath 图片的全路径，包含图片文件名
	 * @param format 图片格式。如果为空，则使用"png"。
	 * @return 文档页数
	 */
	public static int toImg(final String pdfFile, final String imgFilePath, final String format) throws FileConverterException {
		logger.info("开始转换{}到{}格式...", pdfFile, format);

		if (!pdfFile.toLowerCase(Locale.ENGLISH).endsWith(".pdf")) {
			throw new FileConverterException("请检查参数pdfName文件名是否以.pdf结尾！ ");
		}
		
		Document document = new Document();

		try {
			document.setFile(pdfFile);
		} catch (Exception e) {
			logger.error("无法转换该文件{}", pdfFile, e);
			throw new FileConverterException("无法转换文件" + pdfFile + ", " + e.getMessage());
		}
		
		String picFormat = "png";
		if (format != null) {
			picFormat = format;
		}

		float scale = 1.3f;
		float rotation = 0f;
		
		File saveDir = new File(FilenameUtils.getFullPath(imgFilePath));
		String baseImageName = FilenameUtils.getBaseName(imgFilePath);
		try {
			FileUtils.forceMkdir(saveDir);
		} catch (IOException e1) {
			throw new FileConverterException("创建图像文件目录失败：" + saveDir.getAbsolutePath(), e1);
		}
		
		int pages = document.getNumberOfPages();

		for (int i = 0; i < pages; i++) {
			BufferedImage image = (BufferedImage) document.getPageImage(i, GraphicsRenderingHints.SCREEN,
					Page.BOUNDARY_CROPBOX, rotation, scale);
			RenderedImage rendImage = image;
			try {
				String newFileName = baseImageName + "." + i + "." + picFormat;
				File pageFile = new File(saveDir, newFileName);
				FileUtils.deleteQuietly(pageFile);
				ImageIO.write(rendImage, picFormat, pageFile);
			} catch (IOException e) {
				logger.error("第{}页转换失败", i, e);
			}
			image.flush();
		}
		document.dispose();
		
		logger.info("转换图片成功，共{}页", pages);
		
		return pages;
	}

	/**
	 * PDF转为PNG
	 * @param pdfPath pdf文件路径
	 * @param imgFilePath 图片的全路径，包含图片将要输出时的文件名
	 * @return 文档页数
	 * @throws FileConverterException 
	 */
	public static int toPng(final String pdfPath, final String imgFilePath) throws FileConverterException {
		return toImg(pdfPath, imgFilePath, "png");
	}
	
	/**
	 * PDF转为图片，在PDF所在目录下直接生成。
	 * @param pdfPath pdf文件路径
	 * @return 文档页数
	 * @throws FileConverterException 
	 */
	public static int toImg(final String pdfPath, String format) throws FileConverterException {
		return toImg(pdfPath, FilenameUtils.removeExtension(pdfPath), format);
	}

	/**
	 * PDF转为PNG，在PDF所在目录下直接生成。
	 * @param pdfPath pdf文件路径
	 * @return 文档页数
	 * @throws FileConverterException 
	 */
	public static int toPng(final String pdfPath) throws FileConverterException {
		return toImg(pdfPath, FilenameUtils.removeExtension(pdfPath), "png");
	}
	
	/**
	 * pdf转单个图片
	 * @param pdfFile pdf文件路径
	 * @param imgFilePath 图片的全路径，包含图片将要输出时的文件名
	 * @param format 图片格式，例如"png"。如果为空，则使用PNG格式
	 * @throws Exception
	 */
	public static void toSingleImg(final String pdfFile, final String imgFilePath, final String format) throws FileConverterException {
		logger.info("开始转换{}到单文件{}格式...", pdfFile, format);
		
		if (!pdfFile.toLowerCase(Locale.ENGLISH).endsWith(".pdf")) {
			throw new FileConverterException("请检查参数pdfName文件名是否以.pdf结尾！");
		}
		
		String picFormat = "png";
		if (format != null) {
			picFormat = format;
		}
		
		Document document = new Document();
		try {
			document.setFile(pdfFile);
		} catch (Exception e) {
			logger.error("无法转换该文件{}", pdfFile, e);
			throw new FileConverterException("无法转换文件" + pdfFile + ", " + e.getMessage());
		}
		float scale = 2f;
		float rotation = 0f;
		List<BufferedImage> bufferImgList = new ArrayList<BufferedImage>();
		for (int i = 0; i < document.getNumberOfPages(); i++) {
			BufferedImage image = (BufferedImage) document.getPageImage(i, GraphicsRenderingHints.SCREEN,
					org.icepdf.core.pobjects.Page.BOUNDARY_CROPBOX, rotation, scale);
			bufferImgList.add(image);
		}
		document.dispose();
		
		File saveDir = new File(FilenameUtils.getFullPath(imgFilePath));
		String baseImageName = FilenameUtils.getBaseName(imgFilePath);

		String imgName = baseImageName + "." + picFormat;
		int height = 0; // 总高度
		int width = 0; // 总宽度
		int _height = 0; // 临时的高度 , 或保存偏移高度
		int __height = 0; // 临时的高度，主要保存每个高度
		int picNum = bufferImgList.size();
		int[] heightArray = new int[picNum]; // 保存每个文件的高度
		BufferedImage buffer = null; // 保存图片流
		List<int[]> imgRGB = new ArrayList<int[]>(); // 保存所有的图片的RGB
		int[] _imgRGB; // 保存一张图片中的RGB数据
		for (int i = 0; i < picNum; i++) {
			buffer = bufferImgList.get(i);
			heightArray[i] = buffer.getHeight();// 图片高度
			_height = buffer.getHeight();
			if (i == 0) {
				width = buffer.getWidth();// 图片宽度
			}
			height += _height; // 获取总高度
			_imgRGB = new int[width * _height];// 从图片中读取RGB
			_imgRGB = buffer.getRGB(0, 0, width, _height, _imgRGB, 0, width);
			imgRGB.add(_imgRGB);
		}
		_height = 0; // 设置偏移高度为0
		// 生成新图片
		BufferedImage imageResult = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
		for (int i = 0; i < picNum; i++) {
			__height = heightArray[i];
			if (i != 0) {
				_height += __height; // 计算偏移高度
			}
			imageResult.setRGB(0, _height, width, __height, imgRGB.get(i), 0, width); // 写入流中
		}
		File outFile = new File(saveDir, imgName);
		
		try {
			FileUtils.deleteQuietly(outFile);
			ImageIO.write(imageResult, picFormat, outFile);
		} catch (IOException e) {
			logger.error("输出图片文件时发生异常：", pdfFile, e);
			throw new FileConverterException("输出图片文件时发生异常:" + pdfFile + ", " + e.getMessage());
		}
		
		imageResult.flush();
	}

	/**
	 * pdf转单个PNG图片
	 * @param pdfFile pdf文件路径
	 * @param imgFilePath 图片的全路径，包含图片将要输出时的文件名
	 * @throws Exception
	 */
	public static void toSinglePng(final String pdfFile, final String imgFilePath) throws FileConverterException {
		toSingleImg(pdfFile, imgFilePath, "png");
	}

	/**
	 * PDF转单个PNG图片，直接生成在PDF所在位置，并使用相同文件名。
	 * @param pdfFile pdf文件路径
	 * @throws Exception
	 */
	public static void toSinglePng(final String pdfFile) throws FileConverterException {
		toSingleImg(pdfFile, FilenameUtils.removeExtension(pdfFile), "png");
	}

	@Override
	public File convert(final File srcFile, final File destFile, final FileConverterSettings settings) throws Exception {
		// 转换为多页png图片，并放在一个以.dir结尾的子目录中
		File saveDir = new File(destFile.getParentFile(), destFile.getName() + FileConverterSettings.PAGED_FILE_DIR_SUFFIX);
		FileUtils.forceMkdir(saveDir);
		File destImageFilePath = new File(saveDir, destFile.getName());
		int count = Pdf2ImgConverter.toImg(srcFile.getAbsolutePath(), destImageFilePath.getAbsolutePath(), FilenameUtils.getExtension(destFile.getName()));
		logger.info("{}转换图片成功，共生成{}张图片", destFile.getName(), count);
		return saveDir;
	}
	
}
