package com.kdgcsoft.jt.xzzf.system.service.impl;

import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.kdgcsoft.jt.xzzf.common.constant.DBMetaDataConstant;
import com.kdgcsoft.jt.xzzf.common.util.IDUtil;
import com.kdgcsoft.jt.xzzf.frame.vo.ComboboxVo;
import com.kdgcsoft.jt.xzzf.system.entity.SysSubSystem;
import com.kdgcsoft.jt.xzzf.system.mapper.SysSubSystemMapper;
import com.kdgcsoft.jt.xzzf.system.service.SysMenuService;
import com.kdgcsoft.jt.xzzf.system.service.SysRoleService;
import com.kdgcsoft.jt.xzzf.system.service.SysSubSystemService;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;

/****
 * @className SysSubSystemServiceImpl
 * @author chris
 * @description 子系统服务层接口实现类
 * @version 1.0
 * @date 2019-12-24 11:12
 */
@Service
@Transactional(rollbackFor = Exception.class)
public class SysSubSystemServiceImpl implements SysSubSystemService {

    @Resource
    private SysSubSystemMapper sysSubSystemMapper;

    @Resource
    private SysMenuService sysMenuService;

    @Resource
    private SysRoleService sysRoleService;

    /*****
     * 分页查询
     * @param current 当前页
     * @param size 每页显示条数
     * @param entity 实体类
     * @return
     */
    @Override
    public Page<SysSubSystem> page(long current, long size, SysSubSystem entity) {
        return (Page<SysSubSystem>) this.sysSubSystemMapper.selectPage(new Page<>(current, size), Wrappers.<SysSubSystem>query().eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED));
    }

    /*****
     * 保存子系统数据
     * @param entity 实体类
     * @param isCreate 是否创建
     */
    @Override
    public void saveOrUpdate(SysSubSystem entity, boolean isCreate) {
        if (isCreate) {
            if (StrUtil.isNotBlank(entity.getId())) {
                throw new RuntimeException("服务异常,请联系管理员");
            }
        } else {
            SysSubSystem dbEntity = this.getEntityById(entity.getId());
            if (ObjectUtil.isNull(dbEntity)) {
                throw new RuntimeException("所选信息不存在,请重新选择");
            }
        }

        //判断名称是否为空
        if (StrUtil.isNotBlank(entity.getName())) {
            SysSubSystem dbEntity = this.getEntityByName(entity.getName());
            if (ObjectUtil.isNotNull(dbEntity)) {
                if (isCreate || !dbEntity.getId().equals(entity.getId())) {
                    throw new RuntimeException("子系统名称[" + entity.getName() + "]已存在,请重新填写");
                }
            }
        } else {
            throw new RuntimeException("名称不可为空");
        }

        //判断地址是否为空
        if (StrUtil.isBlank(entity.getUrl())) {
            throw new RuntimeException("地址不可为空");
        }

        //保存数据
        if (isCreate) {
            entity.setId(IDUtil.uuid());
            this.sysSubSystemMapper.insert(entity);
        } else {
            this.sysSubSystemMapper.updateById(entity);
        }
    }

    /******
     * 获取子系统信息
     * @param id 主键ID
     * @return
     */
    @Override
    public SysSubSystem getEntityById(String id) {
        if (StrUtil.isNotBlank(id)) {
            QueryWrapper<SysSubSystem> queryWrapper = new QueryWrapper<SysSubSystem>();
            queryWrapper.eq("SYSTEM_ID", id)
                    .eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED);

            SysSubSystem entity = this.sysSubSystemMapper.selectOne(queryWrapper);

            if (ObjectUtil.isNull(entity)) {
                throw new RuntimeException("记录不存在,请重试");
            }
            return entity;
        } else {
            throw new RuntimeException("缺少关键字,请重新输入");
        }
    }

    /******
     * 删除子系统信息
     * @param ids 主键Id集合
     */
    @Override
    public void deleteDataByIds(String ids) {
        if (StrUtil.isNotEmpty(ids)) {
            String[] idList = StrUtil.split(ids, ",");
            if (ArrayUtil.isNotEmpty(idList)) {
                for (String id : idList) {
                    //判断子系统下是否关联角色
                    Integer relRoleCnt = this.sysRoleService.getRoleCountBySubSysId(id);
                    if (relRoleCnt != 0) {
                        throw new RuntimeException("子系统已关联角色信息,暂不能删除");
                    }

                    //判断子系统下是否关联菜单
                    Integer relMenuCnt = this.sysMenuService.getMenuCountBySubSysId(id);
                    if (relMenuCnt != 0) {
                        throw new RuntimeException("子系统已关联菜单信息,暂不能删除");
                    }
                    //执行数据操作
                    SysSubSystem entity = new SysSubSystem();
                    entity.setDeleteFlag(DBMetaDataConstant.DELETED);
                    QueryWrapper<SysSubSystem> queryWrapper = new QueryWrapper<SysSubSystem>();
                    queryWrapper.eq("SYSTEM_ID", id);
                    this.sysSubSystemMapper.update(entity, queryWrapper);
                }
            }
        } else {
            throw new RuntimeException("请先选择待删除的数据");
        }
    }

    /*****
     * 查询子系统下拉框的键值对
     * @param filter 关键字过滤
     * @return
     */
    @Override
    public List<ComboboxVo> queryEntityCombobox(String filter) {
        return this.sysSubSystemMapper.queryEntityCombobox(filter);
    }

    /*****
     * 获取子系统信息通过名称
     * @param name 名称
     * @return
     */
    private SysSubSystem getEntityByName(String name) {
        return this.sysSubSystemMapper.selectOne(Wrappers.<SysSubSystem>query().eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED).eq("NAME", name));
    }
}
