package com.kdgcsoft.jt.xzzf.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.kdgcsoft.jt.xzzf.common.constant.DBMetaDataConstant;
import com.kdgcsoft.jt.xzzf.common.util.IDUtil;
import com.kdgcsoft.jt.xzzf.frame.constants.SystemConstants;
import com.kdgcsoft.jt.xzzf.system.entity.SysMenu;
import com.kdgcsoft.jt.xzzf.system.entity.SysSubSystem;
import com.kdgcsoft.jt.xzzf.system.mapper.SysMenuMapper;
import com.kdgcsoft.jt.xzzf.system.service.SysMenuService;
import com.kdgcsoft.jt.xzzf.system.service.SysRoleMenuService;
import com.kdgcsoft.jt.xzzf.system.service.SysSubSystemService;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

/**
 * <p>
 * 菜单 服务实现类
 * </p>
 *
 * @author mapp
 * @since 2019-12-03
 */
@Service
@Transactional(rollbackFor = Exception.class)
public class SysMenuServiceImpl implements SysMenuService {

    @Resource
    private SysMenuMapper sysMenuMapper;

    @Resource
    private SysRoleMenuService sysRoleMenuService;

    @Resource
    private SysSubSystemService sysSubSystemService;

    /****
     * 分页查询
     * @param current 当前页
     * @param size    每页显示条数
     * @param entity
     * @return
     */
    @Override
    public Page<SysMenu> page(long current, long size, SysMenu entity) {
        return (Page<SysMenu>) this.sysMenuMapper.selectPage(new Page<>(current, size), Wrappers.<SysMenu>query().eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED));
    }

    /*****
     * 保存组系统菜单数据
     * @param entity
     * @param isCreate
     */
    @Override
    public void saveOrUpdate(SysMenu entity, boolean isCreate) {
        if (isCreate) {
            if (StrUtil.isNotBlank(entity.getId())) {
                throw new RuntimeException("服务异常,请联系管理员");
            }
        } else {
            SysMenu dbEntity = this.getEntityById(entity.getId());
            if (ObjectUtil.isNull(dbEntity)) {
                throw new RuntimeException("所选信息不存在,请重新选择");
            }
        }

        //判断父级ID是否为空
        if (StrUtil.isNotBlank(entity.getMenuPid())) {
            SysMenu dbEntity = this.getEntityById(entity.getMenuPid());
            if (ObjectUtil.isNull(dbEntity)) {
                throw new RuntimeException("所属父级不存在,请重新选择");
            }

            //判断父级节点和子集节点是否为同一个
            if (!isCreate) {
                if (entity.getMenuPid().equals(entity.getMenuId())) {
                    throw new RuntimeException("所属父级选择错误,请重新选择");
                }
            }
        }

        //判断编码是否存在
        if (StrUtil.isNotBlank(entity.getMenuCode())) {
            //当菜单类型为按钮时,需要对编码进行下一步处理
            if (SystemConstants.SYS_MENU_TYPE_1.equals(entity.getType())) {
                SysMenu dbParentEntity = this.getEntityById(entity.getMenuPid());
                if (ObjectUtil.isNotNull(dbParentEntity)) {
                    //赋值按钮的编码
                    entity.setMenuCode(dbParentEntity.getMenuCode() + "." + entity.getMenuCode());
                }
            }
            SysMenu dbEntity = this.getEntityByCode(entity.getMenuCode());
            if (ObjectUtil.isNotNull(dbEntity)) {
                if (isCreate || !dbEntity.getId().equals(entity.getId())) {
                    throw new RuntimeException("编码[" + entity.getMenuCode() + "]已存在,请重新填写");
                }
            }
        } else {
            throw new RuntimeException("编码不可为空");
        }

        //判断子系统Id是否为空
        if (StrUtil.isNotBlank(entity.getSubSysId())) {
            SysSubSystem dbSysSubSystem = this.sysSubSystemService.getEntityById(entity.getSubSysId());
            if (ObjectUtil.isNull(dbSysSubSystem)) {
                throw new RuntimeException("所属子系统不存在,请重新选择");
            }
        } else {
            throw new RuntimeException("所属子系统不可为空");
        }

        //保存数据
        if (isCreate) {
            entity.setId(IDUtil.uuid());
            this.sysMenuMapper.insert(entity);
        } else {
            this.updateEntityInfo(entity);
        }
    }

    /*****
     * 删除菜单信息
     * @param ids
     */
    @Override
    public void deleteDataByIds(String ids) {
        if (StrUtil.isNotBlank(ids)) {
            String[] idList = StrUtil.split(ids, ",");
            if (ArrayUtil.isNotEmpty(idList)) {
                for (String id : idList) {
                    //判断菜单下是否有子菜单
                    Integer menuChildCnt = this.getMenuChildCountById(id);
                    if (menuChildCnt != 0) {
                        throw new RuntimeException("菜单下有子菜单，不能删除");
                    }

                    //判断菜单是否跟角色有关联信息
                    Integer roleMenuCnt = this.sysRoleMenuService.getRoleMenuRelCountByMenuId(id);
                    if (roleMenuCnt != 0) {
                        throw new RuntimeException("菜单已与角色相关联，不能删除");
                    }

                    //删除按钮
                    SysMenu sysMenuBtn = new SysMenu();
                    sysMenuBtn.setDeleteFlag(DBMetaDataConstant.DELETED);
                    this.sysMenuMapper.update(sysMenuBtn, Wrappers.<SysMenu>update().eq("MENU_PID", id));
                    //删除菜单
                    SysMenu sysMenu = new SysMenu();
                    sysMenu.setId(id);
                    sysMenu.setDeleteFlag(DBMetaDataConstant.DELETED);
                    this.updateEntityInfo(sysMenu);
                }
            }
        } else {
            throw new RuntimeException("请先选择待删除的数据");
        }
    }

    /*****
     * 根据ID获取菜单信息
     * @param id
     * @return
     */
    @Override
    public SysMenu getEntityById(String id) {
        return this.sysMenuMapper.selectById(id);
    }

    /******
     * 获取菜单总数
     * @param subSysId 子系统Id
     * @return
     */
    @Override
    public Integer getMenuCountBySubSysId(String subSysId) {
        return this.sysMenuMapper.getMenuCountBySubSysId(subSysId);
    }

    /*****
     * 获取所有菜单
     * @param sysMenu
     * @return
     */
    @Override
    public List<SysMenu> queryAllMenus(SysMenu sysMenu) {
        List<SysMenu> sysMenus = this.sysMenuMapper.selectList(Wrappers.<SysMenu>query()
                .eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED)
                .eq("TYPE", SystemConstants.SYS_MENU_TYPE_0));
        if (CollUtil.isNotEmpty(sysMenus)) {
            // 菜单下的按钮
            for (SysMenu menu : sysMenus) {
                List<SysMenu> btns = this.sysMenuMapper.selectList(Wrappers.<SysMenu>query()
                        .eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED)
                        .eq("TYPE", SystemConstants.SYS_MENU_TYPE_1)
                        .eq("MENU_PID", menu.getMenuId()));
                if (CollUtil.isNotEmpty(btns)) {
                    menu.setBtns(btns);
                }
            }
        }

        return sysMenus;
    }

    /******
     * 查询角色菜单信息
     * @param roleId 角色主键Id
     * @return
     */
    @Override
    public List<SysMenu> queryRoleMenuByRoleId(String roleId) {
        return sysMenuMapper.queryRoleMenuByRoleId(roleId);
    }

    /******
     * 获取用户所拥有的菜单
     * @param userId 用户Id
     * @return
     */
    @Override
    public List<SysMenu> queryUserMenuByUserId(String userId) {
        return sysMenuMapper.queryUserMenuByUserId(userId);
    }

    /******
     * 获取菜单信息通过编码
     * @param code
     * @return
     */
    private SysMenu getEntityByCode(String code) {
        QueryWrapper<SysMenu> queryWrapper = new QueryWrapper<SysMenu>();
        queryWrapper.eq("MENU_CODE", code)
                .eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED);
        return this.sysMenuMapper.selectOne(queryWrapper);
    }

    /*****
     * 更新实体类信息
     * @param entity
     */
    private void updateEntityInfo(SysMenu entity) {
        this.sysMenuMapper.updateById(entity);
    }

    /******
     * 获取某菜单下子菜单的总数
     * @param id 菜单主键
     * @return
     */
    private Integer getMenuChildCountById(String id) {
        return this.sysMenuMapper.selectCount(Wrappers.<SysMenu>query().eq("MENU_PID", id)
                .eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED)
                .eq("TYPE", SystemConstants.SYS_MENU_TYPE_0));
    }
//
//    /**
//     * 分页查询
//     *
//     * @param current 当前页
//     * @param size    每页显示条数
//     * @param sysMenu
//     * @return
//     */
//    @Override
//    public Page<SysMenu> page(long current, long size, SysMenu sysMenu) {
//        return (Page<SysMenu>) sysMenuMapper.selectPage(new Page<>(current, size),
//                Wrappers.<SysMenu>query().eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED));
//    }
//
//    /**
//     * 新增菜单信息
//     *
//     * @param sysMenu
//     * @return
//     */
//    @Override
//    public Integer insert(SysMenu sysMenu) {
//        if ("1".equals(sysMenu.getType())) {
//            // 按钮编码为 所属菜单编码.按钮编码  eg: USER_MENU.ADD
//            SysMenu menu = getById(sysMenu.getMenuPid());
//            sysMenu.setMenuCode(menu.getMenuCode() + "." + sysMenu.getMenuCode());
//        }
//
//        // 判断编码是否重复
//        Integer res = sysMenuMapper.selectCount(
//                Wrappers.<SysMenu>query()
//                        .eq("MENU_CODE", sysMenu.getMenuCode())
//                        .eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED));
//        if (res > 0) {
//            return 0;
//        }
//        sysMenuMapper.insert(sysMenu);
//        return 1;
//    }
//
//    /**
//     * 修改菜单信息
//     *
//     * @param sysMenu
//     * @return
//     */
//    @Override
//    public boolean updateById(SysMenu sysMenu) {
//        return SqlHelper.retBool(sysMenuMapper.updateById(sysMenu));
//    }
//
//    /**
//     * 根据ID删除菜单信息
//     *
//     * @param id 　菜单ID
//     * @return
//     */
//    @Override
//    public Integer deleteById(String id) {
//        // 判断是否有子菜单，有子菜单先删除子菜单
//        Integer res = sysMenuMapper.selectCount(Wrappers.<SysMenu>query().eq("MENU_PID", id)
//                .eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED)
//                .eq("TYPE", "0"));
//        if (res > 0) {
//            return 0;
//        }
//
//        SysMenu sysMenu = new SysMenu();
//        sysMenu.setId(id);
//        sysMenu.setDeleteFlag(DBMetaDataConstant.DELETED);
//        this.updateEntityInfo(sysMenu);
//        // 删除按钮
//        SysMenu btn = new SysMenu();
//        btn.setDeleteFlag(DBMetaDataConstant.DELETED);
//        sysMenuMapper.update(btn, Wrappers.<SysMenu>update().eq("MENU_PID", id));
//        return 1;
//    }
//
//
//    /**
//     * 保存角色菜单信息
//     *
//     * @param roleId
//     * @param menuIds
//     * @return
//     */
//    @Override
//    public boolean saveRoleMenu(String roleId, String menuIds) {
//        // 删除角色菜单信息
//        sysMenuMapper.deleteRoleMenu(roleId);
//        String[] menuIdArr = menuIds.split(",");
//        for (String menuId : menuIdArr) {
//            sysMenuMapper.saveRoleMenu(roleId, menuId);
//        }
//        return true;
//    }

}
