package com.kdgcsoft.jt.xzzf.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.kdgcsoft.jt.xzzf.common.constant.DBMetaDataConstant;
import com.kdgcsoft.jt.xzzf.common.util.IDUtil;
import com.kdgcsoft.jt.xzzf.frame.constants.DictConstants;
import com.kdgcsoft.jt.xzzf.system.entity.SysDept;
import com.kdgcsoft.jt.xzzf.system.entity.SysOrg;
import com.kdgcsoft.jt.xzzf.system.mapper.SysDeptMapper;
import com.kdgcsoft.jt.xzzf.system.service.SysDeptService;
import com.kdgcsoft.jt.xzzf.system.service.SysDicService;
import com.kdgcsoft.jt.xzzf.system.service.SysOrgService;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

/****
 * @className SysDeptServiceImpl
 * @author chris
 * @description 部门服务层接口实现类
 * @version 1.0
 * @date 2019-12-23 19:31
 */
@Service
@Transactional(rollbackFor = Exception.class)
public class SysDeptServiceImpl implements SysDeptService {

    @Resource
    private SysDeptMapper sysDeptMapper;

    @Resource
    private SysOrgService sysOrgService;

    @Resource
    private SysDicService sysDicService;

    /****
     * 分页查询
     * @param page 分页信息
     * @param entity 实体类
     * @return
     */
    @Override
    public Page<SysDept> page(Page<SysDept> page, SysDept entity) {
        //获取实体类封装信息
        Wrapper<SysDept> wrapper = this.getQueryWrapper(entity, true);

        //查询数据信息
        List<SysDept> dataList = this.queryOrExportData(wrapper, page, true);

        return page.setRecords(dataList);
    }

    /****
     * 保存组织机构数据
     * @param entity
     * @param isCreate
     */
    @Override
    public void saveOrUpdate(SysDept entity, boolean isCreate) {
        if (isCreate) {
            if (StrUtil.isNotBlank(entity.getId())) {
                throw new RuntimeException("服务异常,请联系管理员");
            }
        } else {
            SysDept dbEntity = this.getEntityById(entity.getId());
            if (ObjectUtil.isNull(dbEntity)) {
                throw new RuntimeException("所选信息不存在,请重新选择");
            }
        }

        //判断父级ID是否为空
        if (StrUtil.isNotBlank(entity.getDeptPid())) {
            //判断父级是否存在
            if (!SysDept.DEPT_ROOT_ID.equals(entity.getDeptPid())) {
                SysDept dbEntity = this.getEntityById(entity.getDeptPid());
                if (ObjectUtil.isNull(dbEntity)) {
                    throw new RuntimeException("所属父级不存在,请重新选择");
                }
            }

            //判断父级节点和子集节点是否为同一个
            if (!isCreate) {
                if (entity.getDeptPid().equals(entity.getDeptId())) {
                    throw new RuntimeException("所属父级选择错误,请重新选择");
                }
            }
        } else {
            //赋值父级节点
            entity.setDeptPid(SysDept.DEPT_ROOT_ID);
        }

        //判断组织机构id是否为空
        if (StrUtil.isNotBlank(entity.getOrgId())) {
            //判断组织机构是否存在
            SysOrg dbEntity = this.sysOrgService.getEntityById(entity.getOrgId());
            if (ObjectUtil.isNull(dbEntity)) {
                throw new RuntimeException("组织机构不存在,请重新选择");
            }
        } else {
            throw new RuntimeException("组织机构不可为空");
        }

        //判断编码是否为空
        if (StrUtil.isNotBlank(entity.getDeptCode())) {
            //判断编码是否重复
            SysDept dbEntity = this.getEntityByCode(entity.getDeptCode());
            if (ObjectUtil.isNotNull(dbEntity)) {
                if (isCreate || !dbEntity.getId().equals(entity.getId())) {
                    throw new RuntimeException("编码[" + entity.getDeptCode() + "]已存在,请重新填写");
                }
            }
        } else {
            throw new RuntimeException("编码不可为空");
        }

        //判断名称是否为空
        if (StrUtil.isNotBlank(entity.getDeptName())) {
            SysDept dbEntity = this.getEntityByName(entity.getDeptName());
            if (ObjectUtil.isNotNull(dbEntity)) {
                if (isCreate || !dbEntity.getId().equals(entity.getId())) {
                    throw new RuntimeException("名称[" + entity.getDeptName() + "]已存在,请重新填写");
                }
            }
        } else {
            throw new RuntimeException("名称不可为空");
        }

        //保存数据
        if (isCreate) {
            entity.setId(IDUtil.uuid());
            this.sysDeptMapper.insert(entity);
        } else {
            this.sysDeptMapper.updateById(entity);
        }
    }

    /****
     * 获取部门信息
     * @param id 主键ID
     * @return
     */
    @Override
    public SysDept getEntityById(String id) {
        if (StrUtil.isNotBlank(id)) {
            QueryWrapper<SysDept> queryWrapper = new QueryWrapper<SysDept>();
            queryWrapper.eq("DEPT_ID", id)
                    .eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED);

            SysDept entity = this.sysDeptMapper.selectOne(queryWrapper);

            if (ObjectUtil.isNull(entity)) {
                throw new RuntimeException("记录不存在,请重试");
            }
            return entity;
        } else {
            throw new RuntimeException("缺少关键字,请重新输入");
        }
    }

    /****
     * 获取部门信息
     * @param ids
     */
    @Override
    public void deleteDataByIds(String ids) {
        if (StrUtil.isNotEmpty(ids)) {
            String[] idList = StrUtil.split(ids, ",");
            if (ArrayUtil.isNotEmpty(idList)) {
                for (String id : idList) {
                    SysDept entity = new SysDept();
                    entity.setDeleteFlag(DBMetaDataConstant.DELETED);
                    QueryWrapper<SysDept> queryWrapper = new QueryWrapper<SysDept>();
                    queryWrapper.eq("DEPT_ID", id);
                    this.sysDeptMapper.update(entity, queryWrapper);
                }
            }
        } else {
            throw new RuntimeException("请先选择待删除的数据");
        }
    }

    /****
     * 获取部门树数据
     * @return
     */
    @Override
    public List<SysDept> queryTreeData() {
        return this.sysDeptMapper.selectList(Wrappers.<SysDept>query().eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED));
    }

    /****
     * 获取部门信息
     * @param code 编码
     * @return
     */
    private SysDept getEntityByCode(String code) {
        return this.sysDeptMapper.selectOne(Wrappers.<SysDept>query().eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED).eq("DEPT_CODE", code));
    }

    /****
     * 获取部门信息
     * @param name 名称
     * @return
     */
    private SysDept getEntityByName(String name) {
        return this.sysDeptMapper.selectOne(Wrappers.<SysDept>query().eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED).eq("DEPT_NAME", name));
    }

    /****
     * 获取实体类的封装信息
     * @param entity 实体类
     * @param isQuery 是否查询
     * @return
     */
    private Wrapper<SysDept> getQueryWrapper(SysDept entity, boolean isQuery) {
        // 条件查询构造器
        QueryWrapper<SysDept> wrapper = Wrappers.query();

        //部门编码
        String deptCode = entity.getDeptCode();

        //部门名称
        String deptName = entity.getDeptName();

        //拼接自定义Sql语句
        wrapper.eq(DBMetaDataConstant.DELETE_FLAG, DBMetaDataConstant.UNDELETED)
                .like(StrUtil.isNotBlank(deptCode), "DEPT_CODE", deptCode)
                .like(StrUtil.isNotBlank(deptName), "DEPT_NAME", deptName)
                .orderByDesc(entity.getDeptId());
        System.out.println("==============wrapper.getCustomSqlSegment():" + wrapper.getCustomSqlSegment());
        return wrapper;
    }

    /*****
     * 获取数据库数据
     * @param wrapper 过滤条件
     * @param page 分页信息
     * @param isQuery 是否查询
     * @return
     * @throws RuntimeException
     */
    private List<SysDept> queryOrExportData(Wrapper wrapper, Page<SysDept> page, boolean isQuery) {
        List<SysDept> dataList = new ArrayList<>();
        if (isQuery) {
            dataList = this.sysDeptMapper.queryEntityInfoPageDataList(page, wrapper);
        }

        //转换数据字典名称
        this.convertListDictNameByCode(dataList);

        return dataList;
    }

    /****
     * 转换数据字典名称
     * @param dataList 实体类List集合
     */
    private void convertListDictNameByCode(List<SysDept> dataList) {
        if (CollUtil.isNotEmpty(dataList)) {
            for (SysDept entity : dataList) {
//                entity.setDeptTypeText(this.sysDicService.getDictText(DictConstants.DICT_CODE_DEPT_TYPE, entity.getDeptType()));
            }
        }
    }
}
