package com.kdgcsoft.jt.xzzf.frame.shiro;

import com.kdgcsoft.jt.xzzf.common.config.XzzfProperties;
import com.kdgcsoft.jt.xzzf.common.entity.LoginUser;
import com.kdgcsoft.jt.xzzf.common.log.AsyncLog;
import com.kdgcsoft.jt.xzzf.common.log.factory.LogRunableFactory;
import com.kdgcsoft.jt.xzzf.common.util.IPUtil;
import com.kdgcsoft.jt.xzzf.common.util.UserAgentUtil;
import com.kdgcsoft.jt.xzzf.system.entity.SysMenu;
import com.kdgcsoft.jt.xzzf.system.entity.SysRole;
import com.kdgcsoft.jt.xzzf.system.service.LoginUserService;
import com.kdgcsoft.jt.xzzf.system.service.SysMenuService;
import com.kdgcsoft.jt.xzzf.system.service.SysRoleService;
import com.kdgcsoft.jt.xzzf.system.service.SysUserService;
import com.kdgcsoft.scrdc.frame.webframe.core.exception.LoginException;
import org.apache.shiro.authc.AuthenticationException;
import org.apache.shiro.authc.AuthenticationInfo;
import org.apache.shiro.authc.AuthenticationToken;
import org.apache.shiro.authc.SimpleAuthenticationInfo;
import org.apache.shiro.authz.AuthorizationInfo;
import org.apache.shiro.authz.SimpleAuthorizationInfo;
import org.apache.shiro.cas.CasRealm;
import org.apache.shiro.cas.CasToken;
import org.apache.shiro.subject.PrincipalCollection;
import org.apache.shiro.subject.SimplePrincipalCollection;
import org.apache.shiro.util.CollectionUtils;
import org.apache.shiro.util.StringUtils;
import org.jasig.cas.client.authentication.AttributePrincipal;
import org.jasig.cas.client.validation.Assertion;
import org.jasig.cas.client.validation.TicketValidationException;
import org.jasig.cas.client.validation.TicketValidator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author chris
 */
@Component
public class CasShiroRealm extends CasRealm {

    @Resource
    private SysUserService sysUserService;
    @Resource
    private XzzfProperties xzzfProperties;
    @Resource
    private LoginUserService loginUserService;
    @Resource
    private SysRoleService sysRoleService;
    @Resource
    private SysMenuService sysMenuService;

    private static final Logger LOG = LoggerFactory.getLogger(CasShiroRealm.class);

    @PostConstruct
    public void initProperties() {
        LOG.info("配置基本属性");
        this.setDefaultRoles("ROLE_USER");
        this.setCasServerUrlPrefix(xzzfProperties.getCasServerUrl());
        this.setCasService(xzzfProperties.getService() + "toIndex");
    }

    /**
     * 授权  使用 @RequiresPermissions,@RequiresRoles 使用
     */
    @Override
    protected AuthorizationInfo doGetAuthorizationInfo(PrincipalCollection principals) {
        LoginUser loginUser = (LoginUser) principals.getPrimaryPrincipal();
        SimpleAuthorizationInfo authorizationInfo = new SimpleAuthorizationInfo();
        // 设置角色（角色编码）
        List<SysRole> roles = sysRoleService.queryUserRoleRelByUserId(loginUser.getLoginUserId());
        authorizationInfo.setRoles(roles.stream().map(SysRole::getRoleCode).collect(Collectors.toSet()));
        // 设置权限（菜单，按钮编码）
        List<SysMenu> menus = sysMenuService.queryUserMenuByUserId(loginUser.getLoginUserId());
        authorizationInfo.setStringPermissions(menus.stream().map(SysMenu::getMenuCode).collect(Collectors.toSet()));
        return authorizationInfo;
    }

    /**
     * 认证
     */
    @Override
    protected AuthenticationInfo doGetAuthenticationInfo(AuthenticationToken token) throws AuthenticationException {
        LOG.info("进入doGetAuthenticationInfo");
        try {
            CasToken casToken = (CasToken) token;
            if (token == null) {
                return null;
            }
            String ticket = (String) casToken.getCredentials();
            if (!StringUtils.hasText(ticket)) {
                return null;
            }
            TicketValidator ticketValidator = ensureTicketValidator();
            Assertion casAssertion;
            try {
                casAssertion = ticketValidator.validate(ticket, getCasService());
            } catch (TicketValidationException e) {
                throw new LoginException("用户名密码出错", e);
            }
            LOG.info("远程验证通过");
            AttributePrincipal casPrincipal = casAssertion.getPrincipal();
            String userName = casPrincipal.getName();
            Map<String, Object> attributes = casPrincipal.getAttributes();
            casToken.setUserId(userName);
            String rememberMeAttributeName = getRememberMeAttributeName();
            String rememberMeStringValue = (String) attributes.get(rememberMeAttributeName);
            boolean isRemembered = rememberMeStringValue != null && Boolean.parseBoolean(rememberMeStringValue);
            if (isRemembered) {
                casToken.setRememberMe(true);
            }

            LoginUser loginUser;
            try {
                loginUser = this.loginUserService.loadLoginUserByUserName(userName);
            } catch (NullPointerException e) {
                throw new LoginException("用户不存在", e);
            }
            // 设置角色（角色ID）
            List<SysRole> roles = sysRoleService.queryUserRoleRelByUserId(loginUser.getLoginUserId());
            loginUser.setRoles(roles.stream().map(SysRole::getRoleId).collect(Collectors.toList()));
            // 设置权限（菜单编码）
            List<SysMenu> menus = sysMenuService.queryUserMenuByUserId(loginUser.getLoginUserId());
            loginUser.setPermissions(menus.stream().map(SysMenu::getMenuCode).collect(Collectors.toList()));
            List<?> principals = CollectionUtils.asList(loginUser, attributes);
            PrincipalCollection principalCollection = new SimplePrincipalCollection(principals, getName());

            // 日志记录
            if (xzzfProperties.isLogEnable()) {
                AsyncLog.getInstance().executeLog(LogRunableFactory.loginLog(loginUser.getLoginUserId(), userName, IPUtil.getIpAddress(),
                        "brower：" + UserAgentUtil.getBrower().getName() + "  version：" + UserAgentUtil.getUserAgent().getVersion()));
            }
            return new SimpleAuthenticationInfo(principalCollection, ticket);
        } catch (Exception e) {
            e.printStackTrace();
            throw new AuthenticationException(e.getMessage(), e);
        }
    }

//	@Override
//	protected Object getAuthorizationCacheKey(PrincipalCollection principals) {
//		return RedisCache.SHIRO_PRE_KEY + principals;
//	}
//
//	@Override
//	protected Object getAuthenticationCacheKey(AuthenticationToken token) {
//		return  token != null ? RedisCache.SHIRO_PRE_KEY + token : null;
//	}

}
