package com.kdgcsoft.scrdc.frame.webframe.core.config.shiro;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import com.kdgcsoft.jt.xzzf.common.config.XzzfProperties;
import com.kdgcsoft.jt.xzzf.common.config.shiro.UniqueUserFilter;
import com.kdgcsoft.jt.xzzf.common.config.shiro.XzzfUniqueUserFilter;
import com.kdgcsoft.jt.xzzf.common.config.shiro.XzzfUserFilter;
import com.kdgcsoft.scrdc.frame.webframe.core.service.FrameService;
import org.apache.shiro.authc.pam.AtLeastOneSuccessfulStrategy;
import org.apache.shiro.authc.pam.ModularRealmAuthenticator;
import org.apache.shiro.mgt.SecurityManager;
import org.apache.shiro.mgt.SubjectFactory;
import org.apache.shiro.realm.Realm;
import org.apache.shiro.session.mgt.SessionManager;
import org.apache.shiro.spring.security.interceptor.AuthorizationAttributeSourceAdvisor;
import org.apache.shiro.spring.web.ShiroFilterFactoryBean;
import org.apache.shiro.web.mgt.DefaultWebSecurityManager;
import org.apache.shiro.web.mgt.DefaultWebSubjectFactory;
import org.apache.shiro.web.session.mgt.DefaultWebSessionManager;
import org.springframework.aop.framework.autoproxy.DefaultAdvisorAutoProxyCreator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import javax.servlet.Filter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

//import com.kdgcsoft.scrdc.frame.webframe.core.config.shiro.filter.UrlPermissionCheckFilter;

/**
 * @author fyin
 * @createTime 2018/7/20/020 15:29
 * @description
 */
@Configuration
public class ShiroConfig {

    @Autowired
    FrameService frameService;
    @Autowired
    ConfigurableApplicationContext configurableContex;
    @Autowired
    private ShiroCasProperties shiroCasProperties;
    @Autowired
    private XzzfProperties xzzfProperties;

    /**
     * 内置超级用户的realm
     */

    @Bean
    public SuperAdminRealm superAdminRealm() {
        return new SuperAdminRealm();
    }


    @Bean
    @ConditionalOnMissingBean
    public SecurityManager securityManager(SessionManager sessionManager) {
        DefaultWebSecurityManager securityManager = new DefaultWebSecurityManager();
        securityManager.setAuthenticator(modularRealmAuthenticator());
        List<Realm> realms = new ArrayList<>();
        if (shiroCasProperties.isAdminEnable()) {
            realms.add(superAdminRealm());
        }
        if (frameService.getRealmClass() != null) {
            List<Class> realmClassList = frameService.getRealmClassList();
            for (Class aClass : realmClassList) {
                realms.add((Realm) configurableContex.getBean(aClass));
            }
        }
        securityManager.setRealms(realms);
        securityManager.setSubjectFactory(subjectFactory());
        // 默认使用 ServletContainerSessionManager 来管理session，使用的是servle容器管理session。
//        securityManager.setSessionManager(sessionManager);
//        ThreadContext.bind(securityManager);
        return securityManager;
    }

    @Bean
    @ConditionalOnMissingBean
    public SubjectFactory subjectFactory() {
        SubjectFactory subjectFactory = new DefaultWebSubjectFactory();
        return subjectFactory;
    }

    /**
     * 多重realm验证,至少一个通过就算通过
     **/
    @Bean
    public ModularRealmAuthenticator modularRealmAuthenticator() {
        //自己重写的ModularRealmAuthenticator
        ModularRealmAuthenticator modularRealmAuthenticator = new ModularRealmAuthenticator();
        modularRealmAuthenticator.setAuthenticationStrategy(new AtLeastOneSuccessfulStrategy());
        return modularRealmAuthenticator;
    }

    /**
     * Filter工厂，设置对应的过滤条件和跳转条件
     */
    @Bean/*(name = "shiroFilter")*/
    public ShiroFilterFactoryBean shiroFilterFactoryBean(SecurityManager securityManager) {
        ShiroFilterFactoryBean shiroFilterFactoryBean = new ShiroFilterFactoryBean();
        shiroFilterFactoryBean.setSecurityManager(securityManager);
        Map<String, Filter> filters = frameService.getShiroFilters();
        // 无法拦截页面内部的请求
//        filters.put("urlCheckFilter", new UrlPermissionCheckFilter());
        filters.putAll(getFilters());
        shiroFilterFactoryBean.setFilters(filters);
        Map<String, String> map = MapUtil.newHashMap(true);
        //忽略静态资源
        map.put("/static/**", "anon");
        map.put("/webjars/**", "anon");
        map.put("/style/**", "anon");
        map.put("/js/**", "anon");
        map.put("/plugins/**", "anon");
        map.put("/theme/**", "anon");
        map.put("/init/**", "anon");
        map.put("/401", "anon");

        //登出
        map.put("/logout", "logout");
        map.put("/quit", "logout");

        // 放行特定接口
        map.put("/xtbadocument/documentTree", "anon");

        /**添加扩展的shiro过滤器**/
        map.putAll(frameService.getShiroFilterChainMap());

        // 第三方系统可能会有权限为anon的请求
        //对所有用户认证
        if (shiroCasProperties.isShiroEnable()) {
            if (xzzfProperties.isFrontEnable()) {
                if (xzzfProperties.isUniqueUser()) {
                    map.put("/**", "xzzfUniqueUser");
                } else {
                    map.put("/**", "xzzfUser");
                }
            } else {
                if (xzzfProperties.isUniqueUser()) {
                    map.put("/**", "uniqueUser");
                } else {
                    map.put("/**", "user");
                }
            }
        } else {
            map.put("/**", "anon");
        }

        //登录
        if (StrUtil.isNotEmpty(frameService.getShiroLoginUrl())) {
            shiroFilterFactoryBean.setLoginUrl(frameService.getShiroLoginUrl());
        } else {
            shiroFilterFactoryBean.setLoginUrl("/login");
        }
        //首页
        shiroFilterFactoryBean.setSuccessUrl(shiroCasProperties.getSuccessUrl());
        //错误页面，认证不通过跳转
        shiroFilterFactoryBean.setUnauthorizedUrl("/login");
        shiroFilterFactoryBean.setFilterChainDefinitionMap(map);
        shiroFilterFactoryBean.setUnauthorizedUrl("/401");
        return shiroFilterFactoryBean;
    }

    private Map<String, Filter> getFilters() {
        Map<String, Filter> filterMap = new HashMap<>();
        filterMap.put("xzzfUser", new XzzfUserFilter());
        filterMap.put("xzzfUniqueUser", new XzzfUniqueUserFilter());
        filterMap.put("uniqueUser", new UniqueUserFilter());
        return filterMap;
    }

    /**
     * 加入注解的使用，不加入这个注解不生效
     */
    @Bean
    public AuthorizationAttributeSourceAdvisor authorizationAttributeSourceAdvisor(SecurityManager securityManager) {
        AuthorizationAttributeSourceAdvisor authorizationAttributeSourceAdvisor = new AuthorizationAttributeSourceAdvisor();
        authorizationAttributeSourceAdvisor.setSecurityManager(securityManager);
        return authorizationAttributeSourceAdvisor;
    }

    @Bean
    public DefaultWebSessionManager sessionManager() {
        // DefaultWebSessionManager 使用的是shiro自己管理的session，而非servlet容器session。在与cas对接时，会与SingleSignOutHandler
        // 中的session不一致，导致重复重定向问题
        DefaultWebSessionManager sessionManager = new DefaultWebSessionManager();
        sessionManager.setSessionIdUrlRewritingEnabled(false);
        return sessionManager;
    }

    @Bean
    public static DefaultAdvisorAutoProxyCreator getDefaultAdvisorAutoProxyCreator() {
        DefaultAdvisorAutoProxyCreator creator = new DefaultAdvisorAutoProxyCreator();
        /**
         * setUsePrefix(false)用于解决一个奇怪的bug。在引入spring aop的情况下。
         * 在@Controller注解的类的方法中加入@RequiresRole注解，会导致该方法无法映射请求，导致返回404。
         * 加入这项配置能解决这个bug
         */
        creator.setUsePrefix(true);
        return creator;
    }
}
