package com.kdgcsoft.jt.xzzf.common.log.aop;

import cn.hutool.core.util.StrUtil;
import com.kdgcsoft.jt.xzzf.common.annotation.Log;
import com.kdgcsoft.jt.xzzf.common.config.XzzfProperties;
import com.kdgcsoft.jt.xzzf.common.enums.OperateType;
import com.kdgcsoft.jt.xzzf.common.log.AsyncLog;
import com.kdgcsoft.jt.xzzf.common.log.factory.LogRunableFactory;
import com.kdgcsoft.jt.xzzf.common.util.BeanUtils;
import com.kdgcsoft.jt.xzzf.common.util.IPUtil;
import com.kdgcsoft.jt.xzzf.common.util.SysUtils;
import com.kdgcsoft.jt.xzzf.dubbo.system.sys.entity.SysUser;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Method;
import java.util.HashMap;


/**
 * 日志记录aop方式
 *
 * @author mappp
 */
@Aspect
@Component
public class LogAop {

    @Autowired
    private XzzfProperties xzzfProperties;

    @Pointcut(value = "@annotation(com.kdgcsoft.jt.xzzf.common.annotation.Log)")
    private void logAop() {

    }

    /**
     * 处理完请求后执行
     *
     * @param joinPoint 切点
     */
    @AfterReturning(pointcut = "logAop()", returning = "result")
    public void doAfterReturning(JoinPoint joinPoint, Object result) {
        if (xzzfProperties.isLogEnable()) {
            handleLog(joinPoint, result);
        }
    }

    /**
     * 异常拦截
     *
     * @param e
     * @return
     */
    @AfterThrowing(value = "execution(public * com.kdgcsoft..*.*controller..*.*(..))", throwing = "e")
    @ResponseBody
    public void exception(final JoinPoint joinPoint, Exception e) {
        if (xzzfProperties.isLogEnable()) {
            errorLog(joinPoint, e);
        }
    }

    /**
     * 记录操作日志
     *
     * @param joinPoint
     * @param jsonResult 方法返回值
     */
    private void handleLog(final JoinPoint joinPoint, final Object jsonResult) {

        Log log = getAnnotationLog(joinPoint);
        if (log != null) {
            ProceedingJoinPoint point = (ProceedingJoinPoint) joinPoint;
            Signature signature = point.getSignature();
            MethodSignature methodSignature = (MethodSignature) signature;

            // 类名
            String className = methodSignature.getDeclaringTypeName();
            // 方法名
            String methodName = methodSignature.getName();
            // 参数名称及值
            HashMap[] argsArr = createArgsObj(point);
            // 操作类型
            OperateType type = log.type();
            // 模块名称
            String model = log.model();
            // 描述
            String des = log.des();
            SysUser user = SysUtils.getCurUser();
            String userId = user != null ? user.getUserId() : "";
            // ip
            HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
            String ip = IPUtil.getIpAddress(request);
            //所属子系统
            String subSysId = BeanUtils.isNotEmpty(xzzfProperties.getSubSysKey()) ? xzzfProperties.getSubSysKey() : "";
            AsyncLog.getInstance().executeLog(LogRunableFactory.operateLog(model, des, type, className, methodName, argsArr, jsonResult, userId, ip, subSysId));
        }
    }


    /**
     * 记录异常日志
     *
     * @param joinPoint
     * @param e
     */
    private void errorLog(final JoinPoint joinPoint, final Exception e) {
        ProceedingJoinPoint point = (ProceedingJoinPoint) joinPoint;
        Signature signature = point.getSignature();
        MethodSignature methodSignature = (MethodSignature) signature;

        // 类名
        String className = methodSignature.getDeclaringTypeName();
        // 方法名
        String methodName = methodSignature.getName();
        // 参数名称及值
        HashMap[] argsArr = createArgsObj(point);
        // 异常信息
        String errMsg = errMsg(e);
        SysUser user = SysUtils.getCurUser();
        String userId = user != null ? user.getUserId() : "";
        //所属子系统
        String subSysId = BeanUtils.isNotEmpty(xzzfProperties.getSubSysKey()) ? xzzfProperties.getSubSysKey() : "";
        AsyncLog.getInstance().executeLog(LogRunableFactory.errorLogin(userId, className, methodName, argsArr, errMsg, subSysId));
    }


    /**
     * 是否存在注解，如果存在就获取
     */
    private Log getAnnotationLog(JoinPoint joinPoint) {
        Signature signature = joinPoint.getSignature();
        MethodSignature methodSignature = (MethodSignature) signature;
        Method method = methodSignature.getMethod();

        if (method != null) {
            return method.getAnnotation(Log.class);
        }
        return null;
    }

    /**
     * 组装参数名称及值
     * [{"argsName" : argsValue}]
     */
    private HashMap[] createArgsObj(ProceedingJoinPoint point) {
        Signature signature = point.getSignature();
        Object[] args = point.getArgs();
        MethodSignature methodSignature = (MethodSignature) signature;
        String[] parameterNames = methodSignature.getParameterNames();

        HashMap[] argsArr = new HashMap[parameterNames.length];
        for (int i = 0; i < parameterNames.length; i++) {
            HashMap<String, Object> arg = new HashMap<>();
            arg.put(parameterNames[i], args[i]);
            argsArr[i] = arg;
        }

        return argsArr;
    }

    /**
     * 异常信息
     *
     * @param error
     * @return
     */
    private String errMsg(Exception error) {
        return "error_class：" + error.getClass().getName() + " error_msg " + error.getMessage();
    }
}
