package com.kdgcsoft.jt.xzzf.common.excel.jxls;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.kdgcsoft.jt.xzzf.common.config.XzzfProperties;
import com.kdgcsoft.jt.xzzf.common.exception.BusinessException;
import com.kdgcsoft.jt.xzzf.common.util.BeanUtils;
import com.kdgcsoft.jt.xzzf.common.util.IDUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.view.AbstractView;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.net.URLEncoder;
import java.util.*;

/****
 * @className JxlsExcelView
 * @author chris
 * @description
 * @version 1.0
 * @date 2020-04-09 21:58
 */
@Slf4j
@Component
public class JxlsExcelView extends AbstractView {

    private static XzzfProperties xzzfProperties;

    /******
     * 转换器
     */
    private XlsTransformerExt transformer;

    /**
     * Default Constructor.
     * Sets the content type of the view to "application/vnd.ms-excel".
     */
    public JxlsExcelView() {
        transformer = new XlsTransformerExt();
        setContentType(JxlsConfig.CONTENT_TYPE);
    }

    /*****
     * 赋值配置文件
     * @param xzzfProperties
     */
    @Autowired
    public void setXzzfProperties(XzzfProperties xzzfProperties) {
        JxlsExcelView.xzzfProperties = xzzfProperties;
    }

    /*****
     *
     * @return
     */
    @Override
    protected boolean generatesDownloadContent() {
        return true;
    }

    /**
     * Subclasses must implement this method to actually render the view.
     * <p>The first step will be preparing the request: In the JSP case,
     * this would mean setting model objects as request attributes.
     * The second step will be the actual rendering of the view,
     * for example including the JSP via a RequestDispatcher.
     *
     * @param model    combined output Map (never {@code null}),
     *                 with dynamic values taking precedence over static attributes
     * @param request  current HTTP request
     * @param response current HTTP response
     * @throws Exception if rendering failed
     */
    @Override
    protected void renderMergedOutputModel(Map<String, Object> model, HttpServletRequest request, HttpServletResponse response) throws Exception {
        Date startTime = DateUtil.date();
        //判断是否为导出
        boolean isExport = JxlsConfig.OPERATE_TYPE_EXPORTS.equals((String) model.get(JxlsConfig.OPERATE_TYPE));
        //操作类型Text值
        String operateTypeText = (isExport ? "导出" : "导入");

        log.info("======================>开始" + operateTypeText + "时间:" + startTime);
        //获取文件名
        String fileName = (String) model.get(JxlsConfig.EXCEL_EXPORT_FILE_NAME);
        //获取模板文件
        String templateName = (String) model.get(JxlsConfig.EXCEL_TEMPLATE_FILE_NAME);
        //获取数据信息
        List dataList = (List) model.get(JxlsConfig.EXCEL_DATA_LIST);
        // 判断浏览器
        String agent = request.getHeader("USER-AGENT").toLowerCase();
        // 处理导出文件名为中文乱码情况
        if (agent.indexOf(JxlsConfig.BROWSER_TYPE_FIREFOX) > -1) {
            // 浏览器为火狐
            fileName = new String(fileName.getBytes("UTF-8"), "ISO-8859-1");
        } else {
            // 默认浏览器为IE，其他的浏览器皆可用
            fileName = URLEncoder.encode(fileName, "UTF-8");
        }

        //设置response响应数据信息
        response.setHeader("content-disposition", "attachment; filename=" + fileName);
        response.setContentType(getContentType());
        //初始化定义模板文件真实路径(不包含文件名)
        String tempRealPath = getTempFileRealPath(isExport, templateName);
        //打印路径信息
        log.info("======================>" + operateTypeText + "模板文件路径:" + tempRealPath);
        ServletOutputStream out = response.getOutputStream();
        transformer.transformXLS(tempRealPath, model, out, operateTypeText);
        out.flush();
        Date endTime = DateUtil.date();
        log.info("======================>共计:" + dataList.size() + "条");
        log.info("======================>结束" + operateTypeText + "时间:" + endTime);
        log.info("======================>共用" + operateTypeText + "时间:" + DateUtil.formatBetween(startTime, endTime));

    }

    /******
     * 下载模板文件
     * @param tempFilePath 模板路径
     * @param request
     * @param response
     */
    public static void downExcelTemp(String tempFilePath, HttpServletRequest request, HttpServletResponse response) {
        downExcelTemp(null, tempFilePath, request, response);
    }

    /*******
     * 下载模板文件
     * @param fileName 文件名称
     * @param tempFilePath 模板路径
     * @param request
     * @param response
     */
    public static void downExcelTemp(String fileName, String tempFilePath, HttpServletRequest request, HttpServletResponse response) {
        downExcelTemp(fileName, null, tempFilePath, request, response);
    }

    /******
     * 下载模板文件
     * @param fileName 文件名称
     * @param suffix 文件后缀名
     * @param tempFilePath 模板路径
     * @param request
     * @param response
     */
    private static void downExcelTemp(String fileName, String suffix, String tempFilePath, HttpServletRequest request, HttpServletResponse response) {
        Map<String, Object> map = new HashMap<>();
        if (BeanUtils.isEmpty(suffix)) {
            suffix = JxlsConfig.EXCEL_DEF_SUFFIX_2007;
        }
        if (BeanUtils.isEmpty(fileName)) {
            fileName = IDUtil.uuid();
        }
        if (BeanUtils.isEmpty(tempFilePath)) {
            throw new BusinessException("导入模板路径不可为空");
        }
        // 文件名
        map.put(JxlsConfig.EXCEL_EXPORT_FILE_NAME, fileName + StrUtil.DOT + suffix);
        //文件模板名
        map.put(JxlsConfig.EXCEL_TEMPLATE_FILE_NAME, tempFilePath);
        //类型
        map.put(JxlsConfig.OPERATE_TYPE, JxlsConfig.OPERATE_TYPE_IMPORTS);
        //数据
        map.put(JxlsConfig.EXCEL_DATA_LIST, new ArrayList<>());
        try {
            JxlsExcelView jxlsExcelView = new JxlsExcelView();
            jxlsExcelView.render(map, request, response);
        } catch (Exception e) {
            e.printStackTrace();
            log.info("======================>下载导入模板错误:" + e.getMessage());
            throw new BusinessException("======================>下载导入模板错误:" + e.getMessage());
        }
    }

    /*******
     * 导出数据
     * @param tempFilePath 导出模板路径
     * @param dataList 待导出的数据
     * @param request
     * @param response
     */
    public static void exportData(String tempFilePath, List<?> dataList, HttpServletRequest request, HttpServletResponse response) {
        exportData(null, tempFilePath, dataList, request, response);
    }

    /*******
     * 导出数据
     * @param fileName 文件名称
     * @param tempFilePath 导出模板路径
     * @param dataList 待导出的数据
     * @param request
     * @param response
     */
    public static void exportData(String fileName, String tempFilePath, List<?> dataList, HttpServletRequest request, HttpServletResponse response) {
        exportData(fileName, null, tempFilePath, dataList, request, response);
    }

    /******
     * 导出数据
     * @param fileName 文件名称
     * @param suffix 文件后缀名
     * @param tempFilePath 导出模板路径
     * @param dataList 待导出的数据
     * @param request
     * @param response
     */
    private static void exportData(String fileName, String suffix, String tempFilePath, List<?> dataList, HttpServletRequest request, HttpServletResponse response) {
        Map<String, Object> map = new HashMap<>();
        if (BeanUtils.isEmpty(suffix)) {
            suffix = JxlsConfig.EXCEL_DEF_SUFFIX_2007;
        }
        if (BeanUtils.isEmpty(fileName)) {
            fileName = IDUtil.uuid();
        }
        if (BeanUtils.isEmpty(tempFilePath)) {
            throw new BusinessException("导出模板路径不可为空");
        }
        // 文件名
        map.put(JxlsConfig.EXCEL_EXPORT_FILE_NAME, fileName + StrUtil.DOT + suffix);
        //文件模板名
        map.put(JxlsConfig.EXCEL_TEMPLATE_FILE_NAME, tempFilePath);
        //数据
        map.put(JxlsConfig.EXCEL_DATA_LIST, dataList);
        //类型
        map.put(JxlsConfig.OPERATE_TYPE, JxlsConfig.OPERATE_TYPE_EXPORTS);
        try {
            JxlsExcelView jxlsExcelView = new JxlsExcelView();
            jxlsExcelView.render(map, request, response);
        } catch (Exception e) {
            e.printStackTrace();
            log.info("======================>文件导出错误:" + e.getMessage());
            throw new BusinessException("======================>文件导出错误:" + e.getMessage());
        }
    }

    /******
     * 获取模板文件真实路径
     * @param isExport 是否导出操作
     * @param tmpFileName 模板文件名称(包含部分路径和文件后缀名)
     * @return
     */
    protected static String getTempFileRealPath(boolean isExport, String tmpFileName) {
        //初始定义模板公共路径
        String tempCommonPath;
        if (xzzfProperties.isTempFileMode()) {
            //获取服务器项目根路径
            tempCommonPath = new File(System.getProperty("user.dir")) + StrUtil.SLASH + JxlsConfig.BASE_PATH + StrUtil.SLASH + JxlsConfig.COMMON_PATH + StrUtil.SLASH;
        } else {
            //获取的项目项目打包后在target路径
            tempCommonPath = JxlsExcelView.class.getClassLoader().getResource("").getPath() + JxlsConfig.BASE_PATH + StrUtil.SLASH + JxlsConfig.COMMON_PATH + StrUtil.SLASH;
        }
        log.info("======================>公共模板路径信息:" + tempCommonPath);
        if (isExport) {
            //拼接模板文件绝对路径
            return tempCommonPath + JxlsConfig.OPERATE_TYPE_EXPORTS + StrUtil.SLASH + tmpFileName;
        } else {
            //拼接模板文件绝对路径
            return tempCommonPath + JxlsConfig.OPERATE_TYPE_IMPORTS + StrUtil.SLASH + tmpFileName;
        }
    }
}
