package com.kdgcsoft.jt.xzzf.common.excel.jxls;

import cn.hutool.core.util.StrUtil;
import cn.hutool.poi.excel.ExcelReader;
import cn.hutool.poi.excel.ExcelUtil;
import com.kdgcsoft.jt.xzzf.common.exception.BusinessException;
import com.kdgcsoft.jt.xzzf.common.util.PropertyUtil;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.io.InputStream;
import java.util.*;

/****
 * @className ExcelReadUtil
 * @author chris
 * @description excel工具类
 * @version 1.0
 * @date 2020-07-09 16:57
 */
public class ExcelReadUtil {
    private static Logger logger = LoggerFactory.getLogger(ExcelReadUtil.class);

    /*****
     * 校验上传的数据信息与当前模板是否匹配
     * @param reader
     * @param tempExcelName 系统中的导入模板名称
     * @return
     */
    public static boolean compareExcelTitle(ExcelReader reader, String tempExcelName) {
        return ExcelReadUtil.compareExcelTitle(reader, tempExcelName, JxlsConfig.OPERATE_TYPE_IMPORTS);
    }

    /*****
     * 校验上传的数据信息与当前模板是否匹配
     * @param reader
     * @param tempExcelName 系统中的导入模板名称
     * @param operateType 操作类型:exports:导出,imports-导入
     * @return
     */
    public static boolean compareExcelTitle(ExcelReader reader, String tempExcelName, String operateType) {
        String[] tempTitleArr = ExcelReadUtil.getTempExcelTitleList(tempExcelName, operateType);
        List<String> tempTitleList = Arrays.asList(tempTitleArr);
        List<Object> uploadTitleList = reader.readRow(0);
        logger.info("==========导入Excel的表头:" + uploadTitleList);
        return uploadTitleList.containsAll(tempTitleList);
    }


    /******
     * 获取模板表头信息并转数组
     * @param tempExcelName
     * @return
     */
    public static String[] getTempExcelTitleList(String tempExcelName) {
        return ExcelReadUtil.getTempExcelTitleList(tempExcelName, JxlsConfig.OPERATE_TYPE_IMPORTS);
    }

    /******
     * 获取模板表头信息并转数组
     * @param tempExcelName
     * @param operateType 操作类型:exports:导出,imports-导入
     * @return
     */
    public static String[] getTempExcelTitleList(String tempExcelName, String operateType) {
        String tempTitle = ExcelReadUtil.getTempExcelTitle(tempExcelName, operateType);
        String[] tempTitleArr = StrUtil.split(tempTitle.substring(1, tempTitle.length() - 1).replace(" ", ""), StrUtil.COMMA);
        return tempTitleArr;
    }

    /*****
     * 校验表头是否有重复
     * @param reader
     * @return true: 重复
     */
    public static boolean isRepeatTitle(ExcelReader reader) {
        List<Object> uploadTitleList = reader.readRow(0);
        return uploadTitleList.stream().distinct().count() < uploadTitleList.size();
    }

    /*****
     * 获取重复表头
     * @param reader
     * @return
     */
    public static List<Object> getRepeatTitle(ExcelReader reader) {
        List<Object> uploadTitleList = reader.readRow(0);
        // 上传Excel表头不能重复
        List<Object> repeats = new ArrayList<>();
        if (isRepeatTitle(reader)) {
            Map<Object, Object> map = new HashMap<>();
            for (Object str : uploadTitleList) {
                if (map.get(str) == null) {
                    map.put(str, str);
                } else {
                    repeats.add(str);
                }
            }
        }
        return repeats;
    }

    /******
     * 获取模板文件表头信息
     * @param tempExcelName 模板文件名称
     * @param operateType 操作类型:exports:导出,imports-导入
     * @return
     */
    public static String getTempExcelTitle(String tempExcelName, String operateType) {
        InputStream in = null;
        ExcelReader reader = null;
        try {
            boolean isExport = JxlsConfig.OPERATE_TYPE_EXPORTS.equals(operateType);
            String tempFilePath = StrUtil.SLASH + JxlsConfig.BASE_PATH + StrUtil.SLASH + JxlsConfig.COMMON_PATH + StrUtil.SLASH + operateType + StrUtil.SLASH + tempExcelName;
            System.out.println("===============模板文件路径:" + tempFilePath);
            in = PropertyUtil.class.getResourceAsStream(tempFilePath);
            reader = ExcelUtil.getReader(in, 0);
            List<List<Object>> readAll = reader.read(0, 0);
            logger.info("==========模板Excel的表头:" + readAll.get(0));
            //返回模板表头信息
            return readAll.get(0).toString();
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("Excel读取失败:" + e.getMessage());
            throw new BusinessException("Excel读取失败:" + e.getMessage());
        } finally {
            IOUtils.closeQuietly(reader);
            IOUtils.closeQuietly(in);
        }
    }

    /******
     * Excel数据转List
     * @param file 上传的文件信息
     * @param tempFilePath 模板文件信息
     * @return
     */
    public static List<Map<String, Object>> gainExcelInfo(MultipartFile file, Class clz, String tempFilePath) {
        ExcelReader reader = null;
        try {
            reader = ExcelUtil.getReader(file.getInputStream());
            boolean canImport = ExcelReadUtil.compareExcelTitle(reader, tempFilePath);
            if (!canImport) {
                throw new BusinessException("模板不匹配,请重新下载模板");
            }

            // 重复表头校验
            if (ExcelReadUtil.isRepeatTitle(reader)) {
                List<Object> repeats = ExcelReadUtil.getRepeatTitle(reader);
                throw new BusinessException("模板存在重复列：" + repeats.toString());
            }
            //返回数据信息
            return reader.readAll();
        } catch (IOException e) {
            e.printStackTrace();
            logger.error("Excel读取失败:" + e.getMessage());
            throw new BusinessException("Excel读取失败:" + e.getMessage());
        } finally {
            IOUtils.closeQuietly(reader);
        }
    }

    /******
     * Excel数据转List
     * @param tempFilePath 模板文件信息
     * @param file 上传的文件信息
     * @param beanType 待转换的实体类class
     * @return
     */
    public static List gainExcelInfoToList(String tempFilePath, MultipartFile file, Class beanType) {
        ExcelReader reader = null;
        try {
            reader = ExcelUtil.getReader(file.getInputStream());
            boolean canImport = ExcelReadUtil.compareExcelTitle(reader, tempFilePath);
            if (!canImport) {
                throw new BusinessException("模板不匹配,请重新下载模板");
            }

            // 重复表头校验
            if (ExcelReadUtil.isRepeatTitle(reader)) {
                List<Object> repeats = ExcelReadUtil.getRepeatTitle(reader);
                throw new BusinessException("模板存在重复列：" + repeats.toString());
            }

            //返回去掉表头和样例数据的数据信息
            return reader.read(0, 3, beanType);
        } catch (IOException e) {
            e.printStackTrace();
            logger.error("Excel读取失败:" + e.getMessage());
            throw new BusinessException("Excel读取失败:" + e.getMessage());
        } finally {
            IOUtils.closeQuietly(reader);
        }
    }
}