/*******************************************************************************
 * Copyright (c) 2005, 2014 springside.github.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 *******************************************************************************/
package com.cpi.framework.modules.persistence;

import java.sql.Connection;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Hibernate;
import org.hibernate.dialect.DB2Dialect;
import org.hibernate.dialect.H2Dialect;
import org.hibernate.dialect.MySQL5InnoDBDialect;
import org.hibernate.dialect.Oracle10gDialect;
import org.hibernate.dialect.PostgreSQL82Dialect;
import org.hibernate.dialect.SQLServer2008Dialect;
import org.springframework.jdbc.core.JdbcTemplate;

import com.cpi.framework.dao.key.UniqueTableApp;
import com.cpi.framework.dao.key.impl.DefaultUniqueTableApp;
import com.cpi.framework.dao.key.impl.MSSQLServerUniqueTableApp;
import com.cpi.framework.dao.key.impl.MySqlUniqueTableApp;
import com.cpi.framework.dao.key.impl.OracleUniqueTableApp;

public class Hibernates {
	/**
	 * Initialize the lazy property value.
	 * 
	 * e.g. Hibernates.initLazyProperty(user.getGroups());
	 */
	public static void initLazyProperty(Object proxyedPropertyValue) {
		Hibernate.initialize(proxyedPropertyValue);
	}

	/**
	 * 从DataSoure中取出connection, 根据connection的metadata中的jdbcUrl判断Dialect类型.
	 * 仅支持Oracle, H2, MySql, PostgreSql, SQLServer，如需更多数据库类型，请仿照此类自行编写。
	 */
	public static String getDialect(DataSource dataSource) {
		String jdbcUrl = getJdbcUrlFromDataSource(dataSource);
		// 根据jdbc url判断dialect
		if (StringUtils.contains(jdbcUrl, ":h2:")) {
			return H2Dialect.class.getName();
		} else if (StringUtils.contains(jdbcUrl, ":mysql:")) {
			return MySQL5InnoDBDialect.class.getName();
		} else if (StringUtils.contains(jdbcUrl, ":oracle:")) {
			return Oracle10gDialect.class.getName();
		} else if (StringUtils.contains(jdbcUrl, ":postgresql:")) {
			return PostgreSQL82Dialect.class.getName();
		} else if (StringUtils.contains(jdbcUrl, ":sqlserver:")) {
			return SQLServer2008Dialect.class.getName();
		} else if (StringUtils.contains(jdbcUrl, ":db2:")) {
			return DB2Dialect.class.getName();
		} else {
			throw new IllegalArgumentException("Unknown Database of " + jdbcUrl);
		}
	}

	/**
	 * 从DataSoure中取出connection, 根据connection的metadata中的jdbcUrl判断数据库类型并返回数据库主键生成策略对象.
	 * 仅支持Oracle, H2, MySql, PostgreSql, SQLServer，如需更多数据库类型，请仿照此类自行编写。
	 */
	public static UniqueTableApp getUniqueTableApp(JdbcTemplate jdbcTemplate) {
		String jdbcUrl = getJdbcUrlFromDataSource(jdbcTemplate.getDataSource());
		// 根据jdbc url判断dialect
		if (StringUtils.contains(jdbcUrl, ":h2:")) {
			return new DefaultUniqueTableApp(jdbcTemplate);
		} else if (StringUtils.contains(jdbcUrl, ":mysql:")) {
			return new MySqlUniqueTableApp(jdbcTemplate);
		} else if (StringUtils.contains(jdbcUrl, ":oracle:")) {
			return new OracleUniqueTableApp(jdbcTemplate);
		} else if (StringUtils.contains(jdbcUrl, ":postgresql:")) {
			return new DefaultUniqueTableApp(jdbcTemplate);
		} else if (StringUtils.contains(jdbcUrl, ":sqlserver:")) {
			return new  MSSQLServerUniqueTableApp(jdbcTemplate);
		} else if (StringUtils.contains(jdbcUrl, ":db2:")) {
			return new DefaultUniqueTableApp(jdbcTemplate);
		}else {
			throw new IllegalArgumentException("Unknown Database of " + jdbcUrl);
		}
	}
	
	private static String getJdbcUrlFromDataSource(DataSource dataSource) {
		Connection connection = null;
		try {
			connection = dataSource.getConnection();
			if (connection == null) {
				throw new IllegalStateException("Connection returned by DataSource [" + dataSource + "] was null");
			}
			return connection.getMetaData().getURL();
		} catch (SQLException e) {
			throw new RuntimeException("Could not get database url", e);
		} finally {
			if (connection != null) {
				try {
					connection.close();
				} catch (SQLException e) {
				}
			}
		}
	}
}
