package org.violet.common.mybatis.plugin;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.SystemClock;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.toolkit.PluginUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.executor.statement.StatementHandler;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.mapping.ParameterMapping;
import org.apache.ibatis.plugin.*;
import org.apache.ibatis.reflection.MetaObject;
import org.apache.ibatis.reflection.SystemMetaObject;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.type.TypeHandlerRegistry;

import java.sql.Statement;
import java.text.DateFormat;
import java.util.*;

@Intercepts({@Signature(
    type = StatementHandler.class,
    method = "query",
    args = {Statement.class, ResultHandler.class}
), @Signature(
    type = StatementHandler.class,
    method = "update",
    args = {Statement.class}
), @Signature(
    type = StatementHandler.class,
    method = "batch",
    args = {Statement.class}
)})
@Slf4j
public class PerformanceInterceptor implements Interceptor {
    private long slowSqlMillis = 1000L;
    private boolean logSql = true;
    private boolean logOnlySlowSql = true;
    private boolean printSql = true;
    private static final String SQL_FMT = "\n┌ ID    : {sqlId}\n├ SQL   : {sql}\n├ Params: {params}\n└ Type:{type},Result:{result},Times:{times}.";

    public PerformanceInterceptor() {
    }

    public Object intercept(Invocation invocation) throws Throwable {
        long start = SystemClock.now();
        Object result = invocation.proceed();
        long timing = SystemClock.now() - start;
        if (!this.printSql && !this.logSql) {
            return result;
        } else {
            boolean isSlowSql = this.slowSqlMillis > 0L && timing > this.slowSqlMillis;
            String fmtSql = this.buildSql(result, invocation, timing);
            if (this.printSql) {
                System.err.println(fmtSql);
            }

            if (this.logSql) {
                if (!this.logOnlySlowSql) {
                    log.info(fmtSql);
                } else if (this.logOnlySlowSql && isSlowSql) {
                    log.info(fmtSql);
                }
            }

            return result;
        }
    }

    public String buildSql(Object result, Invocation invocation, long timing) {
        String resultInfo = "";
        if (result != null) {
            if (result instanceof List) {
                List list = (List)result;
                resultInfo = list.size() + " rows";
            } else if (result instanceof Map) {
                Map map = (Map)result;
                resultInfo = map.size() + " rows";
            } else {
                resultInfo = result.getClass().getSimpleName();
            }
        }

        StatementHandler statementHandler = (StatementHandler) PluginUtils.realTarget(invocation.getTarget());
        Object target = PluginUtils.realTarget(invocation.getTarget());
        MetaObject metaObject = SystemMetaObject.forObject(target);
        MappedStatement ms = (MappedStatement)metaObject.getValue("delegate.mappedStatement");
        BoundSql boundSql = statementHandler.getBoundSql();
        Map<String, Object> fmtMap = new HashMap();
        fmtMap.put("sqlId", ms.getId());
        fmtMap.put("sql", StrUtil.replace(boundSql.getSql(), "\n", " "));
        fmtMap.put("params", this.getParamsString(boundSql, ms));
        fmtMap.put("type", ms.getSqlCommandType().name());
        fmtMap.put("result", resultInfo);
        fmtMap.put("times", timing + "ms");
        return StrUtil.format("\n┌ ID    : {sqlId}\n├ SQL   : {sql}\n├ Params: {params}\n└ Type:{type},Result:{result},Times:{times}.", fmtMap);
    }

    public Object plugin(Object target) {
        return target instanceof StatementHandler ? Plugin.wrap(target, this) : target;
    }

    public void setProperties(Properties prop) {
    }

    private String getParamsString(BoundSql boundSql, MappedStatement ms) {
        List<ParameterMapping> parameterMappings = boundSql.getParameterMappings();
        Object parameterObject = boundSql.getParameterObject();
        List<String> paramValuesList = new ArrayList();
        if (parameterMappings.size() > 0 && parameterObject != null) {
            TypeHandlerRegistry typeHandlerRegistry = ms.getConfiguration().getTypeHandlerRegistry();
            if (typeHandlerRegistry.hasTypeHandler(parameterObject.getClass())) {
                paramValuesList.add(this.getParameterValue(parameterObject));
            } else {
                MetaObject metaObject = ms.getConfiguration().newMetaObject(parameterObject);
                Iterator var8 = parameterMappings.iterator();

                while(var8.hasNext()) {
                    ParameterMapping parameterMapping = (ParameterMapping)var8.next();
                    String propertyName = parameterMapping.getProperty();
                    Object obj;
                    if (metaObject.hasGetter(propertyName)) {
                        obj = metaObject.getValue(propertyName);
                        paramValuesList.add(this.getParameterValue(obj));
                    } else if (boundSql.hasAdditionalParameter(propertyName)) {
                        obj = boundSql.getAdditionalParameter(propertyName);
                        paramValuesList.add(this.getParameterValue(obj));
                    }
                }
            }
        }

        return CollUtil.join(paramValuesList, ",");
    }

    private String getParameterValue(Object obj) {
        String value = null;
        if (obj instanceof String) {
            value = "'" + obj.toString() + "'";
        } else if (obj instanceof Date) {
            DateFormat formatter = DateFormat.getDateTimeInstance(2, 2, Locale.CHINA);
            String var10000 = formatter.format(obj);
            value = "to_date('" + var10000 + "','yyyy-MM-dd hh24:mi:ss')";
        } else if (obj != null) {
            value = obj.toString();
        } else {
            value = "";
        }

        return value;
    }

    public PerformanceInterceptor setSlowSqlMillis(final long slowSqlMillis) {
        this.slowSqlMillis = slowSqlMillis;
        return this;
    }

    public long getSlowSqlMillis() {
        return this.slowSqlMillis;
    }

    public PerformanceInterceptor setLogSql(final boolean logSql) {
        this.logSql = logSql;
        return this;
    }

    public boolean isLogSql() {
        return this.logSql;
    }

    public void setLogOnlySlowSql(final boolean logOnlySlowSql) {
        this.logOnlySlowSql = logOnlySlowSql;
    }

    public boolean isLogOnlySlowSql() {
        return this.logOnlySlowSql;
    }

    public void setPrintSql(final boolean printSql) {
        this.printSql = printSql;
    }

    public boolean isPrintSql() {
        return this.printSql;
    }
}