package org.violet.common.launch.entity;


import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson2.JSON;
import com.fasterxml.jackson.annotation.JsonInclude;


import java.io.Serializable;
import java.util.function.Consumer;
import java.util.function.Supplier;

/**
 * 全局统一返回实体
 * 始终返回全部属性，无论属性值是否为null
 *
 * @param <T>
 */
@JsonInclude(JsonInclude.Include.ALWAYS)
public class JsonResult<T> implements Serializable {
    private Boolean success = true;
    private int code = 200;
    private String msg = "操作成功";
    private T data;

    public JsonResult() {
    }

    public JsonResult(boolean success, String message) {
        this.success = success;
        this.msg = message;
    }

    public static JsonResult OK() {
        return OK("操作成功");
    }

    public static JsonResult OK(String message) {
        return new JsonResult(true, message);
    }

    public static JsonResult OK(String messageTemplate, Object... params) {
        String message = StrUtil.format(messageTemplate, params);
        return OK(message);
    }

    public static JsonResult ERROR() {
        return ERROR("操作失败");
    }

    public static JsonResult ERROR(String message) {
        return new JsonResult(false, message);
    }

    public static JsonResult ERROR(String messageTemplate, Object... params) {
        String message = StrUtil.format(messageTemplate, params);
        return ERROR(message);
    }

    public JsonResult<T> data(T data) {
        this.data = data;
        return this;
    }

    public JsonResult msg(String msg) {
        this.msg = msg;
        return this;
    }

    public JsonResult code(int code) {
        this.code = code;
        return this;
    }

    public JsonResult setSuccess(Boolean success) {
        this.success = success;
        return this;
    }

    public JsonResult ifSuccessOfElse(Consumer<JsonResult> successAction, Consumer<JsonResult> errorAction) {
        if (this.success) {
            successAction.accept(this);
        } else {
            errorAction.accept(this);
        }

        return this;
    }

    public JsonResult successThen(Supplier<? extends JsonResult> supplier) {
        return this.success ? (JsonResult) supplier.get() : this;
    }

    public JsonResult errorThen(Supplier<JsonResult> supplier) {
        return !this.success ? (JsonResult) supplier.get() : this;
    }

    public String toJsonString() {
        return JSON.toJSONString(this);
    }

    public String toString() {
        return JSON.toJSONString(this);
    }

    public void setCode(final int code) {
        this.code = code;
    }

    public void setMsg(final String msg) {
        this.msg = msg;
    }

    public void setData(final T data) {
        this.data = data;
    }

    public Boolean getSuccess() {
        return this.success;
    }

    public int getCode() {
        return this.code;
    }

    public String getMsg() {
        return this.msg;
    }

    public T getData() {
        return this.data;
    }
}