package org.violet.common.datascope.interceptor;

import com.baomidou.mybatisplus.core.toolkit.PluginUtils;
import com.baomidou.mybatisplus.core.toolkit.StringPool;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.mapping.SqlCommandType;
import org.apache.ibatis.mapping.StatementType;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;
import org.violet.common.core.embed.dict.CommonDataAuth;
import org.violet.common.core.entity.LoginUser;
import org.violet.common.core.enums.LoginUserType;
import org.violet.common.core.util.ClassUtil;
import org.violet.common.core.util.SpringUtil;
import org.violet.common.core.util.StringUtil;
import org.violet.common.datascope.annotations.DataScope;
import org.violet.common.datascope.config.DataScopeProperties;
import org.violet.common.datascope.handler.DataScopeHandler;
import org.violet.common.mybatis.intercept.QueryInterceptor;
import org.violet.common.security.util.SecurityUtil;

import java.lang.reflect.Method;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;


@RequiredArgsConstructor
@Slf4j
@SuppressWarnings({"rawtypes"})
public class DataScopeInterceptor implements QueryInterceptor {
    //注解缓存，提升性能
    private final ConcurrentMap<String, DataScope> dataAuthMap = new ConcurrentHashMap<>(8);

    private final DataScopeHandler dataScopeHandler;
    private final DataScopeProperties dataScopeProperties;

    @Override
    public void intercept(Executor executor, MappedStatement ms, Object parameter, RowBounds rowBounds, ResultHandler resultHandler, BoundSql boundSql) {
        //未启用则放行
        if (!dataScopeProperties.getEnabled()) {
            return;
        }

        //未取到用户则放行
        LoginUser longinUser = SecurityUtil.getLoginUser();
        if (longinUser == null) {
            return;
        }

        if (SqlCommandType.SELECT != ms.getSqlCommandType() || StatementType.CALLABLE == ms.getStatementType()) {
            return;
        }

        String originalSql = boundSql.getSql();
        //查找注解
        DataScope dataScope = findDataScopeAnnotation(ms);
        //查找mapper是否存在关键字或者是排除的mapper
        String mapperId = ms.getId();
        String className = mapperId.substring(0, mapperId.lastIndexOf(StringPool.DOT));
        String mapperName = ClassUtil.getShortName(className);
        String methodName = mapperId.substring(mapperId.lastIndexOf(StringPool.DOT) + 1);
        boolean mapperSkip = dataScopeProperties.getMapperExclude().stream().anyMatch(mapperName::contains);
        //注解为空或者数据权限方法名未匹配到,则放行
        if (dataScope == null || mapperSkip) {
            return;
        }
        //如果是所有数据权限则放行
        if (dataScope.dataAuth() == CommonDataAuth.ALL_DATA) {
            return;
        }
        //判断是否超级管理员,超级管理员无需数据权限,放行
        if (longinUser.getLoginUserType() == LoginUserType.ROOT) {
            return;
        }
        //获取数据权限规则对应的筛选Sql
        String sqlCondition = dataScopeHandler.sqlCondition(mapperId, dataScope, longinUser, originalSql);
        if (!StringUtil.isBlank(sqlCondition)) {
            PluginUtils.MPBoundSql mpBoundSql = PluginUtils.mpBoundSql(boundSql);
            mpBoundSql.sql(sqlCondition);
        }

    }

    //查找注解
    private DataScope findDataScopeAnnotation(MappedStatement ms) {

        String id = ms.getId();
        return dataAuthMap.computeIfAbsent(id, (key) -> {
            String className = key.substring(0, key.lastIndexOf(StringPool.DOT));
            String mapperBean = StringUtil.firstCharToLower(ClassUtil.getShortName(className));
            Object mapper = SpringUtil.getBean(mapperBean);
            String methodName = key.substring(key.lastIndexOf(StringPool.DOT) + 1);
            Class<?>[] interfaces = ClassUtil.getAllInterfaces(mapper);
            for (Class<?> mapperInterface : interfaces) {
                for (Method method : mapperInterface.getDeclaredMethods()) {
                    if (methodName.equals(method.getName()) && method.isAnnotationPresent(DataScope.class)) {
                        return method.getAnnotation(DataScope.class);
                    }
                }
            }
            return null;
        });
    }

}
