package org.violet.common.cloud.http;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import okhttp3.ConnectionPool;
import okhttp3.OkHttpClient;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.violet.common.cloud.feign.header.VioletFeignHeadersProperties;
import org.violet.common.cloud.http.client.OkHttpSlf4jLogger;
import org.violet.common.core.ssl.DisableValidationTrustManager;
import org.violet.common.core.ssl.TrustAllHostNames;
import org.violet.common.core.util.Holder;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.TimeUnit;

/***
 * 更换feign调用客户端为okhttp3client
 * @author wjx
 * @date 2024/11/28 17:35
 */
@AllArgsConstructor
@Slf4j
@AutoConfiguration
@ConditionalOnClass(OkHttpClient.class)
@ConditionalOnProperty(value = "violet.http.enabled", matchIfMissing = true)
@EnableConfigurationProperties({VioletFeignHeadersProperties.class, VioHttpProperties.class})
public class VioletHttpAutoConfiguration {
    private final VioHttpProperties properties;

    /**
     * okhttp3 请求日志拦截器
     *
     * @return HttpLoggingInterceptor
     */
    @Bean
    public HttpLoggingInterceptor loggingInterceptor() {
        HttpLoggingInterceptor interceptor = new HttpLoggingInterceptor(new OkHttpSlf4jLogger());
        interceptor.setLevel(properties.getLevel());
        return interceptor;
    }

    /**
     * okhttp3 链接池配置
     *
     * @return okhttp3.ConnectionPool
     */
    @Bean
    @ConditionalOnMissingBean
    public ConnectionPool httpClientConnectionPool() {
        int maxTotalConnections = properties.getMaxConnections();
        long timeToLive = properties.getTimeToLive();
        TimeUnit ttlUnit = properties.getTimeUnit();
        return new ConnectionPool(maxTotalConnections, timeToLive, ttlUnit);
    }


    /**
     * 配置OkHttpClient
     *
     * @param connectionPool 链接池配置
     * @param interceptor    拦截器
     * @return OkHttpClient
     */
    @Bean
    @ConditionalOnMissingBean
    public OkHttpClient okHttpClient(ConnectionPool connectionPool, HttpLoggingInterceptor interceptor) {
        boolean followRedirects = properties.isFollowRedirects();
        int connectTimeout = properties.getConnectionTimeout();
        return this.createBuilder(properties.isDisableSslValidation())
                .connectTimeout(connectTimeout, TimeUnit.MILLISECONDS)
                .writeTimeout(30, TimeUnit.SECONDS)
                .readTimeout(30, TimeUnit.SECONDS)
                .followRedirects(followRedirects)
                .connectionPool(connectionPool)
                .addInterceptor(interceptor)
                .build();
    }

    private OkHttpClient.Builder createBuilder(boolean disableSslValidation) {
        OkHttpClient.Builder builder = new OkHttpClient.Builder();
        if (disableSslValidation) {
            try {
                X509TrustManager disabledTrustManager = DisableValidationTrustManager.INSTANCE;
                TrustManager[] trustManagers = new TrustManager[]{disabledTrustManager};
                SSLContext sslContext = SSLContext.getInstance("SSL");
                sslContext.init(null, trustManagers, Holder.SECURE_RANDOM);
                SSLSocketFactory disabledSslSocketFactory = sslContext.getSocketFactory();
                builder.sslSocketFactory(disabledSslSocketFactory, disabledTrustManager);
                builder.hostnameVerifier(TrustAllHostNames.INSTANCE);
            } catch (NoSuchAlgorithmException | KeyManagementException e) {
                log.warn("Error setting SSLSocketFactory in OKHttpClient", e);
            }
        }
        return builder;
    }


}
