package com.kdgcsoft.power.wmfconverter;


import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.GZIPOutputStream;

import javax.imageio.ImageIO;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.batik.transcoder.TranscoderInput;
import org.apache.batik.transcoder.TranscoderOutput;
import org.apache.batik.transcoder.image.ImageTranscoder;
import org.apache.batik.transcoder.image.JPEGTranscoder;
import org.apache.batik.transcoder.image.PNGTranscoder;
import org.freehep.graphicsio.emf.EMFInputStream;
import org.freehep.graphicsio.emf.EMFRenderer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import net.arnx.wmf2svg.gdi.svg.SvgGdi;
import net.arnx.wmf2svg.gdi.wmf.WmfParser;

/**
 * 
 * WMF、EMF、SVG转PNG工具，参考 https://blog.csdn.net/www1056481167/article/details/56835043
 */
public class WmfConverter {
	
	private static final Logger logger = LoggerFactory.getLogger(WmfConverter.class);
	
	private WmfConverter(){}

	/**
	 * emf转换为png图片格式
	 * 
	 * @param is emf输入流
	 * @param outPngStream 要输出的目标png数据流
	 * @param width 目标输出宽度，像素单位
	 * @throws IOException 
	 */
	public static void emf2PngStream(final InputStream is, final OutputStream outPngStream, final int width) throws IOException {
		try (EMFInputStream eis = new EMFInputStream(is, EMFInputStream.DEFAULT_VERSION)) {
			
			EMFRenderer emfRenderer = new EMFRenderer(eis);
			final int srcWidth = (int) eis.readHeader().getBounds().getWidth();
			final int srcHeight = (int) eis.readHeader().getBounds().getHeight();
			// 设置图片的大小和样式
//			final BufferedImage result = new BufferedImage(width, (int)((1.0f*srcHeight/srcWidth)*width), BufferedImage.TYPE_4BYTE_ABGR);
			final BufferedImage result = new BufferedImage(srcWidth + 60, srcHeight + 40, BufferedImage.TYPE_4BYTE_ABGR);
			Graphics2D g2 = (Graphics2D) result.createGraphics();
			emfRenderer.paint(g2);

			// 写入到磁盘中(格式设置为png背景不会变为橙色)
			ImageIO.write(result, "png", outPngStream);
		} catch (IOException e) {
			logger.error("emf转png失败!", e);
		} finally {
			try {
				is.close();
			} catch ( Exception e ) {
	            logger.error("Unable to close resource: ", e);
	        }
		}
	}
	
	/**
	 * emf转换为png图片格式
	 * 
	 * @param is emf输入流
	 * @param outPngFile 要输出的目标png图片
	 * @param width 目标输出宽度，像素单位
	 * @throws IOException 
	 */
	public static void emf2Png(final InputStream is, final File outPngFile, final int width) throws IOException {
		try (FileOutputStream outStream = new FileOutputStream(outPngFile);) {
			emf2PngStream(is, outStream, width);
		}
	}
	
	/**
	 * emf转换为png图片格式
	 * 
	 * @param in emf字节数组
	 * @param outPngFile 要输出的目标png图片
	 * @return width 目标输出宽度，像素单位
	 * @throws IOException 
	 */
	public static void emf2Png(final byte[] in, final File outPngFile, final int width) throws IOException {
		try (ByteArrayInputStream inStream = new ByteArrayInputStream(in)) {
			emf2Png(inStream, outPngFile, width);
		}
	}
	
	/**
	 * wmf转换为png，输出为流
	 * 
	 * @param is wmf输入流
	 * @param pngOutStream 要向其中写入转换后png图片的目标数据流
	 * @param width 希望输出的图片宽度，像素
	 * @throws Exception 转换失败
	 */
	public static void wmf2PngStream(final InputStream is, final OutputStream pngOutStream, final int width) throws Exception {
			try {
//				// 使用管道连接svg数据的输入输出流。实测速度比使用临时文件的方式快。
//				// 因为代码比较复杂，对整体性能提升有限，不使用。
//				final PipedOutputStream svgOutStream = new PipedOutputStream();
//				final PipedInputStream svgInStream = new PipedInputStream();
//				svgInStream.connect(svgOutStream);
//				// 将wmf转svg
//				new Thread(new Runnable() {
//
//					@Override
//					public void run() {
//						try {
//							wmfToSvgStream(is, svgOutStream, false);
//							svgOutStream.close();
//						} catch (IOException e) {
//							logger.error("wmf转svg失败", e);
//						}
//					}
//					
//				}).start();
//				
//				svgToPngStream(svgInStream, pngOutStream, width);
//				svgInStream.close();
//				return true;
				/////////////////////////////////////////////////////////////
				
				// 使用svg临时文件的转换方式
				File svgFile = File.createTempFile("e2p_", ".svg");
				// 将wmf转svg
				wmfToSvg(is, svgFile);
				// 将svg转png
				try (InputStream svgInStream = new FileInputStream(svgFile)) {
					svgToPngStream(svgInStream, pngOutStream, width );
				}
			} catch (IOException e) {
				logger.error("转换WMF到PNG失败", e);
				throw e;
			}
	}
	
	/**
	 * wmf转换为png图片格式
	 * 
	 * @param is wmf输入流
	 * @param outPngFile 要输出的目标png图片
	 * @param width 希望输出的图片宽度，像素
	 * @throws Exception 
	 */
	public static void wmf2Png(final InputStream is, final File outPngFile, final int width) throws Exception {
		try (OutputStream svgOutStream = new FileOutputStream(outPngFile)) {
			wmf2PngStream(is, svgOutStream, width);
		} catch (FileNotFoundException e) {
			logger.error("转换WMF到PNG失败", e);
			throw e;
		}
		
//		// 创建临时文件
//		File svgFile;
//		try {
//			svgFile = File.createTempFile("e2p_", ".svg");
//			// 将wmf转svg
//			if (wmfToSvg(is, svgFile)) {
//				// 将svg转png
////				String pngFile = svgFile + ".png";
//				if (svgToPng(svgFile, outPngFile, width)){
//					// 删除svg文件
//					System.out.println(svgFile);
//					svgFile.delete();
//					return true;
//				}
//			}
//		} catch (IOException e) {
//			logger.error("转换WMF到PNG失败", e);
//		}
//		
//		return false;

	}
	
	/**
	 * wmf转换为png图片格式
	 * 
	 * @param in wmf字节数组
	 * @param outPngFile 要输出的目标png图片
	 * @param width 希望输出的图片宽度，像素
	 * @throws Exception 
	 */
	public static void wmf2Png(final byte[] in, File outPngFile, final int width) throws Exception {
		ByteArrayInputStream inStream = new ByteArrayInputStream(in);
		wmf2Png(inStream, outPngFile, width);
	}

	/**
	 * 将wmf转换为svg
	 * @param in wmf字节流
	 * @param dest 要输出的svg文件名
	 * @return 是否转换成功
	 * @throws Exception 
	 */
	public static void wmfToSvg(InputStream in, File dest) throws Exception {
		try (OutputStream svgOutStream = new FileOutputStream(dest);){
			boolean isCompress = false;
			if (dest.getAbsolutePath().endsWith(".svgz")) {
				isCompress = true;
			}
			wmfToSvgStream(in, svgOutStream, isCompress);
		}
	}
	
	/**
	 * 将wmf输入流转换为svg输出流
	 * @param in wmf字节流
	 * @param svgOutStream 要输出的svg字节流
	 * @param isCompress 是否要转换成.svgz，压缩格式的svg
	 * @throws Exception 
	 */
	public static void wmfToSvgStream(InputStream in, OutputStream svgOutStream, boolean isCompress) throws Exception {
		boolean compatible = false;
		Document doc = null;
		try {
			WmfParser parser = new WmfParser();
			final SvgGdi gdi = new SvgGdi(compatible);
			parser.parse(in, gdi);

			doc = gdi.getDocument();
			if (isCompress) {
				svgOutStream = new GZIPOutputStream(svgOutStream);
			}
			outputSvgFile(doc, svgOutStream);
		} catch (Exception e) {
			logger.error("转换WMF到SVG失败", e);
			throw e;
		}
	}

	/**
	 * 输出svg数据流
	 * @param doc svg的xml对象
	 * @param out 目标svg文件的输出流
	 * @throws TransformerException 
	 * @throws Exception
	 */
	private static void outputSvgFile(Document doc, OutputStream out) throws TransformerException  {
		TransformerFactory factory = TransformerFactory.newInstance();
		Transformer transformer = factory.newTransformer();
		transformer.setOutputProperty(OutputKeys.METHOD, "xml");
		transformer.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
		transformer.setOutputProperty(OutputKeys.INDENT, "yes");
		transformer.setOutputProperty(OutputKeys.DOCTYPE_PUBLIC, "-//W3C//DTD SVG 1.0//EN");
		transformer.setOutputProperty(OutputKeys.DOCTYPE_SYSTEM,
				"http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd");
		transformer.transform(new DOMSource(doc), new StreamResult(out));
	}

	/**
	 * svg文件转换为png文件。转换结束会删除svg文件。
	 * @param svgFile 源svg文件
	 * @param outPngFile 输出目标png文件路径
	 * @param width 转换目标宽度
	 * @throws Exception 
	 */
	public static void svgToPng(final File svgFile, final File outPngFile, final int width) throws Exception {
		try(FileInputStream svgInputStream = new FileInputStream(svgFile);
			 FileOutputStream pngOutStream = new FileOutputStream(outPngFile)	) {
			
			svgToPngStream(svgInputStream, pngOutStream, width);
		}
	}
	
	/**
	 * svg数据流转换为png数据流。
	 * <br>注意本函数不会关闭输入和输出流。这需要调用者完成。
	 * @param svgInputStream 源svg二进制数据流
	 * @param pngOutStream 要写入的输出数据流
	 * @param width 转换目标宽度
	 * @throws Exception 
	 */
	public static void svgToPngStream(final InputStream svgInputStream, final OutputStream pngOutStream, final int width) throws Exception {
		ImageTranscoder it = new PNGTranscoder();
		it.addTranscodingHint(JPEGTranscoder.KEY_QUALITY, new Float(1f));
		it.addTranscodingHint(ImageTranscoder.KEY_WIDTH, new Float(width));

		it.transcode(new TranscoderInput(svgInputStream), new TranscoderOutput(pngOutStream));
		return;
	}
}
