package com.kdgcsoft.power.filestore.strategy;

import java.util.UUID;

import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.filestore.FileStoreException;

public class UUIDKeyStrategy implements IKeyStrategy {
	
	private static final Logger logger = LoggerFactory.getLogger(UUIDKeyStrategy.class);

	/**
	 * 基于uuid的首字母建立目录层级，方便以后用uuid定位查找文件
	 * 目前路径使用uuid的前两个字符分别建立两级目录，
	 * 即每级目录名为0~F共16个，所有文件总共分配在256个目录下存储。
	 * @param key uuid
	 * @return 目录层级字符串，形如"a/b"
	 */
	@Override
	public String getRelativePathByKey(String key) throws FileStoreException {
		try {
			UUID.fromString(key);
		} catch (IllegalArgumentException e) {
			logger.error("不是合法的UUID:{}", key);
			throw new FileStoreException("不是合法的UUID:" + key, e);
		}
		
		return "" + key.charAt(0) + "/" + key.charAt(1);
	}

	/**
	 * 使用UUID作为文件Key
	 * @see com.kdgcsoft.power.filestore.strategy.IKeyStrategy#generateKey()
	 */
	@Override
	public String generateKey() {
		return UUID.randomUUID().toString();
	}

	@Override
	public String generateStoreFileName(String key, String originFileName) {
		String filename;
		// 决定最终的文件名
		if (originFileName == null || "".equals(originFileName)) {
			filename = "";
		} else {
			filename = FilenameUtils.getName(originFileName);
		}
		filename = key + "_" + filename;
		return filename;
	}
	
	@Override
	public String extractOriginalFileName(String key, String storeFileName) {
		return storeFileName.substring(key.length() + 1);
	}
	
	@Override
	public String extractKey(String storeFileName) {
		return storeFileName.substring(0, storeFileName.indexOf('_'));
	}
}
