package com.kdgcsoft.power.filestore.strategy;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.commons.io.FilenameUtils;

import com.kdgcsoft.power.filestore.FileStoreException;

public class TimeStampKeyStrategy implements IKeyStrategy {
	
	public final static String KEY_DATE_FORMAT = "yyyyMMddHHmmss";
	public final static int KEY_LENGTH = KEY_DATE_FORMAT.length() +4;
	
	/**
	 * 原子计数器，在同一秒内递增
	 */
	private AtomicInteger id = new AtomicInteger(0);
	/**
	 * 记录上一次产生计数的秒值
	 */
	private long lastSecond = 0;

	/**
	 * 基于时间创建三级相对目录结构，层级为年(4位数字)/月(2位数字)/日(2位数字)
	 * @see com.kdgcsoft.power.filestore.strategy.IKeyStrategy#getRelativePathByKey(java.lang.String)
	 */
	@Override
	public String getRelativePathByKey(String key) throws FileStoreException {
		StringBuilder buf = new StringBuilder(10);
		buf.append(key.subSequence(0, 4)).append("/")
				.append(key.subSequence(4, 6)).append("/")
				.append(key.subSequence(6, 8));
		return buf.toString();
	}

	/** 
	 * 使用时间戳作为文件Key，精确到秒，后面添加四位数的计数
	 * @see com.kdgcsoft.power.filestore.strategy.IKeyStrategy#generateKey()
	 */
	@Override
	public String generateKey() {
		Date date = new Date();
		// 在同一秒内循环计数
		long curSecond = date.getTime() / 1000;
		if (curSecond != lastSecond) {
			id.set(0);
			lastSecond = curSecond;
		}
		SimpleDateFormat sdf = new SimpleDateFormat(KEY_DATE_FORMAT);
		return sdf.format(date) + String.format("%04d", getNextSeq());
	}
	
	
	
	/**
	 * 获取全局唯一的序列数，自动递增。当超过100000时会重新从1开始
	 * @return 下一个序列数
	 */
	private int getNextSeq() {
		if (id.get() >= 10000) {
			id.set(0);
		}
		return id.incrementAndGet();
	}

	@Override
	public String generateStoreFileName(String key, String originFileName) {
		String filename;
		// 决定最终的文件名
		if (originFileName == null || "".equals(originFileName)) {
			filename = "";
		} else {
			filename = FilenameUtils.getName(originFileName);
		}
		filename = key + "_" + filename;
		return filename;
	}

	@Override
	public String extractOriginalFileName(String key, String storeFileName) {
		return storeFileName.substring(key.length() + 1);
	}

	@Override
	public String extractKey(String storeFileName) {
		return storeFileName.substring(0, KEY_LENGTH);
	}

}
