package com.kdgcsoft.power.filestore.strategy;

import com.kdgcsoft.power.filestore.FileStoreException;

/**
 * 文件存储Key策略的接口定义。该接口定义了文件Key的生成接口，以及文件Key与实际的FileStore存储结构之间的映射接口。<br>
 * 对于不使用内置数据库等较高级的存储元数据管理方式的FileStore实现来说，文件Key作为唯一接口参数，通常会与存储结构之间存在明确的对应关系。
 * 例如当使用时间戳作为文件Key且文件存储在磁盘上时，通常会建立年月日多级存储结构，文件Key与存储结构之间存在相关性。
 * 本接口是这些对应关系的抽象，从而实现FileStore的具体存储结构与文件Key格式之间的无关性。
 * @author hling
 *
 */
public interface IKeyStrategy {

	/**
	 * 根据本策略生成的Key，计算用于存储的相对路径。路径形如"a/b/c"，开头和结尾不含路径分隔符。
	 * 计算出来的相对路径，可以被FileStore的具体实现类使用，用于其内部存储管理。
	 * 但FileStore实现类也可以不使用此函数而自行组织内部存储结构。
	 * @param key 通过 {@link #generateKey()}得到的Key
	 * @return 根据Key计算得到的路径
	 * @throws FileStoreException 仓库异常
	 */
	String getRelativePathByKey(final String key)  throws FileStoreException;
	
	/**
	 * 生成一个Key。算法应确保对同一个策略实例，每次调用都不会重复。
	 * @return Key
	 */
	String generateKey();
	
	/**
	 * 基于Key和原始文件名，生成一个最终存储文件名，用于最终的存储结构，避免重复。<br>
	 * 策略实现算法应确保在相同参数时计算结果是唯一的、可重复的。
	 * 根据具体实现方法不同，FileStore实现类可以不使用此函数，自行组织内部存储文件名。
	 * 
	 * @param key Key
	 * @param originalFileName 原始文件名。可能为null或空。
	 * @return 结合后的最终文件名。在整个存储库中是唯一的。
	 */
	String generateStoreFileName(final String key, final String originalFileName);
	
	
	/**
	 * 基于最终存储文件名，解析出原来的文件名。用于FileStore内部基于存储结构反向计算得出源文件名。
	 * 如果FileStore内部没有存储原文件名，例如采用流保存方式写入的文件，则本函数应该返回""空字符串。
	 * 根据具体实现方法不同，FileStore实现类可以忽略此函数中的第二个参数，自行根据Key找出原始文件名。
	 * @param key 文件Key 
	 * @param storeFileName 通过{@link #generateStoreFileName(String, String)}生成的最终存储文件名 
	 * @return 基于最终存储文件名得到的原始文件名。如果没有，则返回空字符串
	 */
	String extractOriginalFileName(final String key, final String storeFileName);
	
	/**
	 * 基于最终存储文件名，解析出原来的文件Key。用于FileStore内部基于存储结构反向计算得出文件Key。<br>
	 * 对不需要用KeyStragegy计算结合文件名的FileStore实现类来说，此函数不需要被使用。
	 * @param generateStoreFileName 通过{@link #generateStoreFileName(String, String)}生成的最终存储文件名 
	 * @return 基于最终存储文件名得到的原始文件名。如果没有，则返回空字符串
	 */
	String extractKey(final String generateStoreFileName);
	
}
