package com.kdgcsoft.power.filestore;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.filestore.strategy.IKeyStrategy;

/**
 * 基于本地磁盘目录方式实现的文件存储库基类，构造时需指定路径策略对象。
 * 
 *  @author hling
 *
 */
public class SimpleFileStore extends AbstractFileStore implements FileStore {

	private static Logger logger = LoggerFactory.getLogger(SimpleFileStore.class);
	private File rootPath;
	private IKeyStrategy keyStrategy;

	public SimpleFileStore(File rootPath, IKeyStrategy keyStrategy) {
		try {
			FileUtils.forceMkdir(rootPath);
			this.keyStrategy = keyStrategy;
		} catch (IOException e) {
			logger.error("创建目录失败，目录名：{}", rootPath.getAbsoluteFile(), e);
			throw new IllegalArgumentException("参数不正确，无法创建目录！目录名：" + rootPath.getAbsolutePath());
		}
		this.rootPath = rootPath;
	}

	public SimpleFileStore(String strRootPath, IKeyStrategy keyStrategy) {
		this(new File(strRootPath), keyStrategy);
	}

	@Override
	public boolean supportFullTextSearch() {
		return false;
	}

	/**
	 * SimpleFileStore不支持此方法，调用无效。
	 */
	@Override
	public void setLoginAccount(final String user, final String password) {
		// 简单磁盘存储方式不需要访问账号。Do nothing
	}

	/* 
	 * 使用本地磁盘文件系统存储一个文件。文件相对路径将基于内部算法生成，因此第三个参数（相对路径）会被忽略。
	 * (non-Javadoc)
	 * @see com.kdgcsoft.power.filestore.FileStore#putFile(java.io.File, java.lang.String, java.lang.String)
	 */
	@Override
	public FileInfo putFile(final File file, final String fileName)
			throws FileStoreException {
		
		logger.info("开始存储文件：源文件：{}， 存储文件名：{}", file.getAbsolutePath(), fileName);
		
//		if (relativePath != null && !"".equals(relativePath)) {
//			logger.warn("本地文件存储方式不需要指定相对路径参数，因此会忽略该参数。");
//		}
		
		String key = keyStrategy.generateKey();
		File dir = getStorageDir(key);

		// 决定最终的文件名
		String finalName = keyStrategy.generateStoreFileName(key, fileName);

		try {
			FileUtils.copyFile(file, new File(dir, finalName));
		} catch (IOException e) {
			logger.error("保存文件失败！", e);
			throw new FileStoreException("保存文件失败，原因：" + e.getMessage(), e);
		}

		logger.info("存储文件结束。key：{}，目录：{}，最终文件名：{}", key, dir.getAbsolutePath(), finalName);
		
		return getFileInfo(key);
	}
	
	/* (non-Javadoc)
	 * @see com.kdgcsoft.power.filestore.FileStore#putFile(java.io.InputStream, java.lang.String, java.lang.String)
	 */
	@Override
	public FileInfo putFileAsStream(final InputStream inputStream, final String fileName)
			throws FileStoreException {
		
		logger.info("开始存储文件流：{}", fileName);
		
		if (fileName == null || "".equals(fileName)) {
			logger.warn("存储时未指定文件名，存储库将使用空文件名");
		}
		
		String key = keyStrategy.generateKey();
		File dir = getStorageDir(key);

		// 决定最终的文件名
		String finalFileName = keyStrategy.generateStoreFileName(key, fileName);

		try {
			FileUtils.copyInputStreamToFile(inputStream, new File(dir, finalFileName));
		} catch (IOException e) {
			logger.error("保存文件失败！", e);
			throw new FileStoreException("保存文件失败，原因：" + e.getMessage(), e);
		}

		logger.info("存储文件流结束。Key：{}，目录：{}，最终文件名：{}", key, dir.getAbsolutePath(), finalFileName);
		return getFileInfo(key);
	}
	
	@Override
	public FileInfo getFileInfo(final String key) throws FileStoreException {
		logger.info("开始获取文件：{}", key);
		File file = getFileDirectly(key);
		if (file == null) {
			logger.warn("文件没找到");
			return null;
		} else {
			FileInfo info = new FileInfo();
			info.setKey(key);
			info.setFileSize(file.length());
			info.setFileName(keyStrategy.extractOriginalFileName(key, file.getName()));
			info.setFilePath(file.getAbsolutePath());
			logger.info("结束获取文件：{}", key);
			
			return info;
		}
	}
	
	@Override
	public InputStream getFileAsStream(final String key) throws FileStoreException {
		logger.info("开始获取文件流：{}", key);
		File file = getFileDirectly(key);
		if (file == null) {
			logger.warn("文件没找到");
			return null;
		} else {
			try {
				return new FileInputStream(file);
			} catch (FileNotFoundException e) {
				logger.error("查询过程中文件被删除:{}", key);
				throw new FileStoreException("查询过程中文件被删除:" + key, e);
			} finally {
				logger.info("获取文件流结束：{}", key);
			}
		}
	}
	
	/**
	 * 根据Key获取唯一的文件
	 * @param key 文件Key
	 * @return File对象
	 * @throws FileStoreException
	 */
	private File getFileDirectly(final String key) throws FileStoreException {

		File dir = getStorageDir(key);
		
		if (!dir.exists() || !dir.isDirectory()) {
			return null;
		}

		Iterator<File> iter = FileUtils.iterateFiles(dir, new IOFileFilter() {

			@Override
			public boolean accept(File file) {
				if (file.getName().startsWith(key)) {
					return true;
				} else {
					return false;
				}
			}

			@Override
			public boolean accept(File dir, String name) {
				return false;
			}

		}, null);

		File result = null;
		if (iter.hasNext()) {
			// 只找第一个。按设计应该只存在一个
			result = iter.next();
		}
		
		if (iter.hasNext()) {
			// 如果存在一个以上的文件，可能存在使用逻辑错误
			logger.warn("存储库中存在多个相同Key的文件，请注意排查使用方法是否错误。当前只使用第一个。");			
		}

		if (result == null) {
			return null;
		} else {
			return result;
		}
	}

	@Override
	public long getFileSize(final String key) throws FileStoreException {
		File file = getFileDirectly(key);
		if (file == null) {
			logger.warn("文件不存在，返回长度-1");
			return -1;
		} else {
			logger.debug("文件{}的长度为{}", key, file.length());
			return file.length();
		}
	}
	
	@Override
	public boolean exist(final String key) throws FileStoreException {
		return getFileDirectly(key) != null;
	}

	@Override
	public boolean deleteFile(final String key) throws FileStoreException {
		logger.info("开始删除文件，key={}", key);
		File file= getFileDirectly(key);
		if (file == null) {
			logger.info("文件不存在，key={}", key);
			return false;
		} else {
			try {
				FileUtils.forceDelete(file);
				logger.info("文件已删除，key={}", key);
			} catch (IOException e) {
				logger.error("删除{}失败！", file.getAbsolutePath(), e);
				throw new FileStoreException("删除文件失败：" + file.getAbsolutePath(), e);
			}
			return true;
		}
	}

	@Override
	public List<FileInfo> searchFullText(final String keyword) throws FileStoreException {
		logger.error("SimpleFileStore不支持全文检索");
		throw new FileStoreException("SimpleFileStore不支持全文检索！");
	}

	@Override
	public Map<FileInfo, String> searchFullTextWithExcerpt(final String keyword) throws FileStoreException {
		logger.error("SimpleFileStore不支持摘要全文检索");
		throw new FileStoreException("SimpleFileStore不支持摘要全文检索！");
	}

	/**
	 * 根据文件Key得到其存储目录的绝对路径，不包括文件名。
	 * 
	 * @param key 文件Key
	 * @return 最终的存储目录对象，不会为null。
	 */
	private File getStorageDir(String key)  throws FileStoreException {
		String realtivePath = keyStrategy.getRelativePathByKey(key);
		File targetDir = new File(rootPath, realtivePath);
		return targetDir;
	}

	@Override
	public void shutdown() {
		// Do Nothing
	}

}
