package com.kdgcsoft.power.filestore;

import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 用于创建和获取内容仓库实例对象的辅助管理类。应用程序可通过该管理类操作多个FileStore实例，用于不同的文件存储目的。
 *  
 * @author hling
 *
 */
public class FileStoreManager {

	private static Logger logger = LoggerFactory.getLogger(FileStoreManager.class);

	private final static String DEFAULT_STORE_NAME = "DefaultStore";

	// 存储仓库实例。因实际应用中通常只有一个，故未考虑并发
	private static Map<String, FileStore> instances = new HashMap<String, FileStore>();

	private FileStoreManager() {
		//工具类不需要new
	}
	
	/**
	 * 按照名称获取仓库实例。仓库实例必须已经用create...()函数创建过。
	 * 
	 * @param storeName
	 *            仓库名称
	 * @return 仓库实例。如果没有找到名称相同的仓库实例，则返回null。
	 * @throws FileStoreException 仓库异常
	 */
	public static FileStore getStore(final String storeName) throws FileStoreException {
		if (instances.get(storeName) == null) {
			logger.warn("没有创建名为{}的存储库。请先使用create...Store()创建！");
		}

		return instances.get(storeName);
	}

	/**
	 * 使用默认名称获取仓库实例。用于获取通过createRemoteStore(final String url)、
	 * createLocalStore(final String localStorePath)方法建立的仓库实例。
	 * 
	 * @return 仓库实例。如果没有找到名称相同的仓库实例，则返回null。
	 * @throws FileStoreException 仓库异常
	 */
	public static FileStore getStore() throws FileStoreException {
		return getStore(DEFAULT_STORE_NAME);
	}
	
	/**
	 * 设置缺省仓库对象。设置后，可以通过{@link #getStore()}获取。
	 * 
	 * @param store
	 *            仓库实例
	 * @return 添加的仓库实例
	 * @throws FileStoreException  仓库异常
	 */
	public static FileStore addStore(FileStore store) throws FileStoreException {
		return addStore(DEFAULT_STORE_NAME, store);
	}

	/**
	 * 通过名称记录一个仓库对象实例。记录后，可以通过{@link #getStore(String)}获取该实例。
	 * 一个FileStoreManager可以容纳多个不同名称的仓库对象实例。
	 * 
	 * @param storeName
	 *            仓库名称
	 * @param store
	 *            仓库实例
	 * @return 添加的仓库实例
	 * @throws FileStoreException 仓库异常
	 */
	public static FileStore addStore(String storeName, FileStore store) throws FileStoreException {
		synchronized (instances) {
			if (instances.get(storeName) != null) {
				logger.error("已经存在同名的存储库实例{}！不同的FileStore实例需要不同的名字，只存在一个实例时才可以使用缺省名", storeName);
				throw new FileStoreException("！请对不同的FileStore实例使用不同的名字。当前重复名为" + storeName);
			}

			instances.put(storeName, store);
			return store;
		}
	}

}
