package com.kdgcsoft.power.filestore;

import java.io.File;
import java.io.InputStream;
import java.util.List;
import java.util.Map;

/**
 * 内容仓库实例接口。
 * 内容仓库的设计思想是一个接口简单，只有存储文件、获取文件等最核心功能的文件存储中心。
 * 应用程序往里面存储文件时会得到一个key，相当于钥匙。根据key可以在仓库中查找、删除文件。<br>
 * 通常建议key以UUID的形式返回，但并不是必须的。
 * 应用程序不需要关心key的内容或格式，只要内容仓库能生成、识别该key，并能根据key获取文件即可。
 * 应用程序不需要关心文件在仓库内部的存储方式。
 * 
 * @author hling
 *
 */
public interface FileStore {

	public boolean supportFullTextSearch();

	/**
	 * 设定访问账号和密码，必须与仓库实际一致。如果不设置，默认为admin/admin。
	 * SimpleFileStore之类的存储实现不支持此方法，即使调用也什么都不做。
	 * @param user 用户名
	 * @param password 密码
	 */
	public void setLoginAccount(final String user, final String password);

	/**
	 * 向内容仓库存储一个文件，并返回将来可用于操作文件的FileInfo，包含将来用于操作文件的Key。<p>
	 * 内容仓库中使用的文件名将基于file参数所含的文件名生成。<p>
	 * <b>注意：本方法返回的FileInfo对象中inputStream为null。</b>
	 * 
	 * @param file
	 *            文件对象
	 * @return 文件在内容仓库中的{@link FileInfo} 文件信息对象。注意：本方法返回的FileInfo对象中inputStream为null。
	 * @throws FileStoreException 存储文件失败时抛出异常
	 */
	public FileInfo putFile(final File file)	throws FileStoreException;
	
	/**
	 * 向内容仓库存储一个文件，并返回将来可用于操作文件的FileInfo，包含将来用于操作文件的Key。<p>
	 * <b>注意：本方法返回的FileInfo对象中inputStream为null。</b>
	 * 
	 * @param file
	 *            文件对象
	 * @param fileName
	 *            内容仓库中使用的文件名。如果为null或空，则使用file参数所含的文件名。
	 * @return 文件在内容仓库中的{@link FileInfo} 文件信息对象。注意：本方法返回的FileInfo对象中inputStream为null。
	 * @throws FileStoreException 存储文件失败时抛出异常
	 */
	public FileInfo putFile(final File file, final String fileName)
			throws FileStoreException;
	
	/**
	 * 向内容仓库存储一个文件流，并返回包含文件Key的文件信息对象{@link FileInfo}（但其内部的InputStream为null）。<p>
	 * @param inputStream 文件流对象
	 * @param fileName
	 *            内容仓库中使用的文件名。如果为null或空，则使用file参数所含的文件名。
	 * @return 文件在内容仓库中的{@link FileInfo} 文件信息对象。注意：本方法返回的FileInfo对象中inputStream为null。
	 * @throws FileStoreException 存储文件失败时抛出异常
	 */
	public FileInfo putFileAsStream(final InputStream inputStream, final String fileName)
			throws FileStoreException;
	
	/**
	 * 向内容仓库存储一个文件流，并返回将来可用于操作文件的Key。<p>
	 * 因文件流中没有文件名，故存储后的文件名就基于Key生成文件名，且无扩展名。
	 * 可能会影响全文检索的结果（检索不出来）
	 * 通常情况下，以流方式存储且不指定文件名的场景中，获取文件也应该以流方式获取。
	 * 
	 * @param inputStream 文件流对象
	 * @return 文件在内容仓库中的{@link FileInfo} 文件信息对象。注意：本方法返回的FileInfo对象中inputStream为null。
	 * @throws FileStoreException 存储文件失败时抛出异常
	 */
	public FileInfo putFileAsStream(final InputStream inputStream)
			throws FileStoreException;

	/**
	 * 从内容仓库获取文件信息，不包含文件流InputStream，即返回的FileInfo对象中inputStream属性为null。
	 * 
	 * @param key 在往内容仓库存放文件时获得的文件key。
	 * @return 文件对象，包含文件key、文件尺寸信息。如果没找到，返回null。
	 * @throws FileStoreException 存储库异常
	 */
	public FileInfo getFileInfo(final String key) throws FileStoreException;
	
	/**
	 * 从内容仓库获取文件流。
	 * 
	 * @param key  在往内容仓库存放文件时获得的文件key。
	 * @return 文件流对象。如果没找到或未设置，返回null。
	 * @throws FileStoreException 存储库异常
	 */
	public InputStream getFileAsStream(final String key) throws FileStoreException;

	/**
	 * 获取文件大小
	 * 
	 * @param key 在往内容仓库存放文件时获得的文件key。
	 * @return 文件字节数。如果文件不存在，返回-1。
	 * @throws FileStoreException 存储库异常
	 */
	public long getFileSize(final String key) throws FileStoreException;

	/**
	 * 判断文件是否存在
	 * @param key 在往内容仓库存放文件时获得的文件key。
	 * @return true/false
	 * @throws FileStoreException 存储库异常
	 */
	public boolean exist(String key) throws FileStoreException;

	/**
	 * 删除指定key的文件
	 * @param key 在往内容仓库存放文件时获得的文件key。
	 * @return true表示删除成功，false表示文件不存在
	 * @throws FileStoreException 存储库异常
	 */
	public boolean deleteFile(String key) throws FileStoreException;

	/**
	 * 全文检索 检索出所有匹配关键字的文件Key。
	 * 注意：FileStore并不一定支持全文检索。如果不支持，应抛出FileStoreException异常。
	 * 
	 * @param keyword
	 *            需要检索的关键字
	 * @return 与关键字匹配的文件信息列表
	 * @throws FileStoreException 不支持全文检索或其他异常
	 */
	public List<FileInfo> searchFullText(String keyword) throws FileStoreException;

	/**
	 * 全文检索并显示摘要。检索出所有匹配关键字的文件Key和摘要，并存到Map中，其中key为文件key，value为摘要。
	 * 
	 * @param keyword
	 *            需要检索的关键字
	 * @return 与关键字匹配的文件信息和摘要的键值对
	 * @throws FileStoreException 不支持全文检索或其他异常
	 */
	public Map<FileInfo, String> searchFullTextWithExcerpt(String keyword) throws FileStoreException;

	/**
	 * 在不使用文件仓库或应用程序退出时，请调用shutdown()方法，以便某些仓库实现类进行清理退出工作。
	 */
	public void shutdown();
}
