package com.kdgcsoft.power.filestore;

import java.io.File;
import java.io.InputStream;
import java.util.List;
import java.util.Map;

import com.kdgcsoft.power.filestore.FileInfo;
import com.kdgcsoft.power.filestore.FileStore;
import com.kdgcsoft.power.filestore.FileStoreException;

/**
 * 能够存储文件和元数据的内容仓库接口
 * @author hling
 */
public interface FileMetaStore extends FileStore {
	
	/**
	 * 向内容仓库存储一个文件，并返回将来可用于操作文件的FileInfo，包含将来用于操作文件的Key。<p>
	 * 内容仓库中使用的文件名将基于file参数所含的文件名生成。<p>
	 * <b>注意：本方法返回的FileInfo对象中inputStream为null。</b>
	 * 
	 * @param file 要存储的文件对象
	 * @param metadata 要存储的文件元数据。数值只支持原生类型及相应的对象类型、Date、Calendar。不支持数组、map等。
	 * @return 文件在内容仓库中的{@link FileInfo} 文件信息对象。注意：本方法返回的FileInfo对象中inputStream为null。
	 * @throws FileStoreException 存储文件失败时抛出异常
	 */
	public FileInfo putFile(final File file, final Map<String, Object> metadata)	throws FileStoreException;
	
	/**
	 * 向内容仓库存储一个文件，并返回将来可用于操作文件的FileInfo，包含将来用于操作文件的Key。<p>
	 * <b>注意：本方法返回的FileInfo对象中inputStream为null。</b>
	 * 
	 * @param file
	 *            文件对象
	 * @param fileName
	 *            内容仓库中使用的文件名。如果为null或空，则使用file参数所含的文件名。
	 * @param metadata 要存储的文件元数据。数值只支持原生类型及相应的对象类型、Date、Calendar。不支持数组、map等。
	 * @return 文件在内容仓库中的{@link FileInfo} 文件信息对象。注意：本方法返回的FileInfo对象中inputStream为null。
	 * @throws FileStoreException 存储文件失败时抛出异常
	 */
	public FileInfo putFile(final File file, final String fileName, final Map<String, Object> metadata)
			throws FileStoreException;
	
	/**
	 * 向内容仓库存储一个文件流，并返回包含文件Key的文件信息对象{@link FileInfo}（但其内部的InputStream为null）。<p>
	 * @param inputStream 文件流对象
	 * @param fileName
	 *            内容仓库中使用的文件名。如果为null或空，则使用file参数所含的文件名。
	 * @param metadata 要存储的文件元数据。数值只支持原生类型及相应的对象类型、Date、Calendar。不支持数组、map等。
	 * @return 文件在内容仓库中的{@link FileInfo} 文件信息对象。注意：本方法返回的FileInfo对象中inputStream为null。
	 * @throws FileStoreException 存储文件失败时抛出异常
	 */
	public FileInfo putFileAsStream(final InputStream inputStream, final String fileName, final Map<String, Object> metadata)
			throws FileStoreException;
	
	/**
	 * 向内容仓库存储一个文件流，并返回将来可用于操作文件的Key。<p>
	 * 因文件流中没有文件名，故存储后的文件名就基于Key生成文件名，且无扩展名。
	 * 可能会影响全文检索的结果（检索不出来）
	 * 通常情况下，以流方式存储且不指定文件名的场景中，获取文件也应该以流方式获取。
	 * 
	 * @param inputStream 文件流对象
	 * @param metadata 要存储的文件元数据。数值只支持原生类型及相应的对象类型、Date、Calendar。不支持数组、map等。
	 * @return 文件在内容仓库中的{@link FileInfo} 文件信息对象。注意：本方法返回的FileInfo对象中inputStream为null。
	 * @throws FileStoreException 存储文件失败时抛出异常
	 */
	public FileInfo putFileAsStream(final InputStream inputStream, final Map<String, Object> metadata)
			throws FileStoreException;

	/**
	 * 从内容仓库获取文件的元数据，不包含文件流InputStream，即返回的FileInfo对象中inputStream属性为null。
	 * 
	 * @param key 在往内容仓库存放文件时获得的文件key。
	 * @param metaKey 要获取的属性名。
	 * @param cls 希望返回的数据类型。最好与调用{@link #setFileMetadata(String, String, Object)}时使用的类型一致。
	 * @return 文件对象，包含文件key、文件尺寸信息。如果没找到，返回null。
	 * @throws FileStoreException 存储库异常
	 */
	public <T> T getFileMetadata(final String key, final String metaKey, final Class<T> cls) throws FileStoreException;
	
	/**
	 * 添加或更新内容仓库中某文件的元数据。对已有的属性会覆盖。
	 * 
	 * @param key 在往内容仓库存放文件时获得的文件key。
	 * @throws FileStoreException 存储库异常
	 */
	public void setFileMetadata(final String key, final Map<String, Object> metadata) throws FileStoreException;
	
	/**
	 * 添加或更新内容仓库中某文件的元数据。更新方式是与已有的合并，并覆盖重复的条目。
	 * 如果要一次写入多个属性，不要循环调用本函数，而应该使用{@link #setFileMetadata(String, Map)}提高性能。
	 * 
	 * @param key 在往内容仓库存放文件时获得的文件key。
	 * @param metaKey 要写入的元数据属性名
	 * @param value 要写入的属性值。数值只支持原生类型及相应的对象类型、Date、Calendar。不支持数组、map等。
	 * @throws FileStoreException 存储库异常
	 */
	public void setFileMetadata(final String key, final String metaKey, final Object value) throws FileStoreException;

	/**
	 * 删除内容仓库中某文件的某一条元数据。
	 * 
	 * @param key 在往内容仓库存放文件时获得的文件key。
	 * @param metaKey 要删除的元数据条目的key
	 * @throws FileStoreException 存储库异常
	 */
	public void removeFileMetadata(final String key, final String metaKey) throws FileStoreException;
	
	/**
	 * 全文检索 检索出所有匹配关键字和指定元数据数值的文件Key
	 * 注意：FileStore并不一定支持全文检索。如果不支持，应抛出FileStoreException异常。
	 * 
	 * @param params 检索参数
	 * @return 与关键字匹配的文件信息列表
	 * @throws FileStoreException 不支持全文检索或其他异常
	 */
	public List<FileInfo> searchFullText(SearchParameter params) throws FileStoreException;

	/**
	 * 全文检索并显示摘要。检索出所有匹配关键字和指定元数据数值的文件Key和摘要，
	 * 并存到Map中，其中key为文件信息，value为摘要。
	 * 
	 * @param params 检索参数
	 * @return 与关键字匹配的文件信息-摘要的键值对
	 * @throws FileStoreException 不支持全文检索或其他异常
	 */
	public Map<FileInfo, String> searchFullTextWithExcerpt(SearchParameter params) throws FileStoreException;

}
