package com.kdgcsoft.power.filestore;

import java.io.File;
import java.io.InputStream;

/**
 * 把文件保存到存储库或查询文件后得到的相关信息，其中最重要的是Key，存储库的所有文件操作接口都需要Key才能进行。
 * 
 * @author hling
 *
 */
public class FileInfo {
	/**
	 * 可用来读取文件的字节流
	 */
//	private InputStream inputStram;
	/**
	 * 文件在存储库中的Key
	 */
	private String key;
	/**
	 * 文件字节数
	 */
	private Long fileSize;
	/**
	 * 原始文件名
	 */
	private String fileName;
	/**
	 * 使用本地文件系统SimpleFileStore存储文件时，文件在本地磁盘上的最终存储路径，包含文件名。<br>
	 * 当存储容器不是本地文件系统时（例如JackRabbit）则永远为null；
	 */
	private String filePath = null;

	public FileInfo() {}
	
	/**
	 * 构造FileInfo对象
	 * 
	 * @param key 文件Key
	 * @param originName 原始文件名
	 * @param file 物理文件对象
	 */
	public FileInfo(String key, String originName, File file) {
		this.key = key;
		this.fileName = originName;
		this.filePath = file.getAbsolutePath();
		this.fileSize = file.length();
//		try {
//			this.inputStram = new FileInputStream(file);
//		} catch (FileNotFoundException e) {
//			this.inputStram = null;
//		}
	}
	
	/**
	 * 构造FileInfo对象
	 * 
	 * @param key 文件Key
	 * @param originName 原始文件名
	 * @param size 文件字节数
	 * @param inputStream 文件字节流
	 */
	public FileInfo(String key, String originName, Long size, InputStream inputStream) {
		this.key = key;
		this.fileName = originName;
		this.fileSize = size;
//		this.inputStram = inputStream;
	}

	/**
	 * 获得进入存储库之前的原始文件名
	 * 
	 * @return 原始文件名
	 */
	public String getFileName() {
		return fileName;
	}
	
	/**
	 * 设置原始文件名
	 * 
	 * @param fileName 文件名
	 */
	public void setFileName(String fileName) {
		this.fileName = fileName;
	}

	/**
	 * 获得存储库中文件的输入字节流，用于读取文件数据。
	 * 如果使用了JackRabbit这样的存储仓库，无法获得独立存在的File对象，只能通过Stream方式读取文件。
	 * 
	 * @return 存储库提供的文件字节流
	 */
//	public InputStream getInputStream() {
//		return inputStram;
//	}

	/**
	 * 设置文件流对象
	 * 
	 * @param is 文件字节流
	 */
//	public void setInputStream(InputStream is) {
//		this.inputStram = is;
//	}

	/**
	 * 获得文件在存储库中的Key。每个放入存储库中的文件，都具有唯一的Key。
	 * 
	 * @return Key
	 */
	public String getKey() {
		return key;
	}

	/**
	 * 设置文件Key
	 * 
	 * @param key 文件在存储库中的Key
	 */
	public void setKey(String key) {
		this.key = key;
	}

	/**
	 * 获得文件字节数
	 * 
	 * @return 文件字节数
	 */
	public Long getFileSize() {
		return fileSize;
	}

	/**
	 * 设置文件字节数
	 * 
	 * @param filesize 字节数
	 */
	public void setFileSize(Long filesize) {
		this.fileSize = filesize;
	}
	
	/**
	 * 返回SimpleFileStore存储库中的文件在本地磁盘上的物理路径，包含文件名。存储库中的物理文件名与原始文件名是不一样的。
	 * 对不使用本地文件系统的存储库实现（例如JackRabbit），则总是返回null。
	 * 
	 * 注意：返回的文件路径是存储库中真实存在的文件，应用程序不应该对该文件进行任何写操作，以防破坏文件库结构。
	 * 
	 * @return 本地存储的物理文件名。如果不使用本地文件存储，则为null
	 */
	public String getFilePath() {
		return filePath;
	}

	/**
	 * 设置存储库中文件的物理路径
	 * 
	 * @param filePath 物理路径
	 */
	public void setFilePath(String filePath) {
		this.filePath = filePath;
	}

	@Override
	public String toString() {
		return "FileInfo [key=" + key + ", fileSize=" + fileSize + ", fileName=" + fileName + ", filePath=" + filePath
				+ "]";
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((fileName == null) ? 0 : fileName.hashCode());
		result = prime * result + ((filePath == null) ? 0 : filePath.hashCode());
		result = prime * result + ((fileSize == null) ? 0 : fileSize.hashCode());
		result = prime * result + ((key == null) ? 0 : key.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		FileInfo other = (FileInfo) obj;
		if (fileName == null) {
			if (other.fileName != null)
				return false;
		} else if (!fileName.equals(other.fileName))
			return false;
		if (filePath == null) {
			if (other.filePath != null)
				return false;
		} else if (!filePath.equals(other.filePath))
			return false;
		if (fileSize == null) {
			if (other.fileSize != null)
				return false;
		} else if (!fileSize.equals(other.fileSize))
			return false;
		if (key == null) {
			if (other.key != null)
				return false;
		} else if (!key.equals(other.key))
			return false;
		return true;
	}

}
