package com.kdgcsoft.power.filemanager;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.fileconverter.Office2PdfEngineType;
import com.kdgcsoft.power.fileconverter.FileConverterService;
import com.kdgcsoft.power.fileconverter.FileConverterSettings;
import com.kdgcsoft.power.fileconverter.StorageType;
import com.kdgcsoft.power.filestore.FileStore;
import com.kdgcsoft.power.filestore.FileStoreException;
import com.kdgcsoft.power.filestore.TimeStampSimpleFileStore;
import com.kdgcsoft.power.filestore.UUIDSimpleFileStore;
import com.kdgcsoft.power.filestore.jackrabbit.LocalJackRabbitFileStore;
import com.kdgcsoft.power.filestore.jackrabbit.RemoteJackRabbitFileStore;

/**
 * FileManager运行参数的集合
 * @author hling
 *
 */
public class FileManagerSettings {
	
	private static final Logger logger = LoggerFactory.getLogger(FileManagerSettings.class); 
	
	private static final String DEFAULT_DIR = System.getProperty("user.dir") + "/FileManager";
	
	private String baseDir = DEFAULT_DIR;
	
	private IDType idType = IDType.TIMESTAMP;
	
	private boolean useJackrabbit = false;
	
	private String jackrabbitURL = "";
	
	private String jackrabbitUser = "admin";
	
	private String jackrabbitPassword = "admin";
	
	private boolean autoConvert = true;

	private AutoConvertStrategy convertStrategy = null;
	
	private FileConverterSettings convertSettings = new FileConverterSettings();

	private FileManagerSettings(){}
	
	/**
	 * 使用缺省配置文件，得到一个参数集实例
	 * @return 参数集实例
	 */
	public static FileManagerSettings createDefault() {
		InputStream is = FileManagerSettings.class.getClassLoader().getResourceAsStream("default.properties");
		return createByPropertiesSteam(is);
	}
	
	/**
	 * 基于外部properties配置文件得到一个参数集实例。注意请确保该文件格式为UTF-8格式。
	 * @param propFile 配置文件
	 * @return 参数集实例
	 */
	public static FileManagerSettings createByPropertiesFile(File propFile) {
		try {
			return createByPropertiesSteam(new FileInputStream(propFile));
		} catch (FileNotFoundException e) {
			logger.error("读取配置文件失败{}！将使用默认配置。", propFile.getAbsolutePath(), e);
			return createDefault();
		}
	}
	
	/**
	 * 基于配置文件得到一个参数集实例
	 * @param is 配置文件输入流。注意请确保该文件格式为UTF-8格式。
	 * @return 根据配置文件创建的参数集实例
	 */
	public static FileManagerSettings createByPropertiesSteam(InputStream is) {
		Properties prop = new Properties();
		try {
			prop.load(new InputStreamReader(is, "utf-8"));
			return createByProperties(prop);
		} catch (Exception e) {
			logger.error("加载配置文件失败{}！", e);
			return new FileManagerSettings();
		}
	}
	
	/**
	 * 基于已经读取的配置文件信息得到一个参数集实例
	 * @param prop 配置信息
	 * @return 根据配置文件创建的参数集实例
	 */
	public static FileManagerSettings createByProperties(Properties prop) {
		FileManagerSettings settings = new FileManagerSettings();
		
		String baseDir = prop.getProperty("dc.dir");
		if (!isBlank(baseDir)) {
			settings.setBaseDir(baseDir.trim());
		}
		
		String idType = prop.getProperty("dc.id-type");
		if (idType != null) {
			try {
				IDType type = IDType.valueOf(idType.trim().toUpperCase());
				settings.setIdType(type);
			} catch (Exception e) {
				logger.warn("无法识别的id-type：{}", idType);
			}
		}
		
		boolean useJackRabbit = Boolean.parseBoolean(prop.getProperty("dc.filestore.jackrabbit.enabled"));
		if (useJackRabbit) {
			settings.setUseJackrabbit(
					prop.getProperty("dc.filestore.jackrabbit.url"),
					prop.getProperty("dc.filestore.jackrabbit.user"),
					prop.getProperty("dc.filestore.jackrabbit.password"));
		}
		
		String autoConvert = prop.getProperty("dc.convert.auto-convert.enabled");
		if (!isBlank(autoConvert)) {
			settings.setAutoConvert(Boolean.valueOf(autoConvert));
		}
		
		String strategyStr = prop.getProperty("dc.convert.auto-convert.strategy");
		if (!isBlank(strategyStr)) {
			// 解析自动转换策略定义字符串
			settings.setConvertStrategy(AutoConvertStrategy.parse(strategyStr.trim()));
		}
		
		String testMode = prop.getProperty("dc.convert.test-mode");
		if (!isBlank(testMode)) {
			settings.setConvertTestMode(Boolean.valueOf(testMode));
		}
		
		String engine = prop.getProperty("dc.convert.engine");
		if (!isBlank(engine)) {
			try {
				Office2PdfEngineType type = Office2PdfEngineType.valueOf(engine.trim().toUpperCase());
				settings.setConvertEngine(type);
			} catch (Exception e) {
				logger.error("无法识别的转换引擎：{}", engine);
			}
		}
		
		String threadStr = prop.getProperty("dc.convert.max-thread");
		if (!isBlank(threadStr)) {
			try {
				settings.setMaxConvertThread(Integer.valueOf(threadStr.trim()));
			} catch (NumberFormatException e) {
				logger.error("dc.convert.max-thread必须是整数！");
			}
		}
		
		String pdf2HtmlStr = prop.getProperty("dc.convert.pdf2html-exe-path");
		if (!isBlank(pdf2HtmlStr)) {
			settings.setPdf2HtmlExePath(pdf2HtmlStr.trim());
		}
		
		String openOfficePath = prop.getProperty("dc.convert.openoffice.path");
		if (!isBlank(openOfficePath)) {
			settings.setPdf2HtmlExePath(openOfficePath.trim());
		}
		
		String openOfficePorts = prop.getProperty("dc.convert.openoffice.ports");
		if (!isBlank(openOfficePorts)) {
			List<Integer> portList = new ArrayList<Integer>();
			String[] ports = openOfficePorts.split(",");
			for (String port : ports) {
				try {
					int p = Integer.parseInt(port.trim());
					portList.add(p);
				} catch (NumberFormatException e) {
					logger.error("端口号必须是整数！当前值：{}", port);
				}
			}
			int[] intPorts = new int[portList.size()];
			for (int i=0;i<portList.size();i++) {
				intPorts[i] = portList.get(i);
			}
			settings.setOpenOfficePorts(intPorts);
		}
		
		String e2h_showHeaders = prop.getProperty("dc.convert.excel2html.show-headers");
		if (!isBlank(e2h_showHeaders)) {
			settings.setExcel2HtmlShowHeaders(Boolean.valueOf(e2h_showHeaders));
		}
		
		String e2h_showHidden = prop.getProperty("dc.convert.excel2html.show-hidden-content");
		if (!isBlank(e2h_showHidden)) {
			settings.setExcel2HtmlShowHeaders(Boolean.valueOf(e2h_showHidden));
		}
		
		String e2h_hideOneTab = prop.getProperty("dc.convert.excel2html.hide-only-one-tab");
		if (!isBlank(e2h_hideOneTab)) {
			settings.setExcel2HtmlShowHeaders(Boolean.valueOf(e2h_hideOneTab));
		}

		return settings;
	}
	

	/**
	 * 设置文档中心的工作根目录。文件库、转换文件库都会自动在这个目录下创建。
	 * @param baseDir 文档中心的工作目录
	 * @return 当前参数集实例
	 */
	public FileManagerSettings setBaseDir(String baseDir) {
		if (!isBlank(baseDir)) {
			this.baseDir = baseDir.trim();
		} else {
			logger.info("FileManager使用缺省存储路径：{}", baseDir);
		}
		
		return this;
	}

	/**
	 * 设置使用哪一种文档ID类型，支持UUID、时间戳。
	 * @param idType 文档ID类型
	 * @return 当前参数集实例
	 */
	public FileManagerSettings setIdType(IDType idType) {
		if (idType != null) {
			this.idType = idType;
		}
		
		return this;
	}

	/**
	 * 设置是否使用Jackrabbit作为文件存储库。默认为使用本地文件系统存储，不使用JackRabbit。
	 * @param urlOrDir JackRabbit的路径，可以是本地路径或URL
	 * @param user JackRabbit的用户名
	 * @param password JackRabbit的密码
	 * @return 当前参数集实例
	 */
	public FileManagerSettings setUseJackrabbit(String urlOrDir, String user, String password) {
		this.useJackrabbit = true;
		this.jackrabbitURL = urlOrDir;
		this.jackrabbitUser = user;
		this.jackrabbitPassword = password;
		
		return this;
	}
	
	/**
	 * 设置是否启用自动转换，默认为true。如果启用了自动转换，将执行内置的转换策略。该策略可以通过 {@link #setConvertStrategy(AutoConvertStrategy)}修改。
	 * @param autoConvert 是否自动转换
	 * @return 当前参数集实例
	 */
	public FileManagerSettings setAutoConvert(boolean autoConvert) {
		this.autoConvert = autoConvert;
		return this;
	}

	/**
	 * 设置自动转换策略。默认的转换策略中除了Excel文件会被转换成HTML，其他Office文档都被转换成PDF。
	 * @param convertStrategy 自定义的转换策略
	 * @return 当前参数集实例
	 */
	public FileManagerSettings setConvertStrategy(AutoConvertStrategy convertStrategy) {
		this.convertStrategy = convertStrategy;
		return this;
	}
	
	/**
	 * 把转换引擎设置为测试模式。该模式下不依赖任何外部转换程序，不进行实际转换，只生成0字节的目标文件。<br>
	 * 在开发调试时可以先使用这种模式来确认系统整体运转情况。默认为非测试模式。
	 * @see com.kdgcsoft.power.fileconverter.FileConverterSettings#setTestMode(boolean)
	 * @param isTestMode 是否是测试模式
	 * @return 当前参数集实例
	 */
	public FileManagerSettings setConvertTestMode(final boolean isTestMode) {
		this.convertSettings.setTestMode(isTestMode);
		return this;
	}
	
	/**
	 * 设置基础转换引擎。可选择Jacob(需要安装微软Office)或OpenOffice/LibreOffice方案。默认是Jacob。
	 * @see com.kdgcsoft.power.fileconverter.FileConverterSettings#setOffice2PdfEngine(Office2PdfEngineType)
	 * @param engine 转换引擎
	 * @return 当前参数集实例
	 */
	public FileManagerSettings setConvertEngine(final Office2PdfEngineType  engine) {
		if (engine != null) {
			this.convertSettings.setOffice2PdfEngine(engine);
		}
		
		return this;
	}
	
	/**
	 * 设置最大并发转换线程数。
	 * @see com.kdgcsoft.power.fileconverter.FileConverterSettings#setMaxConvertThread(int)
	 * @param maxConvertThread 最大转换并发线程数
	 * @return 当前参数集实例
	 */
	public FileManagerSettings setMaxConvertThread(final int maxConvertThread) {
		if (maxConvertThread > 0) {
			this.convertSettings.setMaxConvertThread(maxConvertThread);
		}
		
		return this;
	}
	
	/**
	 * 设置Pdf2HtmlEx工具程序的可执行文件路径。
	 * @see com.kdgcsoft.power.fileconverter.FileConverterSettings#setPdf2HtmlExePath(String)
	 * @param exePath 可执行文件路径
	 * @return 当前参数集实例
	 */
	public FileManagerSettings setPdf2HtmlExePath(final String exePath) {
		if (!isBlank(exePath)) {
			this.convertSettings.setPdf2HtmlExePath(exePath.trim());
		}
		
		return this;
	}
	
	/**
	 * 设置OpenOffice/LibreOffice的安装目录。如果使用OpenOffice作为转换引擎且未设置安装目录的话，程序会自动按照操作系统默认安装位置查找OpenOffice/LibreOffice。
	 * @see com.kdgcsoft.power.fileconverter.FileConverterSettings#setOpenOfficePath(String)
	 * @param openOfficeDir OpenOffice的安装路径
	 * @return 当前参数集实例
	 */
	public FileManagerSettings setOpenOfficeDirectory(final String openOfficeDir) {
		if (!isBlank(openOfficeDir)) {
			this.convertSettings.setOpenOfficePath(openOfficeDir.trim());
		}
		return this;
	}
	
	/**
	 * 设置OpenOffice/LibreOffice的监听端口。参见 {@link FileConverterSettings#setOpenOfficePorts(int[])}
	 * @param ports 端口号。
	 * @return 当前参数集实例
	 * 
	 * @deprecated 可以自动探测空闲端口，不再需要指定
	 */
	@Deprecated
	public FileManagerSettings setOpenOfficePorts(final int[] ports) {
		if (ports != null) {
			this.convertSettings.setOpenOfficePorts(ports);
		}
		
		return this;
	}
	
	/**
	 * Excel2Html转换时是否显示行列坐标。默认为不显示
	 * @param excel2html_showHeaders 是否显示行列坐标
	 * @return 当前参数集实例
	 * @see com.kdgcsoft.power.fileconverter.FileConverterSettings#setExcel2HtmlShowHeaders(boolean)
	 */
	public FileManagerSettings setExcel2HtmlShowHeaders(final boolean excel2html_showHeaders) {
		this.convertSettings.setExcel2HtmlShowHeaders(excel2html_showHeaders);
		return this;
	}

	/**
	 * Excel2Html转换时是否在HTML中隐藏Excel中的隐藏内容。默认为隐藏。
	 * @param excel2html_showHiddenContent 是否显示Excel中的隐藏内容
	 * @return 当前参数集实例
	 * @see com.kdgcsoft.power.fileconverter.FileConverterSettings#setExcel2HtmlShowHiddenContent(boolean)
	 */
	public FileManagerSettings setExcel2HtmlShowHiddenContent(final boolean excel2html_showHiddenContent) {
		this.convertSettings.setExcel2HtmlShowHiddenContent(excel2html_showHiddenContent);
		return this;
	}

	/**
	 * Excel2Html转换时，如果只有一个Sheet，则是否隐藏该Sheet的Tab标签。默认为不隐藏，即总是显示Tab。
	 * @param excel2html_hideTabIfOnlyOne 是否隐藏唯一Sheet的Tab标签
	 * @return 当前参数集实例
	 * @see com.kdgcsoft.power.fileconverter.FileConverterSettings#setExcel2HtmlHideTabIfOnlyOne(boolean)
	 */
	public FileManagerSettings setExcel2HtmlHideTabIfOnlyOne(final boolean excel2html_hideTabIfOnlyOne) {
		this.convertSettings.setExcel2HtmlHideTabIfOnlyOne(excel2html_hideTabIfOnlyOne);
		return this;
	}

	/**
	 * 定制Excel2Html输出的Html模板字符串。一般情况下Excel2Html自带模板就可以了。
	 * @param excel2html_htmlTemplate 自定义Html模板。
	 * @return 当前参数集实例
	 * @see com.kdgcsoft.power.fileconverter.FileConverterSettings#setExcel2HtmlTemplateStr(String)
	 */
	public FileManagerSettings setExcel2HtmlTemplateStr(final String excel2html_htmlTemplate) {
		if (!isBlank(excel2html_htmlTemplate)) {
			this.convertSettings.setExcel2HtmlTemplateStr(excel2html_htmlTemplate);
		}
		
		return this;
	}
	
	/**
	 * 一次性设置所有转换参数。
	 * @param settings 转换设置
	 * @return 当前参数集实例
	 */
	public FileManagerSettings setConvertSettings(final FileConverterSettings settings) {
		if (settings != null) {
			this.convertSettings = settings;
		} else {
			logger.warn("setConvertSettings参数为null。将使用默认转换设置。");
		}
		return this;
	}

	/**
	 * 基于已经设置的参数，创建文档中心实例。
	 * @return FileManager instance
	 * @throws IOException 读写文件系统时发生异常
	 * @throws FileStoreException 仅在使用JackRabbit作为文件存储库，且创建或连接JackRabbit库失败时抛出FileStoreException
	 */
	void apply(FileManager filemanager) throws FileStoreException, IOException {
		
		// 初始化目录
		File root = new File(this.baseDir);
		// 转换成绝对路径，便于调试
		root = new File(root.getAbsolutePath());
		
		// 构建FileStore
		FileStore fileStore;
		if (useJackrabbit) {
			if (jackrabbitURL.startsWith("http")) {
				// 远程JackRabbit库
				fileStore = new RemoteJackRabbitFileStore(jackrabbitURL, jackrabbitUser, jackrabbitPassword);
			} else {
				// 本地JackRabbit库
				fileStore = new LocalJackRabbitFileStore(jackrabbitURL, jackrabbitUser, jackrabbitPassword);
			}
			
			// 使用JackRabbit时，key必须为UUID
			idType = IDType.UUID;

		} else {
			String fileStoreBaseDir = new File(root, "file_storage").getAbsolutePath(); 
			if (IDType.TIMESTAMP.equals(idType)) {
				fileStore = new TimeStampSimpleFileStore(fileStoreBaseDir);
			} else {
				fileStore = new UUIDSimpleFileStore(fileStoreBaseDir);
			}
		}		
		filemanager.setFileStore(fileStore);
		
		// 初始化转换引擎
		String convertBaseDir = new File(root, "convert_storage").getAbsolutePath();
		convertSettings.setWorkdir(new File(convertBaseDir));
		if (IDType.TIMESTAMP.equals(idType)) {
			convertSettings.setStorageType(StorageType.TIMESTAMP);
		} else {
			convertSettings.setStorageType(StorageType.UUID);
		}
		FileConverterService.init(convertSettings);

		filemanager.setAutoConvert(autoConvert);
		if (convertStrategy != null) {
			filemanager.setConvertStrategy(convertStrategy);
		}
		
		return;
	}

	/**
	 * 判断字符串是否为空
	 * @param cs 字符串
	 * @return 是否为空
	 */
	private static boolean isBlank(CharSequence cs) {
		int strLen;
		if (cs == null || (strLen = cs.length()) == 0) {
			return true;
		}
		for (int i = 0; i < strLen; i++) {
			if (Character.isWhitespace(cs.charAt(i)) == false) {
				return false;
			}
		}
		return true;
	}

	/**
	 * 文档中心使用的文件ID类型，目前有UUID和时间戳两种。
	 * @author hling
	 *
	 */
	public enum IDType {
		/**
		 * UUID类型。
		 */
		UUID, 
		/**
		 * 时间戳类型 
		 */
		TIMESTAMP
	}

}
