package com.kdgcsoft.power.filemanager;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Properties;

import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.fileconverter.FileConverterException;
import com.kdgcsoft.power.fileconverter.FileConverterService;
import com.kdgcsoft.power.fileconverter.OutputType;
import com.kdgcsoft.power.filestore.FileInfo;
import com.kdgcsoft.power.filestore.FileStore;
import com.kdgcsoft.power.filestore.FileStoreException;

final public class FileManager {
	
	private static final Logger logger = LoggerFactory.getLogger(FileManager.class);
	
	/**
	 * Single Instance
	 */
	private static final FileManager instance = new FileManager();

	private FileStore fileStore;
	
	private boolean autoConvert = true;

	private AutoConvertStrategy convertStrategy = AutoConvertStrategy.createDefaultStrategy();

	private volatile boolean inited = false;

	private FileManagerSettings settings = FileManagerSettings.createDefault() ;
	
	FileManager(){} {}
	
	/**
	 * 获取FileManager单例对象
	 * @return FileManager实例
	 */
	public static FileManager getInstance() {
		return instance;
	}
	
	/**
	 * 指定FileManagerSettings实例作为初始化参数
	 * @param settings 设置参数
	 */
	public static void init(FileManagerSettings settings) {
		if (instance.inited) {
			logger.error("FileManager已经初始化，不能多次调用init接口！");
		}
		
		instance.settings = settings;
	}
	
	/**
	 * 设定外部配置文件作为初始化参数
	 * @param propFile FileManager的配置文件
	 */
	public static void init(File propFile) {
		if (instance.inited) {
			logger.error("FileManager已经初始化，不能多次调用init接口！");
		}
		
		instance.settings = FileManagerSettings.createByPropertiesFile(propFile);
	}
	
	/**
	 * 设定外部配置文件的字节流作为初始化参数
	 * @param is FileManager的配置文件字节流
	 */
	public static void init(InputStream is) {
		if (instance.inited) {
			logger.error("FileManager已经初始化，不能多次调用init接口！");
		}
		
		instance.settings = FileManagerSettings.createByPropertiesSteam(is);
	}

	/**
	 * 设定配置集合对象作为初始化参数
	 * @param props 配置集合对象
	 */
	public static void init(Properties props) {
		if (instance.inited) {
			logger.error("FileManager已经初始化，不能多次调用init接口！");
		}
		
		instance.settings = FileManagerSettings.createByProperties(props);
	}
	
	/**
	 * 检查FileManager是否已经初始化，如果没有则进行初始化，否则什么都不做。
	 * @throws FileStoreException 文件仓库异常
	 */
	protected void prepare() throws FileStoreException {
		if (inited) {
			return;
		}
		
		synchronized (this) {
			try {
				if (inited) {
					return;
				}
				
				settings.apply(this);
				inited = true;
			} catch (IOException e) {
				logger.error("FileManager初始化失败！", e.getMessage());
				throw new FileStoreException(e);
			} catch(FileStoreException e) {
				logger.error("FileManager初始化失败！", e.getMessage());
				throw new FileStoreException(e);
			}
		}
	}
	
	/**
	 * 设置文件存储容器（内容仓库）
	 * @param store 文件存储器（内容仓库实例）
	 */
	void setFileStore(FileStore store) {
		this.fileStore = store;
	}
	 
	/**
	 * 设置是否进行自动转换。如果设置了，则当文件被保存到文件存储器时，立即按照预定策略同步进行格式转换
	 * @param autoConvert 是否自动转换
	 */
	void setAutoConvert(boolean autoConvert) {
		this.autoConvert = autoConvert;
	}
	
	/**
	 * 设置文件格式转换策略
	 * @param convertStrategy 转换策略
	 */
	void setConvertStrategy(AutoConvertStrategy convertStrategy) {
		this.convertStrategy = convertStrategy;
	}
	
	/**
	 * 保存一个文件。如果开启了自动转换，则同时创建转换任务。
	 * @param file 要保存的文件
	 * @return 保存中的文件信息
	 * @throws FileStoreException 文件仓库异常
	 * @throws FileConverterException 文件转换异常
	 */
	public FileInfo putFile(File file) throws FileStoreException, FileConverterException{
		prepare();
		return putFile(file, file.getName());
	}
	
	/**
	 * 往文档中心保存一个文件。如果设置了自动转换，则同时进行自动转换。
	 * @param file 要保存的文件
	 * @param fileName 真正的文件名。第一个参数中的文件可能是临时文件，其文件名将被忽略。
	 * 本参数将作为文件仓库中该文件真正的逻辑文件名，后续的文件转换过程将基于该文件名进行。
	 * @return 保存后的文件信息
	 * @throws FileStoreException 文件仓库异常
	 * @throws FileConverterException 文件转换异常
	 */
	public FileInfo putFile(File file, String fileName) throws FileStoreException, FileConverterException{
		prepare();
		
		try {
			return putFileAsStream(new FileInputStream(file), fileName);
		} catch (FileNotFoundException e) {
			throw new FileStoreException(e);
		}
	}
	
	/**	
	 * 往文档中心保存一个文件流。如果初始化时设置为自动转换模式，则会自动启动转换线程进行异步转换。
	 * 如果不是自动转换模式，即仅当请求某格式文件时才进行同步转换。
	 * 自动转换策略基于初始化时的设置。
	 * @param stream 要保存的文件流
	 * @param fileName 真正的文件名。第一个参数中的文件可能是临时文件，其文件名将被忽略。
	 * 本参数将作为文件仓库中该文件真正的逻辑文件名，后续的文件转换过程将基于该文件名进行。
	 * @return 保存后的文件信息
	 * @throws FileStoreException 文件仓库异常
	 * @throws FileConverterException 文件转换异常
	 */
	public FileInfo putFileAsStream(InputStream stream, String fileName) throws FileStoreException, FileConverterException{
		prepare();
		
		FileInfo info = fileStore.putFileAsStream(stream, fileName);
		if (autoConvert) {
			String ext = FilenameUtils.getExtension(fileName);
			Collection<OutputType> strategy = convertStrategy.getConvertStrategy(ext);
			if (strategy != null) {
				for (OutputType type : strategy) {
					InputStream is = fileStore.getFileAsStream(info.getKey());
					if (FileConverterService.canCanvert(fileName, type)) {
						FileConverterService.convert(is, fileName, info.getKey(), type, true);
					}
				}
			}
		}
		
		return info;
	}

	/**
	 * 返回存储在文件仓库的源文件信息，即通过{@link #putFile(File)}保存的文件的信息。
	 * @param key 文件Key
	 * @return 文件信息对象。如果该Key对应的文件不存在，返回null。
	 * @throws FileStoreException 文件仓库异常
	 */
	public FileInfo getFileInfo(String key) throws FileStoreException {
		prepare();
		return fileStore.getFileInfo(key);
	}
	
	/**
	 * 返回存储在文档中心的源文件数据流，即通过{@link #putFile(File)}保存的文件。
	 * @param key 文件Key
	 * @return 文件字节流。如果文件不存在，返回null。
	 * @throws FileStoreException 文件仓库出现异常状况
	 */
	public InputStream getFileAsStream(String key) throws FileStoreException{
		prepare();
		
		return fileStore.getFileAsStream(key);
	}

	/**
	 * 从文档中心中彻底删除一个文件Key，包括源文件和转换后的各个文件。
	 * @param key 文件Key
	 * @throws FileStoreException 文件仓库异常
	 */
	public void deleteFile(String key) throws FileStoreException{
		prepare();
		
		fileStore.deleteFile(key);
		FileConverterService.deleteConvertedFileQuietly(key);		
	}
	
	/**
	 * 把源文件转换为目标格式
	 * @param key 文件Key
	 * @param outType 目标格式
	 * @param isAsync 是否异步转换
	 * @throws FileStoreException 文件仓库异常
	 * @throws FileConverterException 文件转换异常
	 */
	public void convert(String key, OutputType outType, boolean isAsync) throws FileConverterException, FileStoreException {
		prepare();
		
		FileInfo info = fileStore.getFileInfo(key);
		if (info != null) {
			FileConverterService.convert(fileStore.getFileAsStream(key), info.getFileName(), outType, isAsync);
		} else {
			logger.error("不存在的文件Key:{}", key);
		}
	}
	
	/**
	 * 关闭文档中心。程序退出时应调用本函数。
	 */
	public void close() {
		FileConverterService.finish();
	}
	
	/**
	 * 获取某文件的期望格式。如果当初保存到文档库中的文件本身就是该格式，则直接返回源文件
	 * 如果支持转换成该格式但尚未转换，且参数autoConvert为true，则同步开始进行转换，转换成功后返回转换后文件的相关信息。
	 * 
	 * @param key 文件Key
	 * @param wantType 想要的文件格式扩展名，不带"."。例如"pdf"、"html"、"zip"
	 * @param autoConvert 当需要的格式尚不存在时，是否同步开始自动进行转换，并等待转换完成后返回文件
	 * @return 文件信息对象。如果不存在该格式的文件且无法转换成该格式，则返回null。
	 * @throws FileStoreException 读取文件仓库时发生异常，抛出FileStoreException异常
	 * @throws FileConverterException 如果文档无法转换为期望预览的格式，抛出DocConvertException异常
	 */
	public RequestDocInfo requestTargetTypeFile(final String key, final String wantType, final boolean autoConvert) throws FileStoreException, FileConverterException {
		prepare();
		
		// 判断是否是可以预览或支持转换后预览的文件类型
		FileInfo originFileInfo = fileStore.getFileInfo(key);
		if (originFileInfo == null) {
			logger.error("在文件仓库中找不到文件。可能是文件仓库被人为删除了文件！Key={}", key);
			return null;
		}
		
		String fileName = originFileInfo.getFileName().toLowerCase();

		RequestDocInfo docInfo = new RequestDocInfo();
		docInfo.setKey(originFileInfo.getKey());
		docInfo.setOriginalFileName(originFileInfo.getFileName());
		docInfo.setWantType(wantType.toLowerCase());
		
		if (fileName.toLowerCase().endsWith("." + wantType.toLowerCase())) {
			// 原始文件就是所需要的格式，直接返回。
			docInfo.setSupportWantType(true);
			docInfo.setWantFilesCount(1);
			return docInfo;
		}
		
		// 查看是否是转换器当前能处理的目标类型
		OutputType outType = null;
		try {
			outType = OutputType.valueOf(wantType.toLowerCase());
		} catch (IllegalArgumentException e) {
			logger.error("当前系统不支持转换成目标类型。文件Key：{}, 文件名：{}, 目标类型：{} ", key, docInfo.getOriginalFileName(), wantType);
			docInfo.setSupportWantType(false);
			docInfo.setWantFilesCount(0);
			return docInfo;
		}
		
		// 查看是否是支持的转换类型
		if (!FileConverterService.canCanvert(fileName, outType)) {
			logger.error("当前系统不支持转换成目标类型。文件Key：{}, 文件名：{}, 目标类型：{} ", key, docInfo.getOriginalFileName(), outType.toString());
			docInfo.setSupportWantType(false);
			docInfo.setWantFilesCount(0);
			return docInfo;
		}
		
		docInfo.setSupportWantType(true);

		// 在转换后的文件中查找是否存在所需的格式
		int count = FileConverterService.getConvertedFilesCount(key, outType);
		if (count == 0) {
			if (autoConvert) {
				// 没有转换过，开始转换（同步过程）
				FileConverterService.convert(fileStore.getFileAsStream(key), fileName, key, outType);
				count = FileConverterService.getConvertedFilesCount(key, outType);
			} else {
				logger.info("{}文件尚未被转换成{}格式", key, outType.toString());
			}
		}
		
		docInfo.setWantFilesCount(count);
		
		return docInfo;
	}
	
	/**
	 * 获取某文件的某页。如果文件还没有被转换过，则同步进行转换，转换完成后返回某页文件。
	 * 目前实际上只有png格式才支持按页获取。
	 * @param key 文件Key
	 * @param wantType 想要以什么格式预览。目前必须是png，否则会出现异常或总是返回null。
	 * @param autoConvert 当需要的格式尚不存在时，是否同步开始自动进行转换，并等待转换完成后返回文件
	 * @return 该页的文件字节流。如果该格式的文件不存在、或转换失败，则返回null。
	 * @throws FileStoreException 文件仓库异常
	 * @throws FileConverterException 文档转换库异常
	 */
	public InputStream getFileAsStream(String key, String wantType, boolean autoConvert) throws FileStoreException, FileConverterException {
		prepare();
		
		RequestDocInfo info = requestTargetTypeFile(key, wantType, autoConvert);
		if (info == null || !info.isSupportWantType() || info.getWantFilesCount() == 0) {
			return null;
		}
		
		if (info.getOriginalFileName().toLowerCase().endsWith("." + wantType.toLowerCase())) {
			return getFileAsStream(key);
		} else {
			try {
				// 获取转换后的目标文件
				OutputType outType = OutputType.valueOf(wantType);
				return FileConverterService.getConvertedFileAsStream(key, outType);
			} catch (IllegalArgumentException e) {
				logger.error("不支持的目标转换格式：{}", wantType);
				return null;
			}
		}
	}
	
	/**
	 * 获取某格式文件的某页。如果文件还没有被转换过，则同步进行转换，转换完成后返回某页文件。
	 * 目前实际上只有png格式才支持按页获取。
	 * @param key 文件Key
	 * @param wantType 想要以什么格式预览。目前必须是png，否则会出现异常或总是返回null。
	 * @param page 需要的页码，基于0
	 * @param autoConvert 当需要的格式尚不存在时，是否同步开始自动进行转换，并等待转换完成后返回文件
	 * @return 该页的文件字节流。如果该格式的文件不存在、或转换失败，则返回null。
	 * @throws FileStoreException 读取文件仓库时发生异常，抛出FileStoreException异常
	 * @throws FileConverterException 如果文档无法转换为期望预览的格式，抛出DocConvertException异常
	 */
	public InputStream getFilePageAsStream(String key, String wantType, int page, boolean autoConvert) throws FileStoreException, FileConverterException {
		prepare();
		
		RequestDocInfo info = requestTargetTypeFile(key, wantType, autoConvert);
		if (info == null || !info.isSupportWantType() || info.getWantFilesCount() == 0) {
			return null;
		}
		
		// 分文件存储的文件，只可能存在于文档转换区，且只能是所支持的几种转换格式
		try {
			OutputType outType = OutputType.valueOf(wantType);
			return FileConverterService.getConvertedPageAsStream(key, outType, page);
		} catch (IllegalArgumentException e) {
			logger.error("不支持的目标转换格式：{}", wantType);
			return null;
		}
		
	}
}
