package com.kdgcsoft.power.filemanager;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.fileconverter.OutputType;

/**
 * 定义转换策略，即应该把某扩展名文件转换成哪些目标格式。
 * <br>未考虑线程安全。
 * @author hling
 *
 */
public class AutoConvertStrategy {
	
	private static final Logger logger = LoggerFactory.getLogger(AutoConvertStrategy.class); 
	
	private Map<String, SortedSet<OutputType>> convertDefMap = new HashMap<String, SortedSet<OutputType>>();
	
	/**
	 * 缺省转换策略
	 */
	private static final String DEFAULT_STRATEGY = "doc/docx/ppt/pptx->pdf, xls/xlsx->html";

	/**
	 * 构造一个空的转换策略，即不做任何自动转换
	 * @param strategyStr 
	 * @return
	 */
	private AutoConvertStrategy(){};
	
	/**
	 * 构造一个转换策略
	 * @param strategyStr 转换策略定义字符串，形如："doc/docx/ppt/pptx-&gt;pdf, xls/xlsx-&gt;html"
	 * @return 转换策略实例对象
	 */
	public static AutoConvertStrategy parse(String strategyStr) {
		AutoConvertStrategy strategy = new AutoConvertStrategy();
		String[] rules = strategyStr.split(",");
		logger.info("设置自动转换策略：");
		for (String rule : rules) {
			String[] convMap = rule.split("\\-\\>");
			if (convMap.length != 2) {
				logger.error("不合法的自动转换策略定义：{}", rule);
			} else {
				String srcExts = convMap[0].trim();
				String outExts = convMap[1].trim();
				for (String srcExt : srcExts.split("/")) {
					for (String outExt : outExts.split("/")) {
						logger.info("{} -> {}", srcExt, outExt);
						strategy.addStrategy(srcExt, outExt);
					}
				}
			}
		}
		return strategy;
	}
	
	/**
	 * 添加一条转换策略
	 * @param inputExt 被转换的源文件扩展名
	 * @param outExt 要转换成为的目标扩展名
	 */
	public void addStrategy(final String inputExt, final String outExt) {
		if (inputExt == null) {
			return;
		}
		
		SortedSet<OutputType> typeList = convertDefMap.get(inputExt);
		if (typeList == null) {
			typeList = new TreeSet<OutputType>();
			convertDefMap.put(inputExt, typeList);
		}
		
		try {
			OutputType outType = OutputType.valueOf(outExt.trim().toLowerCase());
			typeList.add(outType);
		} catch (IllegalArgumentException e) {
			logger.error("不支持的转换目标扩展名：{}", outExt);
		}
	}
	
	/**
	 * 添加一条转换策略
	 * @param srcExtName 被转换的文件扩展名，不包含"."，例如pdf、doc
	 * @param outType 要转换成的文档类型
	 */
	public void addStrategy(final String srcExtName, final OutputType outType) {
		SortedSet<OutputType> typeSet = convertDefMap.get(srcExtName);
		if (typeSet == null) {
			typeSet = new TreeSet<OutputType>();
			convertDefMap.put(srcExtName, typeSet);
		}
		typeSet.add(outType);		
	}
	
	/**
	 * 查询某扩展名当前的转换策略
	 * @param extName 被转换的源文件扩展名
	 * @return 该扩展名所需要转换成为的目标类型。如果没有定义过相应策略，返回null。
	 */
	public Collection<OutputType> getConvertStrategy(final String extName) {
		return convertDefMap.get(extName);
	}
	
	/**
	 * 创建缺省的转换策略。这种策略下，word和powerpoint文档将被转换为pdf，excel被转换为html，其他格式文件不做转换。
	 * @return 缺省策略对象
	 */
	public static AutoConvertStrategy createDefaultStrategy() {
		return parse(DEFAULT_STRATEGY);
	}

}
