package com.kdgcsoft.power.fileconverter.impl;

import java.io.File;

import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jacob.activeX.ActiveXComponent;
import com.jacob.com.ComThread;
import com.jacob.com.Dispatch;
import com.jacob.com.Variant;
import com.kdgcsoft.power.fileconverter.FileConverterException;
import com.kdgcsoft.power.fileconverter.FileConverterSettings;
import com.kdgcsoft.power.fileconverter.IFileConverter;

/**
 * 基于WPS进行文档转换。
 * 注：对ppt/pptx文件，转换是顺序进行的，任意时刻只有一个转换任务在运行，否则容易出现错误。
 * 在WPS 2016测试结果正常。WPS 2019版在转换后可能会有wps进程残留，尤其是大量并发转换的时候，因此不能使用。
 * 
 * @author hling
 *
 */
public class WpsOfficeConverter implements IFileConverter {
	private static final String MSG_CONVERT_FAILED = "文件转换失败!";

	private static final Logger logger = LoggerFactory.getLogger(WpsOfficeConverter.class);

	/**
	 * PPT转换线程锁。与Office一样，WPS的PPT程序是单进程的，多线程并发转换件时很容易出现异常，故改为单线程转换。
	 */
	private static final Object pptLock = new Object();

	public static final String WORDSERVER_STRING = "KWPS.Application";
	public static final String PPTSERVER_STRING = "KWPP.Application";
	public static final String EXECLSERVER_STRING = "KET.Application";
	
	/** wdFormatPDF */ 
	private static final int FORMAT_PDF = 17; // WPS word转PDF的宏值
	/** xlTypePDF */
	private static final int TYPE_PDF = 0;
	/** ppSaveAsPDF */
	private static final int SAVE_AS_PDF = 32;
	
	// 是否可见。可用于调试
	private static final boolean IS_VISIBLE = false;
	
	// private static final int wdFormatHTML = 9;
	/* 转HTML格式值 */
	// private static final int wdFormatHTML = 8;
	// private static final int ppFormatHTML = 12;
	// private static final int xlFormatHTML = 44;

	@Override
	public File convert(File srcFile, File destFile, FileConverterSettings settings) throws Exception {
        logger.info("开始使用wps进行转换...");

        if (!srcFile.exists()) {
			logger.info("{}该文件不存在!", srcFile.getName());
		} else {
			String suffix = FilenameUtils.getExtension(srcFile.getName());
			if (suffix.equals("pdf")) {
				logger.info("{} PDF文件不需要转换!", srcFile.getName());
			}
			if (suffix.equalsIgnoreCase("doc") || suffix.equalsIgnoreCase("docx")) {
				word2PDF(srcFile.getAbsolutePath(), destFile.getAbsolutePath());
			} else if (suffix.equalsIgnoreCase("ppt") || suffix.equalsIgnoreCase("pptx")) {
				synchronized (pptLock) {
					ppt2PDF(srcFile.getAbsolutePath(), destFile.getAbsolutePath());
				}
			} else if (suffix.equalsIgnoreCase("xls") || suffix.equalsIgnoreCase("xlsx")) {
				excel2PDF(srcFile.getAbsolutePath(), destFile.getAbsolutePath());
			} else {
				logger.error("{}文件格式不支持转换!", srcFile.getName());
				throw new FileConverterException(srcFile.getName() + "文件格式不支持转换!");
			}
		}

		return destFile;
	}

	/**
	 * Word文档转为PDF
	 * 
	 * @param inputFile
	 *            源Word文件
	 * @param pdfFile
	 *            目标PDF文件
	 * @throws FileConverterException
	 *             转换异常
	 */
	public void word2PDF(String inputFile, String pdfFile) throws FileConverterException {
		ActiveXComponent app = null;
		Dispatch doc = null;
		try {
			initActiveX();
			// 打开word应用程序
			app = new ActiveXComponent(WORDSERVER_STRING);
			// 设置Word不可见
			app.setProperty("Visible", new Variant(IS_VISIBLE));
			// 禁用宏
			app.setProperty("AutomationSecurity", new Variant(3));
			
			// 获得word中所有打开的文档,返回Documents对象  
	        Dispatch docs = app.getProperty("Documents").toDispatch();  
	        // 调用Documents对象中Open方法打开文档，并返回打开的文档对象Document  
			doc = Dispatch.call(
									docs,
									"Open",
									inputFile,
									false,
									true  // readOnly
									).toDispatch();
	        // 将文档保存为pdf格式  
			 Dispatch.call(doc, "ExportAsFixedFormat", pdfFile, FORMAT_PDF );
			return;
		} catch (Exception e) {
			throw new FileConverterException(inputFile + MSG_CONVERT_FAILED, e);
		} finally {
			safeClose(doc, new Variant(false));
			quitApp(app);
		}
	}

	/**
	 * PowerPoint文档转换为PDF
	 * 
	 * @param inputFile
	 *            源PPT文件
	 * @param pdfFile
	 *            目标PDF文件
	 * @throws FileConverterException
	 *             转换异常
	 */
	public void ppt2PDF(String inputFile, String pdfFile) throws FileConverterException {
		ActiveXComponent app = null;
		Dispatch ppt = null;
		try {
			initActiveX();
			app = new ActiveXComponent(PPTSERVER_STRING);
			Dispatch ppts = app.getProperty("Presentations").toDispatch();  
			ppt = Dispatch.call(ppts,
                    "Open",
                    inputFile,  
                    true, //ReadOnly  
                    //false, //Untitled指定文件是否有标题
                    false //WithWindow指定文件是否可见  
                    ).toDispatch();  
			Dispatch.call(ppt, "SaveAs", pdfFile, SAVE_AS_PDF);
			return;
		} catch (Exception e) {
			throw new FileConverterException(inputFile + MSG_CONVERT_FAILED, e);
		} finally {
			safeClose(ppt, null);
			quitApp(app);
		}
	}

	/**
	 * Excel文档转换为PDF
	 * 
	 * @param inputFile
	 *            Excel文件
	 * @param pdfFile
	 *            目标PDF文件
	 * @throws FileConverterException
	 *             转换异常
	 */
	public void excel2PDF(String inputFile, String pdfFile) throws FileConverterException {
		ActiveXComponent app = null;
		Dispatch workbook = null;
		try {
			initActiveX();
			app = new ActiveXComponent(EXECLSERVER_STRING);
			app.setProperty("Visible", IS_VISIBLE);
			app.setProperty("AutomationSecurity", new Variant(3)); // 禁用宏
			Dispatch workbooks = app.getProperty("Workbooks").toDispatch();
			workbook = Dispatch.call(workbooks,  
                    "Open",
                    inputFile,  
                    false,
                    true
                    ).toDispatch();  
			// 转换格式
			Dispatch.invoke(workbook, "ExportAsFixedFormat", Dispatch.Method, 
					               new Object[] {
					            		new Variant(TYPE_PDF), // PDF格式=0
										pdfFile,
										new Variant(0) // 0=标准 (生成的PDF图片不会变模糊)  1=最小文件(生成的PDF图片糊的一塌糊涂)
									},
					               new int[1]);
		} catch (Exception e) {
			throw new FileConverterException(inputFile + MSG_CONVERT_FAILED, e);
		} finally {
			safeClose(workbook, new Variant(false));
			quitApp(app);
		}
	}
	
	/**
	 * 初始化COM
	 */
	private void initActiveX() {
		ComThread.InitMTA(true);
	}

	/**
	 * 关闭Dispatch对象
	 * @param dispatch Dispatch对象
	 * @param param 参数
	 */
	private void safeClose(Dispatch dispatch, Variant param) {
		if (dispatch != null) {
			try {
				if (param == null) {
					Dispatch.call(dispatch, "Close");
				} else {
					Dispatch.call(dispatch, "Close", param);
				}
			} catch (Exception e) {
				logger.warn("Close wps document失败。", e);
			}
			dispatch.safeRelease();
		}
	}

	/**
	 * 退出COM进程
	 * @param app
	 */
	private void quitApp(ActiveXComponent app) {
		if (app != null) {
			try {
				app.invoke("Quit");
			} catch (Exception e) {
				logger.warn("Quit wps失败。", e);
			}
			app.safeRelease();
		}
		ComThread.Release();
		ComThread.quitMainSTA();
	}

//	public static void main(String[] args) throws FileConverterException {
//		WpsOfficeConverter convert = new WpsOfficeConverter();
//		convert.word2PDF("D://aaaaa.docx", "D://aaaaa.pdf");
//	}
}
