package com.kdgcsoft.power.fileconverter.impl;

import java.io.File;

import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jacob.activeX.ActiveXComponent;
import com.jacob.com.ComThread;
import com.jacob.com.Dispatch;
import com.jacob.com.Variant;
import com.kdgcsoft.power.fileconverter.FileConverterException;
import com.kdgcsoft.power.fileconverter.FileConverterSettings;
import com.kdgcsoft.power.fileconverter.IFileConverter;

/**
 * 基于Jacob的转换器，将office文档转为pdf文件
 *
 */
public class MsOfficeConverter implements IFileConverter {

	private static final String MSG_CONVERT_FAILED = "文件转换失败!";

	private static final Logger logger = LoggerFactory.getLogger(MsOfficeConverter.class);
	
	/** wdFormatPDF */
	private static final int FORMAT_PDF = 17; //PDF 格式
	/** xlTypePDF */
	private static final int TYPE_PDF = 0;
	/** ppSaveAsPDF */
    private static final int SAVE_AS_PDF = 32;
    
	// 用Office2007实测，PowerPoint多线程调用很容易出错，故需要控制单线程使用
	// Jacob官方测试代码有注释说明：https://github.com/joval/jacob/blob/master/unittest/com/jacob/test/powerpoint/PowerpointTest.java
    private static Object pptLock = new Object();
    
	@Override
	public File convert(final File srcFile, final File destFile, final FileConverterSettings settings) throws FileConverterException{
        logger.info("开始使用微软Office进行转换...");
		if(!srcFile.exists()){  
        	logger.info("{}该文件不存在!", srcFile.getName());            
        	
        }else{
        	String suffix =  FilenameUtils.getExtension(srcFile.getName());
	        if(suffix.equalsIgnoreCase("pdf")){
	            logger.info("{}PDF not need to convert!", srcFile.getName());
	        }  
	        if(suffix.equalsIgnoreCase("doc")||suffix.equalsIgnoreCase("docx")||suffix.equalsIgnoreCase("txt")){  
	            word2PDF(srcFile.getAbsolutePath(),destFile.getAbsolutePath());  
	        }else if(suffix.equalsIgnoreCase("ppt")||suffix.equalsIgnoreCase("pptx")){
	        	// 用Office2007测试发现对多线程支持不好，各种报错，故改为单线程调用
        		synchronized (pptLock) {
        			ppt2PDF(srcFile.getAbsolutePath(),destFile.getAbsolutePath());
				}
	        }else if(suffix.equalsIgnoreCase("xls")||suffix.equalsIgnoreCase("xlsx")){  
	            excel2PDF(srcFile.getAbsolutePath(),destFile.getAbsolutePath());  
	        }else{
	            logger.error("{}文件格式不支持转换!", srcFile.getName());
	            throw new FileConverterException(srcFile.getName()+"文件格式不支持转换!");
	        }  
        }
        
        return destFile;
	}

	/**
	 * Word文档转为PDF
	 * @param inputFile 待转换文档的路径
	 * @param pdfFile 要转换成的目标PDF文件全路径
	 * @throws FileConverterException 转换异常
	 */
	public void word2PDF(String inputFile, String pdfFile) throws FileConverterException {
		ActiveXComponent app=null;
		Dispatch doc = null;
		try {
			ComThread.InitMTA(true);
			// 打开word应用程序
			app = new ActiveXComponent("Word.Application");
			// 设置word不可见
			app.setProperty("Visible", false);
			// 获得word中所有打开的文档,返回Documents对象
			Dispatch docs = app.getProperty("Documents").toDispatch();
			// 调用Documents对象中Open方法打开文档，并返回打开的文档对象Document
			doc = Dispatch.call(docs, "Open", inputFile, false, false).toDispatch();
			// 调用Document对象的SaveAs方法，将文档保存为pdf格式
			deletePdfFile(pdfFile);
			// Dispatch.call(doc, "SaveAs", pdfFile, wdFormatPDF);
			//word保存为pdf格式宏，值为17 );
			 
			Dispatch.call(doc, "ExportAsFixedFormat", pdfFile, FORMAT_PDF );// word保存为pdf格式宏，值为17
			return;
		} catch (Exception e) {
	         throw new FileConverterException(inputFile+MSG_CONVERT_FAILED, e);
		}finally {
			safeClose(doc, "Word", true);
            quitApp(app, new Variant(0));
        }  
	}

	/**
	 * Excel文档转换为PDF
	 * @param inputFile Excel文件
	 * @param pdfFile 目标PDF文件
	 * @throws FileConverterException 转换异常
	 */
	public void excel2PDF(String inputFile, String pdfFile) throws FileConverterException {
		ActiveXComponent app=null;
		Dispatch excel=null;
		try {
			ComThread.InitMTA(true);
			app = new ActiveXComponent("Excel.Application");
			app.setProperty("Visible", false);
			app.setProperty("AutomationSecurity", new Variant(3)); //禁用宏
			Dispatch excels = app.getProperty("Workbooks").toDispatch();
			excel = Dispatch.call(excels, "Open", inputFile, false, true).toDispatch();
			deletePdfFile(pdfFile);
			Dispatch.call(excel, "ExportAsFixedFormat", TYPE_PDF, pdfFile);
			return;
		} catch (Exception e) {
	         throw new FileConverterException(inputFile+MSG_CONVERT_FAILED, e);
		}finally {
			safeClose(excel, "Excel", false);
            quitApp(app, null);
        } 
	}

	/**
	 * PowerPoint文档转换为PDF
	 * @param inputFile 源PPT文件
	 * @param pdfFile 目标PDF文件
	 * @throws FileConverterException 转换异常
	 */
	public void ppt2PDF(String inputFile, String pdfFile) throws FileConverterException {
		ActiveXComponent app=null;
		Dispatch ppt=null;
		try {
			ComThread.InitMTA(true);
			app = new ActiveXComponent("PowerPoint.Application");
			Dispatch ppts = app.getProperty("Presentations").toDispatch();
			ppt = Dispatch.call(ppts, "Open", inputFile, true,// ReadOnly
					true,// Untitled指定文件是否有标题
					false// WithWindow指定文件是否可见
					).toDispatch();
			deletePdfFile(pdfFile);
			Dispatch.call(ppt, "SaveAs", pdfFile, SAVE_AS_PDF);
			return;
		} catch (Exception e) {
	         throw new FileConverterException(inputFile+MSG_CONVERT_FAILED, e);
		}finally {
			safeClose(ppt, "PowerPoint", null);
            quitApp(app, null);
        } 
	}
	
	private void safeClose(Dispatch dispatch, String name, Object param) {
		if (dispatch == null) {
			return;
		}
		
		try {
			if (param == null) {
				Dispatch.call(dispatch, "Close");
			} else {
				Dispatch.call(dispatch, "Close", param);
			}
		} catch (Exception e) {
			logger.warn("关闭 {} ActiveX 失败", name, e);
		}
		dispatch.safeRelease();
	}
	
	private void quitApp(ActiveXComponent app, Variant param) {
		if (app != null) {
			try {
				if (param == null) {
					app.invoke("Quit");
				} else {
					app.invoke("Quit", param);
				}
			} catch (Exception e) {
				logger.warn("Quit Office失败", e);
			}
			app.safeRelease();
		}
		// 释放进程
		ComThread.Release();
		ComThread.quitMainSTA();
	}
	
	/**
	 * MSoffice转PDF时如果目标文件已经存在会报错，所以需要在转换前检查，如存在则删除。
	 * @param pdfFile 要删除的PDF文件
	 */
	public void deletePdfFile(String pdfFile) {
		File tofile = new File(pdfFile);
		if (tofile.exists()) {
		   logger.info("{}文件已存在,删除后重新生成!", pdfFile);   
           tofile.delete();
		}
	}
	
//	public static void main(String[] args) throws FileConverterException {
//		MsOfficeConverter conver =new MsOfficeConverter();
//		conver.excel2PDF("D:/666.xls", "D:/666.pdf");
//	}
}
