package com.kdgcsoft.power.fileconverter.impl;

import java.io.File;
import java.util.Set;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.jodconverter.OfficeDocumentConverter;
import org.jodconverter.document.DocumentFormat;
import org.jodconverter.document.DocumentFormatRegistry;
import org.jodconverter.office.DefaultOfficeManagerBuilder;
import org.jodconverter.office.OfficeException;
import org.jodconverter.office.OfficeManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.fileconverter.util.SocketUtils;

/**
 * JodConverter(https://github.com/sbraconnier/jodconverter)的调用类。
 * @author hling
 *
 */
public class JodHelper {

	private static Logger logger = LoggerFactory.getLogger(JodHelper.class);

	protected static volatile OfficeManager officeManager = null;
	protected static volatile OfficeDocumentConverter converter;
	protected static volatile DocumentFormatRegistry formatRegistry;
	
	/** 启动锁 */
	private static final Object START_LOCK = new Object();

	/**
	 * 在缺省位置启动OpenOffice/LibreOffice。缺省位置与操作系统有关。
	 * 如果是Windows系统，则位于系统盘的program files目录下。
	 * @param openOfficeDir OpenOffice的安装路径
	 * @param maxConvertThread 指定最大并发转换线程，并基于该最大线程数启动监听相同数量的转换端口，最大不能超过50。仅在ports为null或空时起作用。
	 * @param ports <b>[不建议使用]</b> 设定转换程序监听端口，可以有多个。
	 * 每个端口同时只能处理一个转换任务，即端口数等同于并发线程数。
	 * 除非特殊需要，建议此参数传null或空，将自动选择空闲端口
	 * @throws OfficeException 启动Office时发生的异常
	 */
	public static synchronized void startOfficeManager(final String openOfficeDir, final int maxConvertThread, final int[] ports) throws OfficeException {
		// 不能两次启动
		if (officeManager != null) {
			return;
		}
		
		// Start an office manager
		DefaultOfficeManagerBuilder builder = new DefaultOfficeManagerBuilder();

		if (ports == null || ports.length == 0 ) {
			// 如果未指定端口号，则自动探测空闲端口，数量为最大线程数。起始端口号为50000
			Set<Integer> portSet = SocketUtils.findAvailableTcpPorts(maxConvertThread<50?maxConvertThread:50, 50000, 60000);
			int[] portArray = new int[portSet.size()];
			int i = 0;
			for(Integer port : portSet){
				portArray[i++] = port;
			}
			builder = builder.setPortNumbers(portArray);
		} else {
			builder = builder.setPortNumbers(ports);
		}

		if (openOfficeDir != null) {
			builder = builder.setOfficeHome(openOfficeDir);
		}
		builder.setTaskQueueTimeout(60000);
		
		officeManager = builder.build();
		
		converter = new OfficeDocumentConverter(officeManager);
		formatRegistry = converter.getFormatRegistry();
		
		officeManager.start();
	}

	public static synchronized void stopOfficeManager() {
		if (officeManager != null) {
			try {
				officeManager.stop();
			} catch (OfficeException e) {
				logger.error("关闭LibreOffice/OpenOffice异常。您可能需要手动Kill进程！", e);
			}
		} else {
			logger.error("OpenOffice尚未启动，无需关闭");
		}
	}
	
	public static void main(String[] args) {
		try {
			startOfficeManager(null, 5, new int[]{5678, 5679});			
//			converter.convert(new File("D:/111.docx"), new File("D:/111.png"));
			stopOfficeManager();
		} catch (OfficeException e) {
			logger.error("启动/关闭OpenOffice/LibreOffice失败", e);
		}
	}
	
	public static boolean canConvert(final File inputFile, final String outputExtension) {
		final String inputExtension = FilenameUtils.getExtension(inputFile
				.getName());
		final DocumentFormat inputFormat = formatRegistry
				.getFormatByExtension(inputExtension);
		
		if (inputFormat == null) {
			logger.error("不支持的输入文件格式:", inputFile.getName());
			return false;
		}
		
		// Get all supported output formats
		final Set<DocumentFormat> outputFormats = formatRegistry
				.getOutputFormats(inputFormat.getInputFamily());
		for (final DocumentFormat outputFormat : outputFormats) {
			// LibreOffice 4 fails natively on those one
			if (inputFormat.getExtension().equals("odg")
					&& outputFormat.getExtension().equals("svg")) {
				logger.info("-- skipping odg to svg test... ");
				continue;
			}
			if (StringUtils.equalsAny(outputFormat.getExtension(), "sxc",
					"sxw", "sxi")) {
				logger.info("-- skipping {} to {} test... ", inputExtension,
						outputFormat.getExtension());
				continue;
			}
			
			if (StringUtils.equals(outputFormat.getExtension(), outputExtension)){
				return true;
			}
		}
		
		logger.error("不支持的输出文件格式:", outputExtension);
		return false;
	}
	
	public static void convert(final File inputFile, final File outputFile, final String openOfficeDir, final int maxConvertThread, final int[] ports) throws OfficeException {
		if (officeManager == null) {
			logger.info("开始启动OpenOffice...");
			synchronized (START_LOCK) {
				startOfficeManager(openOfficeDir, maxConvertThread, ports);
			}
			logger.info("启动OpenOffice结束");
		}
		
		if (!officeManager.isRunning()) {
			synchronized (START_LOCK) {
				if (!officeManager.isRunning()) {
					logger.error("OpenOffice未处于运行状态，尝试启动...");
					officeManager.start();
				}
			}
		}
		
		converter.convert(inputFile, outputFile);		
	}
}
