package com.kdgcsoft.power.fileconverter.impl;

import java.io.File;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.fileconverter.ConverterFactory;
import com.kdgcsoft.power.fileconverter.FileConverterService;
import com.kdgcsoft.power.fileconverter.FileConverterSettings;
import com.kdgcsoft.power.fileconverter.IFileConverter;
import com.kdgcsoft.power.fileconverter.OutputType;
import com.kdgcsoft.power.fileconverter.util.FileExtensionUtil;

/**
 * 链式转换器。可连续调用多个Converter，分步完成转换过程。<br>
 * 需注意在使用时，中间步骤的转换器输出的必须是单个完整文档，只有最后一步的转换器可以输出分页文件。
 * TODO 是否需要提供连续转换分页文件的能力？
 * <p>
 * 使用方法：按照需要的顺序调用 {@link #addConverter(IFileConverter, OutputType) }添加转换器实例和转换对象，
 * 然后调用{@link #convert(File, File, FileConverterSettings)}执行。 
 * @author hling
 *
 */
public class ChainConverter implements IFileConverter {
	
	private static Logger logger = LoggerFactory.getLogger(ChainConverter.class);
	
	private Map<IFileConverter, OutputType> converters = new LinkedHashMap<IFileConverter, OutputType>();
	
	/**
	 * 添加一个转换器
	 * @param converter 转换器实例
	 * @param outType 打算要用该转换器转换成的目标类型
	 */
	public void addConverter(IFileConverter converter, OutputType outType) {
		converters.put(converter, outType);
	}
	
	/**
	 * 直接根据要连续转换的类型，创建一个链式转换器
	 * @param srcFileName 源文件名，用于决定第一个转换器。只要带有".扩展名"即可。
	 * @param settings 转换设定参数
	 * @param outTypeList 连续转换的类型数组。转换器将按顺序逐个进行转换。前一个转换类型是下一个转换目标的输入类型。
	 * 例如通常应该是 pdf -&gt; html
	 * @return 链式转换器实例
	 */
	public static ChainConverter createConverter(final String srcFileName, final FileConverterSettings settings, final OutputType... outTypeList) {
		ChainConverter chain = new ChainConverter();
		String filename = FilenameUtils.getName(srcFileName);
		for (OutputType type : outTypeList) {
			IFileConverter converter = ConverterFactory.createConverter(filename, type, settings);
			chain.addConverter(converter, type);
			filename = "." +type.toString();
		}
		
		return chain;
	}

	@Override
	public File convert(final File srcFile, final File destFile, final FileConverterSettings settings) throws Exception {
		String lastFileName = srcFile.getAbsolutePath();
		String nextFileName = destFile.getAbsolutePath();
		// 顺序执行每个Converter
		logger.info("开始分步转换");
		
		File result = null;
		int n = 1;
		for (Entry<IFileConverter, OutputType> item : converters.entrySet()) {
			try {
				OutputType outType = item.getValue();
				nextFileName = FileExtensionUtil.replaceExtension(nextFileName, outType.toString());
				logger.info("分步转换-第{}步：把{}转换到{}", n, lastFileName, nextFileName);
				result = FileConverterService.addConvertTask(new File(lastFileName), (new File(nextFileName)).getParentFile(), outType, false);
				lastFileName = result.getAbsolutePath();
				n++;
			} catch (Exception e){
				// 删除未转换成功的文件，防止留下垃圾文件
				FileUtils.deleteQuietly(new File(nextFileName));
				logger.error("转换失败。源文件：{}，目标文件：{}", lastFileName, nextFileName);
				throw e;
			}
		}
		logger.info("分步转换结束");
		
		return result;
	}

}
