package com.kdgcsoft.power.fileconverter;

import java.io.File;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.fileconverter.impl.ChainConverter;
import com.kdgcsoft.power.fileconverter.impl.CopyOnlyConverter;
import com.kdgcsoft.power.fileconverter.impl.Doc2MultiplePicHtmlConverter;
import com.kdgcsoft.power.fileconverter.impl.DummyConverter;
import com.kdgcsoft.power.fileconverter.impl.Excel2HtmlConverter;
import com.kdgcsoft.power.fileconverter.impl.JodOfficeConverter;
import com.kdgcsoft.power.fileconverter.impl.MsOfficeConverter;
import com.kdgcsoft.power.fileconverter.impl.Pdf2HtmlConverter;
import com.kdgcsoft.power.fileconverter.impl.WpsOfficeConverter;

/**
 * 转换器工厂类，用于创建特定文件类型之间转换的转换器。
 * @author hling
 *
 */
public class ConverterFactory {
	
	private static final Logger logger = LoggerFactory.getLogger(ConverterFactory.class);
	
	private ConverterFactory() {
		// 工具类不需要new
	}
	
	public static IFileConverter createConverter(final String srcFileName, final OutputType outType, final FileConverterSettings settings) {
		
		// 测试模式下始终使用虚拟转换器
		if (settings.isTestMode()) {
			return new DummyConverter();
		}
		
		String srcExtName = FilenameUtils.getExtension(srcFileName).toLowerCase();
		
		// 扩展名相同，不需要转换，只要拷贝
		if (srcExtName.equals(FileConverterSettings.getTypeExtension(outType))) {
			return new CopyOnlyConverter();
		}
		
		if (OutputType.PDF.equals(outType)){
			return createConverterForPDF(settings);
		} else if (OutputType.HTML.equals(outType)) {
			return createConverterForSingleHtml(srcFileName, settings);
		} else if (OutputType.LINK_HTML.equals(outType)) {
			return createConverterForLinkHtml(srcFileName, settings);
		} else if (OutputType.PNG.equals(outType) || OutputType.JPG.equals(outType) ) {
			return createConverterForImage(srcFileName, outType, settings);
		} else {
			logger.error("不可能出现的错误：代码中增加了转换目标类型定义枚举，但尚未实现？");
			return null;
		}
	}

	/**
	 * 创建用于生成PDF的转换器
	 * @param settings 转换参数
	 * @return 转换器实例
	 */
	private static IFileConverter createConverterForPDF(final FileConverterSettings settings) {
		// Office文档转换为PDF，必须使用外部转换引擎
		Office2PdfEngineType engine = settings.getOffice2PdfEngine();
		if (engine == null) {
			logger.error("要转换office文档到pdf格式时，必须配置openoffice、msoffice、wps的其中之一！");
			return null;
		}
		if (Office2PdfEngineType.OPENOFFICE.equals(engine)) {
			return new JodOfficeConverter();
		} else if (Office2PdfEngineType.MSOFFICE.equals(engine)) {
			return new MsOfficeConverter();
		} else if (Office2PdfEngineType.WPS.equals(engine)) {
			return new WpsOfficeConverter();
		} else {
			// 不可能走到这里
			logger.error("不支持的转换引擎：{}", engine);
			return null;
		}
	}
		
	/**
	 * 创建用于生成单文件HTML（内嵌图片）的转换器
	 * @param srcFileName 源文件名
	 * @param settings 转换参数
	 * @return 转换器实例
	 */
	private static IFileConverter createConverterForSingleHtml(final String srcFileName, final FileConverterSettings settings) {
		if (isExcel(srcFileName)) {
			return new Excel2HtmlConverter();
		} else if (isPdf(srcFileName)) {
			// 检查pdf2htmlEx引擎是否存在
			boolean hasPdf2HtmlExEngine = false;
			if (!StringUtils.isEmpty(settings.getPdf2HtmlExePath())) {
				File exeFile = new File(settings.getPdf2HtmlExePath());
				if (exeFile.exists() && exeFile.isFile()) {
					hasPdf2HtmlExEngine = true;
				}
			}
			// 如果有Pdf2HtmlEx引擎，则总是使用它
			if (hasPdf2HtmlExEngine) {
				return new Pdf2HtmlConverter();
			} else {
				return new Doc2MultiplePicHtmlConverter(true);
			}
		} else {
			// 其他Office文档，先判断是否需要先转pdf以保证最佳的输出质量
			if (settings.isOffice2PdfFirst()) {
				// 先转成pdf，再转成html
				return ChainConverter.createConverter(srcFileName, settings, OutputType.PDF, OutputType.HTML);
			} else {
				// 直接转HTML，效果较差
				return new Doc2MultiplePicHtmlConverter();
			}
		}
	}

	/**
	 * 创建用于生成带外部图片链接的HTML的转换器
	 * @param srcFileName 源文件名
	 * @param settings 转换参数
	 * @return 转换器实例
	 */
	private static IFileConverter createConverterForLinkHtml(final String srcFileName, final FileConverterSettings settings) {
		// Excel文件只能转为单HTML
		if (isExcel(srcFileName)) {
			return new Excel2HtmlConverter();
		} else if (isPdf(srcFileName)) {
			return new Doc2MultiplePicHtmlConverter(false);
		} else {
			// 判断是否经过PDF引擎
			if (settings.isOffice2PdfFirst()) {
				// 先转成pdf，再转成html
				return ChainConverter.createConverter(srcFileName, settings, OutputType.PDF, OutputType.LINK_HTML);
			} else {
				return new Doc2MultiplePicHtmlConverter(false);
			}
		}
	}

	/**
	 * 创建用于生成图片的转换器
	 * @param srcFileName 源文件名
	 * @param outType 目标图片类型
	 * @param settings 转换参数
	 * @return 转换器实例
	 */
	private static IFileConverter createConverterForImage(final String srcFileName,
			final OutputType outType, final FileConverterSettings settings) {
		if (isPdf(srcFileName)) {
			return new Doc2MultiplePicHtmlConverter();
		} else {
			// 判断是否先进行pdf转换。对Word文档必须先转换成PDF才能转成图片
			if (settings.isOffice2PdfFirst() || isWord(srcFileName)) {
				// 先转成pdf，再转成html
				return ChainConverter.createConverter(srcFileName, settings, OutputType.PDF, outType);
			} else {
				if (isExcel(srcFileName)) {
					logger.error("不支持把Excel文件转换为图片！");
					return null;
				}
				// 对PowerPoint文档可以直接转换为多页图片格式
				return new Doc2MultiplePicHtmlConverter(false);
			}
		}
	}

	private static boolean isWord(String fileName) {
		return StringUtils.endsWithIgnoreCase(fileName, ".doc") || StringUtils.endsWithIgnoreCase(fileName, ".docx");
	}

	private static boolean isExcel(String fileName) {
		return StringUtils.endsWithIgnoreCase(fileName, ".xls") || StringUtils.endsWithIgnoreCase(fileName, ".xlsx");
	}

	private static boolean isPdf(String fileName) {
		return StringUtils.endsWithIgnoreCase(fileName, ".pdf");
	}

}
